//
//  HomeListDatasource.swift
//  CTFit
//
//  Created by Mac on 2020/11/23.
//  Copyright © 2020 jpaxh. All rights reserved.
//

import Foundation
import RxSwift

final class HomeListDatasource {
    
    lazy var dateBarItem: DateBarItem = DateBarItem(date: Date())

    lazy var listItems = [stepItem, heartItem, exerciseItem, sleepItem]

    init() {
        onHandler()
        queary()
        observers()
        dateBarItem.onValueChanged { [weak self] (dateBar) in
            guard let strong = self else { return }
            strong.queary()
        }
    }
    
    func showToady() {
        dateBarItem.date = Date()
        dateBarItem.reload()
        queary()
    }
    
    private func queary() {
        BandData.queryHome(with: dateBarItem.date, series_s: true) { [weak self] total in
            guard let strong = self else { return }
            strong.update(total ?? BDTotal())
        }
    }
    
    private func observers() {
        BandData.updatedOutput.observeOn(MainScheduler.instance).subscribe(onNext: { [weak self] (type) in
            guard let strong = self else { return }
            switch type {
            case .record(let type, let dateStr):
                guard dateStr == strong.dateStr else { return }
                switch type {
                case .total, .heart, .exercise: strong.queary()
                default: break
                }
            default : break
            }
        }).disposed(by: disposeBag)
        
        BandSettings.observable.subscribe(onNext: { [weak self] (options) in
            guard let strong = self else { return }
            guard options.contains(.system) || options.contains(.goal) else { return }
            strong.stepItem.reload()
        }).disposed(by: disposeBag)
    }
    
    private func update(_ total: BDTotal) {
        stepItem.update(total)
        heartItem.update(total)
        exerciseItem.update(total)
        sleepItem.update(total)
    }
    
    private var dateStr: String { dateFormatter.string(from: dateBarItem.date) }
    
    private lazy var dateFormatter :DateFormatter = {
        let formatter = DateFormatter();
        formatter.dateFormat = "yyyy-MM-dd";
        return formatter;
    }()

    private lazy var stepItem: HomeCircleItem = HomeCircleItem()
    private lazy var heartItem: HomeListItem = HomeListItem(.heart)
    private lazy var exerciseItem: HomeListItem = HomeListItem(.exercise)
    private lazy var sleepItem: HomeListItem = HomeListItem(.sleep)
    private let disposeBag = DisposeBag()
}

extension HomeListDatasource {
    private enum ListType {
        case step, heart, sleep, exercise
    }
    
    private func onHandler() {
        stepItem.setSelectionHandler { [weak self] (item) in
            guard let strong = self else { return }
            strong.didSelectedHandler(.step)
        }
        heartItem.setSelectionHandler { [weak self] (item) in
            guard let strong = self else { return }
            strong.didSelectedHandler(.heart)
        }
        exerciseItem.setSelectionHandler { [weak self] (item) in
            guard let strong = self else { return }
            strong.didSelectedHandler(.exercise)
        }
        sleepItem.setSelectionHandler { [weak self] (item) in
            guard let strong = self else { return }
            strong.didSelectedHandler(.sleep)
        }
    }
    
    private func didSelectedHandler(_ type: ListType) {
        switch type {
        case .step, .exercise: push(RecordStepViewController(date: dateBarItem.date))
        case .heart: push(HeartRateDetailViewController(date: dateBarItem.date))
        case .sleep: push(RecordSleepViewController(date: dateBarItem.date))
        }
    }
    
    private func push(_ vc: BaseViewController) {
        guard let currentVc = Helper.currentVc as? HomeListController else { return }
        currentVc.navigationController?.pushViewController(vc, animated: true)
    }
    
}
