//
//  HomeCircleCell.swift
//  CTFit
//
//  Created by Mac on 2020/11/24.
//  Copyright © 2020 jpaxh. All rights reserved.
//

import UIKit
import SnapKit
import TableViewManager;

final class HomeCircleCell: BaseTableViewCell, ZJCellProtocol {

    // MARK: - ZJCellProtocol
    var item: HomeCircleItem!
    typealias ZJCelltemClass = HomeCircleItem
    override func cellWillAppear() {
        circleBar.step = Int(item.total.steps)
        calorieAttr.total = item.total
        distanceAttr.total = item.total
    }
    
    // MARK: - Initialize
    override func prepareForReuse() {
        super.prepareForReuse()
    }
    
    override init(style: UITableViewCell.CellStyle, reuseIdentifier: String?) {
        super.init(style: .default, reuseIdentifier: reuseIdentifier)
        setConstraints()
    }
   
    // MARK: - View customer
    private func setConstraints() {
        [circleBar, calorieAttr, distanceAttr].forEach { view in
            view.translatesAutoresizingMaskIntoConstraints = false
            addSubview(view)
        }
        circleBar.snp.makeConstraints { (make) in
            make.center.equalToSuperview()
            make.top.equalToSuperview().offset(12)
            make.width.equalTo(circleBar.snp.height)
        }
        
        calorieAttr.snp.makeConstraints { (make) in
            make.centerX.equalTo(self.snp.leadingMargin).offset(Constraints.large+18)
            make.bottom.equalToSuperview().offset(-Constraints.small)
        }
        
        distanceAttr.snp.makeConstraints { (make) in
            make.centerX.equalTo(self.snp.trailing).offset(-Constraints.large-18)
            make.bottom.equalToSuperview().offset(-Constraints.small)
        }
    }
        
    private lazy var circleBar: HomeStepCircleProgressBar = HomeStepCircleProgressBar()
    
    private lazy var calorieAttr: HomeCircleCellAttr = HomeCircleCellAttr(.calorie)
    
    private lazy var distanceAttr: HomeCircleCellAttr = HomeCircleCellAttr(.distance)


}

class HomeStepCircleProgressBar: HomeCircleProgressBar {
    var step: Int = 0 {
        didSet{
            valueLabel.text = "\(step)"
            let goalStep = BandSettings.goal.step
            let progress: CGFloat = CGFloat(step)/CGFloat(goalStep)
            setProgress(progress, animated: true)
        }
    }
    
    override func initialize() {
        super.initialize()
        setupSubviews()
    }
        
    private lazy var imageView: UIImageView = {
        UIImageView(image: ImageRepo.HomeList.step)
    }()
    
    private lazy var valueLabel :UILabel = {
        let label = UILabel(style: Stylesheet.Label.text)
        label.font = UIFont.lightCustomFont(ofSize: 40)
        label.text  = "0"
        return label
    } ()
    
    private lazy var unitLabel: UILabel  = {
        let label = UILabel(style: Stylesheet.Label.text)
        label.font = UIFont.lightCustomFont(ofSize: 20)
        label.text  = SRString.Step.unit_step.locastr
        return label
    } ()
    
    
    func setupSubviews() {
        let items = [imageView, valueLabel, unitLabel]
        items.forEach { view in
            view.translatesAutoresizingMaskIntoConstraints = false
            addSubview(view)
        }
        
        let spacing1 = spacingView()
        let spacing2 = spacingView()
        let spacing3 = spacingView()
        let spacing4 = spacingView()
                
        imageView.snp.makeConstraints { (make) in
            make.centerX.equalToSuperview()
            make.top.equalTo(spacing1.snp.bottom)
        }
        
        valueLabel.snp.makeConstraints { (make) in
            make.centerX.equalToSuperview()
            make.top.equalTo(spacing2.snp.bottom)
        }
        
        unitLabel.snp.makeConstraints { (make) in
            make.centerX.equalToSuperview()
            make.top.equalTo(spacing3.snp.bottom)
        }
        
        spacing1.snp.makeConstraints { (make) in
            make.top.centerX.equalToSuperview()
        }
        
        spacing2.snp.makeConstraints { (make) in
            make.centerX.equalToSuperview()
            make.top.equalTo(imageView.snp.bottom)
            make.height.equalTo(spacing1.snp.height)
        }
        spacing3.snp.makeConstraints { (make) in
            make.centerX.equalToSuperview()
            make.top.equalTo(valueLabel.snp.bottom)
            make.height.equalTo(spacing1.snp.height)
        }
        spacing4.snp.makeConstraints { (make) in
            make.bottom.centerX.equalToSuperview()
            make.top.equalTo(unitLabel.snp.bottom)
            make.height.equalTo(spacing1.snp.height)
        }
    
    }
    
    func spacingView() -> UIView {
        let view = UIView()
        view.translatesAutoresizingMaskIntoConstraints = false
        addSubview(view)
        return view
    }
}

class HomeCircleProgressBar: UIView {
        
    required init?(coder aDecoder: NSCoder) {
        super.init(coder: aDecoder)
        initialize()
    }
    
    override init(frame: CGRect) {
        super.init(frame: frame)
        initialize()
    }
    
    override func draw(_ rect: CGRect) {
        /// 设置背景环的path路径
        let trackLayerRect: CGRect = CGRect(x: cicleWidth, y: cicleWidth, width: self.frame.size.width-cicleWidth*2, height: self.frame.size.height-cicleWidth*2)
        let trackLayerPath: UIBezierPath = UIBezierPath(ovalIn: trackLayerRect)
        self.groundLayer.path = trackLayerPath.cgPath
        
        /// 设置进度条环的path路径
        let center: CGPoint = CGPoint(x: self.frame.size.width/2, y: self.frame.size.height/2)
        let radius: CGFloat = (self.frame.size.width-cicleWidth*2)/2
        let tintPath: UIBezierPath = UIBezierPath(arcCenter: center, radius: radius, startAngle: kAngleStart, endAngle: kAngleEnd, clockwise: true)
        self.progressLayer.path = tintPath.cgPath
        super.draw(rect)
    }
    
    func initialize() {
        self.layer.backgroundColor = UIColor.clear.cgColor;
        self.layer.addSublayer(self.groundLayer)
        self.layer.addSublayer(self.progressLayer)
    }
    
    // MARK: - Private fields
    
    /// 圆环背景颜色
    private var groundColor: UIColor = Colors.lightGray
    /// 进度条颜色
    private var progressColor: UIColor = Colors.orangeProgress
    /// 圆环宽度
    private var cicleWidth: CGFloat = 4.0
    /// 进度值（0.0-1.0）
    private var percent: CGFloat = 0
    
    private lazy var groundLayer: CAShapeLayer = {
        let layer = CAShapeLayer()
        layer.name = "circleBgLayer"
        layer.strokeColor = groundColor.cgColor
        layer.fillColor = UIColor.clear.cgColor
        layer.lineWidth = cicleWidth
        layer.lineCap = CAShapeLayerLineCap.round
        return layer
    } ()

    private lazy var progressLayer: CAShapeLayer  = {
        let layer = CAShapeLayer()
        layer.name = "progressLayer"
        layer.strokeColor = progressColor.cgColor
        layer.fillColor = UIColor.clear.cgColor
        layer.lineWidth = cicleWidth
        layer.lineCap = CAShapeLayerLineCap.round;//线头
        return layer
    } ()

    
    private let kPI_2 = Double.pi / 2
    private let kPi: CGFloat = CGFloat(Float.pi)
    private let kAngleStart: CGFloat = CGFloat(Float.pi * 1.5)
    private let kAngleEnd: CGFloat = CGFloat(Float.pi * 3.5)
    private var progress: CGFloat = 0
    
    func setProgress(_ progress: CGFloat, animated: Bool) {
        let toProgress = max(0, min(1, progress))
        if animated {
            CATransaction.begin()
            let animation = CABasicAnimation(keyPath: "strokeEnd")
            animation.isRemovedOnCompletion = false
            animation.fillMode = CAMediaTimingFillMode.forwards
            animation.duration = 1
            animation.fromValue = self.progress
            animation.toValue = toProgress
            self.progressLayer.add(animation, forKey: "progress")
            // self.progressLayer.strokeEnd = toProgress
            self.progress = toProgress
            CATransaction.commit()
        } else {
            self.progressLayer.strokeEnd = toProgress
            self.progress = toProgress
            setNeedsDisplay()
        }
    }
}

enum HomeCircleCellAttrType {
    case calorie, distance
    
    var icon: UIImage {
        switch self {
        case .calorie: return ImageRepo.HomeList.calorie
        case .distance: return ImageRepo.HomeList.distance
        }
    }
    
}

class HomeCircleCellAttr: UIControl {
    
    // MARK: - Properties
    
    var type: HomeCircleCellAttrType = .calorie
    
    var total: BDTotal = BDTotal() {
        didSet {
            switch type {
            case .calorie: valueLabel.attributedText = calorieText(Int(total.calories/1000))
            case .distance: valueLabel.attributedText = distanceText(Int(total.distance))
            }
        }
    }

    // MARK: - Initialize
    
    @available(*, unavailable)
    required init?(coder aDecoder: NSCoder) {
        fatalError("init(coder:) has not been implemented")
    }
    
    init(_ type: HomeCircleCellAttrType) {
        super.init(frame: .zero)
        self.type = type
        initialize()
    }
    
    
    private func initialize() {
        [imageView, valueLabel].forEach { view in
            view.translatesAutoresizingMaskIntoConstraints = false
            addSubview(view)
        }
        imageView.snp.makeConstraints { (make) in
            make.top.centerX.equalToSuperview();
        }
        valueLabel.snp.makeConstraints { (make) in
            make.centerX.equalToSuperview();
            make.top.equalTo(imageView.snp.bottom).offset(Constraints.small/2)
            make.bottom.equalToSuperview()
        }
    }

    private lazy var imageView: UIImageView = {
        let view = UIImageView(image: type.icon);
        view.contentMode = .scaleAspectFit
        return view
    } ()
    
    private lazy var valueLabel: UILabel = {
        let label = UILabel(style: Stylesheet.Label.text);
        label.font = .lightCustomFont(ofSize: 25)
        return label
    } ()
    
    private func calorieText(_ calorie: Int) -> NSMutableAttributedString {
        let unit = " " + SRString.Step.unit_calories.locastr
        let value = String(format: "%d", calorie);
        let text = value + unit;
        let attributesString = NSMutableAttributedString(string: text);
        let attributes: [NSAttributedString.Key: Any] = [.foregroundColor: Colors.gray, .font: UIFont.systemFont(ofSize: 15)]
        attributesString.addAttributes(attributes, range: NSRange.init(location: value.count, length: unit.count));
        return attributesString
    }
    
    private var isImperial: Bool { BandSettings.isImperial }

    private func distanceText(_ device: Int) -> NSMutableAttributedString {
        var distance: Double = BandCalculator.km(device: BDCalculatorType(device))
        if isImperial { distance = BandCalculator.mile(device: BDCalculatorType(device)) }
        let unit = " " + (isImperial ? SRString.Unit.mile.locastr : SRString.Unit.km.locastr)
        let value = String(format: "%.2f", distance);
        let text = value + unit;
        let attributesString = NSMutableAttributedString(string: text);
        let attributes: [NSAttributedString.Key: Any] = [.foregroundColor: Colors.gray, .font: UIFont.systemFont(ofSize: 15)]
        attributesString.addAttributes(attributes, range: NSRange.init(location: value.count, length: unit.count));
        return attributesString
    }
}
