//
//  HomeListItem.swift
//  CTFit
//
//  Created by Mac on 2020/11/26.
//  Copyright © 2020 jpaxh. All rights reserved.
//

import UIKit
import TableViewManager;

enum HomeListType {
    case heart, exercise, sleep
    var icon: UIImage {
        switch self {
        case .heart: return ImageRepo.HomeList.heartrate
        case .exercise: return ImageRepo.HomeList.exercise
        case .sleep: return ImageRepo.HomeList.sleep
        }
    }
}

final class HomeListItem: BaseTableViewItem {
    
    /// : - Cell datasouce
    var icon: UIImage { type.icon }
    var text: NSMutableAttributedString {
        switch type {
        case .heart: return heartrateText(Int(total.latestHeart))
        case .exercise: return exerciseTimeText(Int(total.exerciseMinutes))
        case .sleep: return sleepTimeText(Int(total.sleepMinutes))
        }
    }
    
    // MARK: - Initialize
    override init() {
        super.init()
        selectionStyle = .default
        accessoryView = .none
        cellHeight = 72
    }
    
    convenience init(_ type: HomeListType , _ total: BDTotal =  BDTotal()) {
        self.init()
        self.type = type
        self.total = total
        onHandler()
    }
    
    func update(_ total: BDTotal) {
        self.total = total
        reload()
    }
    
    // MARK: - Cell source
    private lazy var type: HomeListType = .heart
    private lazy var total: BDTotal = BDTotal()
}

extension HomeListItem {
    
    private func onHandler(){
        setSelectionHandler { [weak self] (item) in
            guard let strong = self else { return }
            strong.didSelectedHandler()
        }
    }
    
    private func didSelectedHandler() {
        switch type {
        case .heart: push(HeartRateDetailViewController())
        case .exercise: push(RecordStepViewController())
        case .sleep: push(RecordSleepViewController())
        }
    }
    
    private func push(_ vc: BaseViewController) {
        guard let currentVc = Helper.currentVc as? HomeListController else { return }
        currentVc.navigationController?.pushViewController(vc, animated: true)
    }
    
    private func heartrateText(_ heartrate: Int) -> NSMutableAttributedString {
        let unit = " " + SRString.Heart.bpm.locastr
        let value = String(format: "%d", heartrate);
        let text = value + unit;
        let attributesString = NSMutableAttributedString(string: text);
        let attributes: [NSAttributedString.Key: Any] = [.foregroundColor: Colors.gray, .font: UIFont.systemFont(ofSize: 15)]
        attributesString.addAttributes(attributes, range: NSRange.init(location: value.count, length: unit.count));
        return attributesString
    }
    
    private func exerciseTimeText(_ exerciseMinute: Int) -> NSMutableAttributedString {
        let unit = " " + SRString.Unit.minute.locastr
        let value = String(format: "%d", exerciseMinute);
        let text = value + unit;
        let attributesString = NSMutableAttributedString(string: text);
        let attributes: [NSAttributedString.Key: Any] = [.foregroundColor: Colors.gray, .font: UIFont.systemFont(ofSize: 15)]
        attributesString.addAttributes(attributes, range: NSRange.init(location: value.count, length: unit.count));
        return attributesString
    }
    
    private func sleepTimeText(_ sleepMinute: Int) -> NSMutableAttributedString {
        let unitHour = " " + SRString.Unit.hour.locastr + " "
        let unitMinute = " " + SRString.Unit.minute.locastr
        let valueHour = String(format: "%d", sleepMinute/60);
        let valueMinute = valueHour + unitHour + String(format: "%d", sleepMinute%60);
        let text = valueMinute + unitMinute;
        let attributesString = NSMutableAttributedString(string: text);
        let attributes: [NSAttributedString.Key: Any] = [.foregroundColor: Colors.gray, .font: UIFont.systemFont(ofSize: 15)]
        attributesString.addAttributes(attributes, range: NSRange.init(location: valueHour.count, length: unitHour.count));
        attributesString.addAttributes(attributes, range: NSRange.init(location: valueMinute.count, length: unitMinute.count));
        return attributesString
    }
}
