//
//  PrivacyView.swift
//  CTFit
//
//  Created by Mac on 2020/9/18.
//  Copyright © 2020 jpaxh. All rights reserved.
//

import UIKit
import WebKit
import RxSwift
import RxCocoa

class PrivacyView: UIView {
    
    func loadPrivacy() { webView.load(urlRequest) }
    
    // MARK: - Initialize
    
    required init?(coder aDecoder: NSCoder) {
        super.init(coder: aDecoder);
    }

    override init(frame: CGRect) {
        super.init(frame: frame);
        self.initialized()
    }
    
    private func initialized() {
        self.backgroundColor = Colors.Base.background;
        addSubviews()
        setupConstraints()
        if UIDevice.simulator { agreed = true }
    }
    
    private func addSubviews() {
        [webView/*, separatorLine, submitButton*/].forEach { view in
            view.translatesAutoresizingMaskIntoConstraints = false
            addSubview(view)
        }
    }

    private func setupConstraints() {
        webView.snp.makeConstraints { (make) in
            make.leading.trailing.equalToSuperview();
            if #available(iOS 11.0, *) {
                make.top.equalTo(self.safeAreaLayoutGuide.snp.top)
                make.bottom.equalTo(self.safeAreaLayoutGuide.snp.bottom).offset(-Constraints.small)
            } else {
                make.top.equalTo(self.layoutMarginsGuide.snp.top)
                make.bottom.equalTo(self.layoutMarginsGuide.snp.bottom).offset(-Constraints.small)
            }
        }
    }
    
    private var agreed: Bool = false {
        didSet {
            var title = SRString.Base.agree.locastr
            if !agreed { title = SRString.Base.disagree.locastr }
            submitBarItem.title = title
        }
    }
    
    // MARK: - View customer
    
    private let disposeBag = DisposeBag()

    private let kScriptMessageName = "BHZ5_iOS";
    
    private lazy var privacyFile: String  = {
        return "privacy-en.html"
    } ()
    
    private lazy var privacyUrl: URL  = {
        var url = Bundle.main.bundleURL
        url.appendPathComponent(privacyFile)
        return url
    } ()
    
    private lazy var urlRequest: URLRequest = URLRequest(url: privacyUrl)
    
    private lazy var contentController: WKUserContentController = {
        let controller = WKUserContentController()
        controller.add(self, name: kScriptMessageName)
        return controller
    } ()
    
    private lazy var configuration: WKWebViewConfiguration = {
        let config = WKWebViewConfiguration()
        config.userContentController = contentController
        return config
    } ()
    
    private lazy var webView: WKWebView = {
        let view = WKWebView(frame: CGRect.zero, configuration: configuration);
        view.navigationDelegate = self
        view.isHidden = true
        return view
    } ()
    
    lazy var submitBarItem: UIBarButtonItem = {
        let item = UIBarButtonItem(title: SRString.Base.disagree.locastr, style: .plain, target: self, action: #selector(onTapSubmit));
        item.tintColor = .systemBlue;
        return item;
    } ()
}

extension PrivacyView {
    @objc private func onTapSubmit() {
        assert(agreed, "Did not agree to the privacy policy")
        PrivacyView.agreed = agreed
        enterToHomeController()
    }
    
    private func enterToHomeController() {
        guard let delegate = UIApplication.shared.delegate as? AppDelegate else { return }
        delegate.enterToHomeController()
    }

    private static let key = "PrivacyState"
    public static var agreed: Bool {
        get {
            UserDefaults.standard.bool(forKey: key)
        }
        set {
            UserDefaults.standard.set(newValue, forKey: key)
            UserDefaults.standard.synchronize()
        }
    }
}

extension PrivacyView: WKScriptMessageHandler {
    func userContentController(_ userContentController: WKUserContentController, didReceive message: WKScriptMessage) {
        guard  message.name == kScriptMessageName else { return }
        self.agreed = message.body as! Bool
    }
}

extension PrivacyView: WKNavigationDelegate {
    func webView(_ webView: WKWebView, decidePolicyFor navigationAction: WKNavigationAction, decisionHandler: @escaping (WKNavigationActionPolicy) -> Void) {
        /// 超链接用 Safari 浏览器打开
        if navigationAction.navigationType == .linkActivated {
            guard let requestUrl = navigationAction.request.url else { return }
            if #available(iOS 10.0, *) {
                UIApplication.shared.open(requestUrl)
            } else {
                UIApplication.shared.openURL(requestUrl)
            }
            decisionHandler(.cancel);
        } else {
            decisionHandler(.allow);
        }
    }
    
    func webView(_ webView: WKWebView, didFinish navigation: WKNavigation!) {
        adapterDark()
    }
    
    override func traitCollectionDidChange(_ previousTraitCollection: UITraitCollection?) {
        super.traitCollectionDidChange(previousTraitCollection)
        /// 每次模式改变的时候, 这里都会执行
        adapterDark()
    }
    
    private func adapterDark() {
        var backgroundColor = "'#FFFFFF'"
        var labelColor = "'#000000'"
        if #available(iOS 13.0, *),
            traitCollection.userInterfaceStyle == .dark {
            backgroundColor = "'#000000'"
            labelColor = "'#FFFFFF'"
        }
        self.webView.evaluateJavaScript("document.body.style.backgroundColor=\(backgroundColor)", completionHandler: nil)
        self.webView.evaluateJavaScript("document.body.style.color=\(labelColor)", completionHandler: nil)
        
        DispatchQueue.main.asyncAfter(deadline: .now()+0.2) {
            let transition = CATransition()
            transition.duration = 0.2
            self.webView.isHidden = false
            self.webView.layer.add(transition, forKey: nil)
        }
    }
}
