//
//  DetectCell.swift
//  CTFit
//
//  Created by Mac on 2020/10/28.
//  Copyright © 2020 jpaxh. All rights reserved.
//

import UIKit
import SnapKit
import TableViewManager
import Charts
import RxSwift

final class DetectCell: BaseTableViewCell, ZJCellProtocol {

    // MARK: - ZJCellProtocol
    var item: DetectItem!
    typealias ZJCelltemClass = DetectItem
    override func cellWillAppear() {
        super.cellWillAppear()
        guard item != nil else { return }
        updateLineChart(item.model)
        update(item.model)
    }
    

    // MARK: - Initiaize
    override init(style: UITableViewCell.CellStyle, reuseIdentifier: String?) {
        super.init(style: .default, reuseIdentifier: reuseIdentifier)
        backgroundColor = .clear
        addSubviews()
        setConstraints()
    }
    
    // MARK: - View customer
    
    private let disposeBag = DisposeBag()

    /// 详情Stack
    private lazy var headerView: DetectHeaderView = {
        let view = DetectHeaderView()
        return view
    } ()
    
    private lazy var lineChart: LineChartView = {
        let chart = LineChartView()
        ChartsUtils.Detect.detect(chart)
        return chart
    } ()
    
    private func addSubviews() {
        [headerView, lineChart].forEach { view in
           // view.translatesAutoresizingMaskIntoConstraints = false
            addSubview(view)
        }
        
        headerView.detectButton.rx.tap.subscribe(onNext: { [weak self] _ in
            guard let strong = self else { return }
            strong.item.onDetect()
        }).disposed(by: disposeBag)
    }
    
    private func setConstraints() {
        headerView.snp.makeConstraints { (make) in
            make.left.right.top.equalToSuperview()
        }
        
        lineChart.snp.makeConstraints { (make) in
            make.left.right.bottom.equalToSuperview()
            make.top.equalTo(headerView.snp.bottom)
            make.height.equalTo(200)
        }
    }
}

extension DetectCell {
    func update(_ model: DetectModel) {
        headerView.update(model)
        
        guard model.reloadChart else { return }
        updateLineChart(model)
        setChartData(model.detectValues)
    }
    
    private func updateLineChart(_ model: DetectModel) {
        lineChart.noDataText = model.detectType.nodata_describe
        lineChart.highlightValue(nil)
        lineChart.xAxis.valueFormatter = AxisFormatterUtils.Detect.detect(model.detectValues)
        let marker = MarkerUtils.Detect(type: model.detectType, items: model.detectValues);  marker.chartView = lineChart
        lineChart.marker = marker;
        
        lineChart.dragXEnabled = true
        let xAxis = lineChart.xAxis
        xAxis.setLabelCount(7, force: true);
        xAxis.axisMinimum = 0;
        xAxis.axisMaximum = 6;
        
        let values = model.detectValues
        if values.count>7 {
            xAxis.setLabelCount(6, force: false)
            xAxis.axisMaximum = Double(values.count-1)+0.05
            lineChart.setVisibleXRangeMaximum(6.1)
            self.lineChart.layoutIfNeeded()
            DispatchQueue.main.asyncAfter(deadline: .now()+0.1) {
                self.lineChart.moveViewToX(Double(Int.max))
            }
        }
    }
    
    /// 设置图表数据: Y轴表示心率值
    private func setChartData(_ values :[BDDetect]) {
        let model = item!.model

        var entries1: [ChartDataEntry] = [ChartDataEntry]();
        var entries2: [ChartDataEntry] = [ChartDataEntry]();
        if values.count > 0 {
            for (index, value) in values.enumerated() {
                entries1.append(ChartDataEntry(x: Double(Int(index)), y: Double(Int(value.heartorsbp))));
                entries2.append(ChartDataEntry(x: Double(Int(index)), y: Double(value.spo2ordbp)));
            }
        } else {
            lineChart.data = nil;
            lineChart.notifyDataSetChanged();
            return
        }
        
        var set1: LineChartDataSet!;
        var set2: LineChartDataSet!;
        if let chartData = lineChart.data, chartData.dataSetCount > 1 {
            set1 = chartData.dataSets[0] as? LineChartDataSet;
            set2 = chartData.dataSets[1] as? LineChartDataSet;
            set1.replaceEntries(entries1)

            if model.detectType == .bpm {
                set2.replaceEntries([ChartDataEntry]())
            } else {
                set2.replaceEntries(entries2)
            }
            set2.drawFilledEnabled = false;
            
            lineChart.data?.notifyDataChanged();
            lineChart.notifyDataSetChanged();
        } else {
            set1 = createSet(entries1, 1);
            if model.detectType == .bpm {
                set2 = createSet([ChartDataEntry](), 2);
            } else {
                set2 = createSet(entries2, 2);
            }
            set2.drawFilledEnabled = false;
            let dataSets = [set1, set2];
            let data = LineChartData(dataSets: dataSets as? [IChartDataSet]);
            self.lineChart.data = data;
        }
        self.lineChart.animate(xAxisDuration:   0.5);
    }
    
    private func createSet(_  entries:[ChartDataEntry]?, _ num: Int) -> LineChartDataSet {
        let model = item!.model
        let tintColor = model.detectType.color
        
        let set = LineChartDataSet(entries: entries, label: "DetectInfoLine\(num)");
        
        set.drawValuesEnabled = false;
        set.valueTextColor = Colors.Base.foreground; //设置折线上的字体颜色

        // attributes for set
        set.lineWidth = 1;
        set.setColor(tintColor)
        set.mode = .cubicBezier;
        
        set.valueFont = UIFont.systemFont(ofSize: 9);
        set.formLineWidth = 1;
        set.formSize = 15;
        set.formLineDashLengths = [5, 2.5];
        
        /// Circle & Hole
        set.drawCirclesEnabled = true;
        set.drawCircleHoleEnabled = true;
        set.circleHoleColor = Colors.Base.background;
        set.circleColors = [tintColor];
        set.setCircleColor(tintColor)
        set.circleRadius = 2.5
        set.circleHoleRadius = 1.5;
        
        set.drawFilledEnabled = true
        let gradientColors = [tintColor.withAlphaComponent(0).cgColor, tintColor.cgColor]
        let gradient = CGGradient(colorsSpace: nil, colors: gradientColors as CFArray, locations: nil)!
        set.fillAlpha = 1
        set.fill = Fill(linearGradient: gradient, angle: 90)
        return set;
    }    
}

fileprivate class DetectHeaderView: UIView {
    var type: BDDetectType = .bpm {
        didSet {
            iconView.image = type.icon
            nameLabel.text = type.name
        }
    }
    
    func update(_ model: DetectModel) {
        type = model.detectType
        
        let text = model.detectSetup ? SRString.Detect.detecting.locastr : SRString.Detect.detect_start.locastr
        detectButton.setTitle(text, for: .normal)
        detectButton.snp.updateConstraints { (make) in
            make.width.equalTo(text.widthForFont(font: detectButton.titleLabel?.font) + Constraints.large)
        }
        
        let color = model.detectEnable ? type.color: Colors.gray
        detectButton.isUserInteractionEnabled = model.detectEnable
        detectButton.layer.borderColor = color.cgColor
        detectButton.setTitleColor(color, for: .normal)
        
        var latestText = SRString.Detect.latest_records.locastr + "："
        if let item = model.detectValues.last {
            switch type {
            case .bpm: latestText += "\(item.heartorsbp)"
            case .bp: latestText += "\(item.heartorsbp)/\(item.spo2ordbp)"
            }
            latestText += " " + type.unit + " " + "(\(item.dateText) \(item.timeText))"
        } else {
            latestText += "--"
        }
        valueLabel.text = latestText
    }
    
    @available(*, unavailable)
    required init?(coder aDecoder: NSCoder) {
        fatalError("init(coder:) has not been implemented")
    }
    
    init() {
        super.init(frame: .zero)
        addSubviews()
        setupConstraints()
    }

    /// View customer
    private func addSubviews() {
        [iconView, nameLabel, valueLabel, detectButton].forEach { view in
            view.translatesAutoresizingMaskIntoConstraints = false
            addSubview(view)
        }
    }

    private func setupConstraints() {
        
        detectButton.snp.makeConstraints { (make) in
            make.top.equalToSuperview().offset(Constraints.normal)
            make.right.equalToSuperview().offset(-Constraints.normal)
            make.height.equalTo(27)
            make.width.equalTo(110)
        }
        
        iconView.snp.makeConstraints { (make) in
            make.centerY.equalTo(detectButton.snp.centerY)
            make.left.equalToSuperview().offset(Constraints.normal)
            make.width.height.equalTo(20)
        }
        
        nameLabel.snp.makeConstraints { (make) in
            make.centerY.equalTo(iconView.snp.centerY)
            make.left.equalTo(iconView.snp.right).offset(4)
        }
        
        valueLabel.snp.makeConstraints { (make) in
            make.leading.equalTo(iconView.snp.leading)
            make.top.equalTo(iconView.snp.bottom).offset(4)
            make.height.equalTo(15)
            make.bottom.equalToSuperview()
            
        }
    }
        
    private lazy var iconView: UIImageView =  UIImageView(image: type.icon)

    private lazy var nameLabel: UILabel = {
        let label = UILabel(style: Stylesheet.Label.text)
        label.font = .customFont(ofSize: 16)
        label.text = type.name
        return label
    }()
    
    private lazy var valueLabel: UILabel = {
        let label = UILabel(style: Stylesheet.Label.description)
        label.font = .customFont(ofSize: 12)
        label.text = SRString.Detect.latest_records.locastr
        return label
    }()

    lazy var detectButton: UIButton = {
        let button = UIButton(type: .system)
        Stylesheet.View.detectButton.apply(to: button)
        button.titleLabel?.font = .customFont(ofSize: 14)
        button.layer.borderColor = type.color.cgColor
        button.setTitleColor(type.color, for: .normal)
        button.setTitle(SRString.Detect.detect_start.locastr, for: .normal)
        return button
    }()
}
