//
//  DetectItem.swift
//  CTFit
//
//  Created by Mac on 2020/10/28.
//  Copyright © 2020 jpaxh. All rights reserved.
//

import Foundation
import TableViewManager
import Repeater

extension BDDetectType {
    
    var color: UIColor {
        switch self {
        case .bpm: return Colors.red
        case .bp: return Colors.orange
        }
    }
    
    var name: String {
        switch self {
        case .bpm: return SRString.Heart.heart_rate.locastr
        case .bp: return SRString.BP.blood_pressure.locastr
        }
    }
    
    var icon: UIImage {
        switch self {
        case .bpm: return ImageRepo.Detect.heart
        case .bp: return ImageRepo.Detect.pressure
        }
    }
    
    var unit: String {
        switch self {
        case .bpm: return SRString.Heart.bpm.locastr
        case .bp: return SRString.BP.mmhg.locastr
        }
    }

    var timeout: TimeInterval {
        switch self {
        case .bpm: return 30
        case .bp: return 60
        }
    }
}

final class DetectItem: BaseTableViewItem {
    
    typealias UpdateDetectStateClosure = (_ detecting: Bool) ->Void
    
    var type: BDDetectType = .bpm
    
    lazy var model: DetectModel = {
        DetectModel(type: type)
    } ()
    
    private override init() {
        super.init()
        selectionStyle = .none
        accessoryType = .none
        isAutoHeight = true
    }
    
    convenience init(with type: BDDetectType = .bpm) {
        self.init()
        self.type = type
    }
      
    
    func onDetect()  {
        BandCommand.detect(type) { [weak self] error in
            guard let strong = self, error == nil else { return }
            strong.model.onDetecting()
        }
    }
}



class DetectModel {
    
    /// 检测类型
    var detectType: BDDetectType = .bpm
    
    /// 是否启动检测
    var detectSetup: Bool = false {
        didSet {
            if detectSetup {
                timer.reStart();  detectEnable = false
            }
        }
    }
    
    /// 检测是否可用
    var detectEnable: Bool = true {
        didSet { if detectEnable { timer.pause() } }
    }
    
    /// 是否刷新图表
    var reloadChart: Bool = true;
    
    /// 检测值项
    var detectValues = [BDDetect]() {
        didSet { reloadChart = true }
    }

    init(type detectType: BDDetectType = .bpm,
         detectValues: [BDDetect] = []) {
        self.detectType = detectType
        self.detectValues = detectValues
    }
    
    func onDetecting() {
        reloadChart = false
        detectSetup = true
        detectStateClosure?(true)
    }
    
    func onDetected() {
        reloadChart = false
        detectSetup = false
        detectEnable = true
        detectStateClosure?(false)
    }
    
    func onDetectState(_ closure: @escaping DetectStateClosure) {
        detectStateClosure = closure
    }
    
    typealias DetectStateClosure = (Bool) -> Void;
    private var detectStateClosure: DetectStateClosure?
    private lazy var timer: Repeater = {
        let repeater = Repeater.once(after: .seconds(detectType.timeout), queue: DispatchQueue.main) {
            [weak self] (_) in
            guard let strong = self else { return }
            strong.detectSetup = false
            strong.detectStateClosure?(false)
        }
        repeater.pause()
        return repeater
    } ()
}

extension BDDetectType {
    var nodata_describe: String {
        switch self {
        case .bpm: return SRString.Base.nodata_describe_detect_heart.locastr
        case .bp: return SRString.Base.nodata_describe_detect_pressure.locastr
        }
    }
}
