//
//  HeartDayCell.swift
//  CTFit
//
//  Created by Mac on 2020/6/6.
//  Copyright © 2020 shirajo. All rights reserved.
//


import UIKit
import SnapKit
import TableViewManager
import Charts
import RxSwift

final class HeartDayCell: BaseTableViewCell, ZJCellProtocol {

    // MARK: -  ZJCellProtocol
    
    var item: HeartDayItem!

    typealias ZJCelltemClass = HeartDayItem
    
    override func cellWillAppear() {
        super.cellWillAppear()
        guard item != nil else { return }
        update(item.heart)
    }
    
    // MARK: - Initialize
    
    override init(style: UITableViewCell.CellStyle, reuseIdentifier: String?) {
        super.init(style: .default, reuseIdentifier: reuseIdentifier)
        backgroundColor = .clear
        addSubviews()
        setConstraints()
    }
    
    // MARK: -  View customer
    
    private func addSubviews() {
        [lineChart, detailStack, detectButton].forEach { view in
            view.translatesAutoresizingMaskIntoConstraints = false
            addSubview(view)
        }
        
        detectButton.rx.tap.subscribe(onNext: { [weak self] _ in
            guard let strong = self else { return }
            strong.item.onDetect()
        }).disposed(by: disposeBag)
        
        if BandConnect.protocolType == .series_z,
           BandData.firmwareFeature.dateformat_detect_brightScrren {
            detectButton.isHidden = false
        } else {
            detectButton.isHidden = true
        }
    }
    
    private func setConstraints() {
        lineChart.snp.makeConstraints { (make) in
            make.left.right.equalToSuperview()
            make.top.equalToSuperview().offset(Constraints.small)
            make.height.equalTo(200)
        }
        
        detailStack.snp.makeConstraints { (make) in
            make.left.right.equalToSuperview()
            make.top.equalTo(lineChart.snp.bottom).offset(Constraints.normal+Constraints.small)
        }
        
        let text = SRString.Detect.manual_detect.locastr
        detectButton.snp.makeConstraints { (make) in
            make.centerX.equalToSuperview()
            make.top.equalTo(detailStack.snp.bottom).offset(44)
            make.bottom.equalToSuperview().offset(-Constraints.small)
            make.height.equalTo(37)
            make.width.equalTo(text.widthForFont(font: detectButton.titleLabel?.font) + Constraints.large)
        }
        separatorView()
        
    }
    
    private let disposeBag = DisposeBag()

    /// 心率图表
    private lazy var lineChart: LineChartView = {
        let chart = LineChartView()
        ChartsUtils.HeartRate.day(chart)
        return chart
    }()
    
    /// detailStack
    private lazy var detailStack: HeartDetailStack = {
        let stack = HeartDetailStack()
        return stack
    }()
    
    /// detectButton
    private lazy var detectButton: UIButton = {
        let button = UIButton(type: .system)
        button.apply(Stylesheet.View.detectButton)
        button.setTitle(SRString.Detect.manual_detect.locastr, for: .normal)
        return button
    }()
    
    var separatorColor = Colors.Chart.separator

    fileprivate func separatorView() {
        let separator1 = UIView();
        separator1.backgroundColor = separatorColor;
        addSubview(separator1);
        separator1.snp.makeConstraints { (make) in
            make.left.right.top.equalToSuperview()
            make.height.equalTo(0.5)
        }
        
        let separator2 = UIView();
        separator2.backgroundColor = separatorColor;
        addSubview(separator2);
        separator2.snp.makeConstraints { (make) in
            make.left.right.equalToSuperview()
            make.top.equalTo(lineChart.snp.bottom).offset(Constraints.normal)
            make.height.equalTo(0.5)
        }
    }
}

// MARK: - Update

extension HeartDayCell {

    private func update(_ heart: BDHeart) {
        updateChartData(heart.records)
        detailStack.update(heart)
    }
    
    /// 设置图表数据: Y轴表示10分钟一次的 静态心率数据
    private func updateChartData(_ values: [UInt8]?) {
        lineChart.highlightValue(nil)
        var entries = [ChartDataEntry]();
        if let hasValues = values {
            for (index, value) in hasValues.enumerated() {
                entries.append(ChartDataEntry(x: Double(index), y: Double(value)));
            }
        } else {
            lineChart.data = nil;
            lineChart.notifyDataSetChanged();
            return;
        }
        
        var set: LineChartDataSet!;
        if let hasData = lineChart.data, hasData.dataSetCount > 0 {
            set = hasData.dataSets[0] as? LineChartDataSet;
            set.replaceEntries(entries)
            hasData.notifyDataChanged();
            lineChart.notifyDataSetChanged();
        } else {
            set = LineChartDataSet(entries: entries, label: "Heartrate Day");
            set.valueTextColor = UIColor.clear//设置折线上的字体颜色 - 运动距离页面
            set.mode = .cubicBezier;
            set.drawCirclesEnabled = false;
            set.drawCircleHoleEnabled = false;
            set.formLineWidth = 1;
            set.formSize = 15;
            set.formLineDashLengths = [5, 2.5];
            
            set.colors = [Colors.red, Colors.blue, Colors.green];
            
            let gradientColors = [ChartColorTemplates.colorFromString("#00ff0000").cgColor,
                                  ChartColorTemplates.colorFromString("#ffff0000").cgColor]
            let gradient = CGGradient(colorsSpace: nil, colors: gradientColors as CFArray, locations: nil)!
            set?.fillAlpha = 1
            set?.fill = Fill(linearGradient: gradient, angle: 90) //.linearGradient(gradient, angle: 90)
            set.drawFilledEnabled = true // Draw the Gradient
      
            let dataSets = [set];
            let data = LineChartData(dataSets: dataSets as? [IChartDataSet]);
            self.lineChart.data = data;
        }
        lineChart.animate(xAxisDuration: 0.35);
    }
}




// MARK: - HeartDetailStack

class HeartDetailStack: UIStackView {
    
    func update(_ heart: BDHeart) {
        hightestAttr.update(heart)
        lowestAttr.update(heart)
        averageAttr.update(heart)
    }
            
    override init(frame: CGRect) {
        super.init(frame: frame);
        initialize();
    }
    
    required init(coder: NSCoder) {
        super.init(coder: coder);
        initialize();
    }
    
    fileprivate func initialize() {
        axis = .vertical;
        distribution = .fillEqually
        spacing = Constraints.small;
        addArrangedSubview(hightestAttr);
        addArrangedSubview(averageAttr);
    }
        
    private lazy var hightestAttr: HeartDayDetailAttr = HeartDayDetailAttr(type: .hightest)
    private lazy var lowestAttr: HeartDayDetailAttr = HeartDayDetailAttr(type: .lowest)
    private lazy var averageAttr: HeartDayDetailAttr = HeartDayDetailAttr(type: .average)
}


// MARK: - HeartDayDetailAttr

class HeartDayDetailAttr: UIView {
    
    enum HeartDayDetailAttrType {
        case hightest, lowest, average
        var title: String {
            switch self {
            case .hightest: return SRString.Heart.hightest_heart_rate.locastr
            case .lowest: return SRString.Heart.lowest_heart_rate.locastr
            case .average: return SRString.Heart.average_heart_rate.locastr
            }
        }
        
        var color: UIColor {
            switch self {
            case .hightest: return Colors.Heart.hightest
            case .lowest: return Colors.Heart.lowest
            case .average: return Colors.Heart.average
            }
        }
    }
    
    func update(_ heart: BDHeart) {
        var value: UInt8 = 0
        switch type {
        case .hightest: value = heart.highest
        case .lowest: value = heart.lowest
        case .average: value = heart.aver
        }
        
        timeLabel.text = ""
        guard value > 30 else { valueLabel.text = "--"; return }
        valueLabel.text = "\(value)"
        
        guard type != .average,
              let idx = heart.records.firstIndex(of: value) else {  return }
        timeLabel.text = timeText(idx)

    }
    
    private func timeText(_ idx: Int) -> String {
        return String(format: "%02d:%01d0", idx/6, idx/6)
    }
    
    // MARK: - initialize
    @available(*, unavailable)
    required init?(coder: NSCoder) {
        fatalError("init(coder:) has not been implemented")
    }
    
    init(type: HeartDayDetailAttrType = .hightest) {
        super.init(frame: .zero);
        self.type = type
        initialize();
    }
    
    // MARK: - View customer
    fileprivate func initialize() {
        [iconView, titleLabel, valueLabel, timeLabel].forEach { view in
            view.translatesAutoresizingMaskIntoConstraints = false
            addSubview(view)
        }
        
        iconView.snp.makeConstraints { (make) in
            make.centerY.equalToSuperview()
            make.left.equalToSuperview().offset(Constraints.normal)
            make.width.height.equalTo(13)
        }
        
        titleLabel.snp.makeConstraints { (make) in
            make.top.bottom.centerY.equalToSuperview()
            make.left.equalTo(iconView.snp.right).offset(Constraints.small)
            make.height.equalTo(15)
        }
        
        valueLabel.snp.makeConstraints { (make) in
            make.centerY.equalToSuperview()
            make.left.equalTo(titleLabel.snp.right)
        }
        
        timeLabel.snp.makeConstraints { (make) in
            make.centerY.equalToSuperview()
            make.right.equalToSuperview().offset(-Constraints.normal)
        }

    }
    
    private lazy var type: HeartDayDetailAttrType = .hightest
    
    private lazy var iconView: UIImageView = {
        let view = UIImageView()
        view.backgroundColor = type.color
        return view
    }()
    
    private lazy var titleLabel: UILabel = {
        let label = UILabel(style: Stylesheet.Label.text);
        label.font = .customFont(ofSize: 14)
        label.text = type.title+"："
        return label;
    }()
    
    private lazy var valueLabel: UILabel = {
        let label = UILabel(style: Stylesheet.Label.text);
        label.textColor = type.color
        label.font = .customFont(ofSize: 14)
        label.text = "--"
        return label;
    }()
    
    private lazy var timeLabel: UILabel = {
        let label = UILabel(style: Stylesheet.Label.description);
        label.font = .customFont(ofSize: 12)
        label.text = ""
        return label;
    }()
    
}
