//
//  SleepDayCell.swift
//  CTFit
//
//  Created by Mac on 2020/6/5.
//  Copyright © 2020 shirajo. All rights reserved.
//

import UIKit
import SnapKit
import TableViewManager
import Charts

final class SleepDayCell: BaseTableViewCell, ZJCellProtocol {

    // MARK: -  ZJCellProtocol
    
    var item: SleepDayItem!
    
    typealias ZJCelltemClass = SleepDayItem
    
    override func cellWillAppear() {
        super.cellWillAppear()
        guard item != nil else { return }
        update(item.sleep)
    }
    
    // MARK: - Initialize
    
    override init(style: UITableViewCell.CellStyle, reuseIdentifier: String?) {
        super.init(style: .default, reuseIdentifier: reuseIdentifier)
        backgroundColor = .clear
        addSubviews()
    }
    
    // MARK: -  View customer

    /// 睡眠详细图标
    private lazy var barChart: BarChartView = {
        let chart = BarChartView()
        ChartsUtils.Sleep.day(chart)
        return chart
    } ()

    /// 总睡眠
    private lazy var totalSleepLabel: UILabel = {
        let label = UILabel(style: Stylesheet.Label.description)
        label.font = .customFont(ofSize: 15)
        return label
    } ()
    
    /// 详情
    private lazy var sleepDetailLabel: UILabel = {
        let label = UILabel(style: Stylesheet.Label.text)
        label.font = .customFont(ofSize: 15)
        label.text = SRString.Sleep.sleep_detail.locastr
        return label
    } ()
    
    /// 睡眠时间段
    private lazy var sleepPeriodLabel: UILabel = {
        let label = UILabel(style: Stylesheet.Label.text)
        label.font = .customFont(ofSize: 12)
        label.text = String(format: "\(SRString.Sleep.sleep_time.locastr) 00:00-00:00")
        return label
    } ()
    
    /// 详情Stack
    private lazy var sleepDetailStack: SleepDetailStack = {
        let stack = SleepDetailStack()
        return stack
    } ()
    
    private func addSubviews() {
        [barChart, totalSleepLabel, sleepDetailLabel, sleepPeriodLabel, sleepDetailStack].forEach { view in
            view.translatesAutoresizingMaskIntoConstraints = false
            addSubview(view)
        }

        barChart.snp.makeConstraints { (make) in
            make.left.right.equalToSuperview()
            make.top.equalToSuperview().offset(Constraints.small)
            make.height.equalTo(200)
        }
        
        totalSleepLabel.snp.makeConstraints { (make) in
            make.top.equalTo(barChart.snp.bottom).offset(Constraints.small)
            make.left.equalToSuperview().offset(Constraints.normal)
        }
        
        sleepDetailLabel.snp.makeConstraints { (make) in
            make.top.equalTo(barChart.snp.bottom).offset(Constraints.large+Constraints.normal)
            make.centerX.equalToSuperview()
        }
        
        sleepPeriodLabel.snp.makeConstraints { (make) in
            make.top.equalTo(sleepDetailLabel.snp.bottom)
            make.centerX.equalToSuperview()
        }
        
        sleepDetailStack.snp.makeConstraints { (make) in
            make.left.equalToSuperview().offset(Constraints.normal)
            make.right.equalToSuperview().offset(-Constraints.normal)
            make.top.equalTo(sleepPeriodLabel.snp.bottom).offset(Constraints.medium)
            make.bottom.equalToSuperview().offset(-Constraints.small)
        }
        
        separatorView()
    }
    
    var separatorColor = Colors.Chart.separator

    fileprivate func separatorView() {
        let separator1 = UIView();
        separator1.backgroundColor = separatorColor;
        addSubview(separator1);
        separator1.snp.makeConstraints { (make) in
            make.left.right.top.equalToSuperview()
            make.height.equalTo(0.5)
        }
        
        let separator2 = UIView();
        separator2.backgroundColor = separatorColor;
        addSubview(separator2);
        separator2.snp.makeConstraints { (make) in
            make.left.equalToSuperview().offset(Constraints.normal)
            make.right.equalToSuperview().offset(-Constraints.normal)
            make.top.equalTo(sleepPeriodLabel.snp.bottom).offset(2)
            make.height.equalTo(0.5)
        }

    }
}

// MARK: - Update

extension SleepDayCell {
    private func update(_ sleep: BDSleep) {
        updateChartData(sleep.records.count>0 ? sleep.records : nil)
        updateTotal(Double(sleep.totalMinutes))
        updateSleepPeriod(sleep)
        updateSleepDeatil(sleep)
    }
    
    private func updateTotal(_ minute: Double) {
        let title = SRString.Sleep.total_sleep.locastr + "："
        let value = String(format: "%.1f", minute/60);
        let text = title + value + " \(SRString.Unit.hours.locastr)";
        let attributesString = NSMutableAttributedString(string: text);
        let attributes: [NSAttributedString.Key: Any] = [.foregroundColor: Colors.Base.foreground, .font: UIFont.systemFont(ofSize: 20)]
        attributesString.addAttributes(attributes, range: NSRange.init(location: title.count, length: value.count));
        totalSleepLabel.attributedText = attributesString
    }
    
    private func updateSleepPeriod(_ sleep: BDSleep) {
        sleepPeriodLabel.text = String(format: "\(SRString.Sleep.sleep_time.locastr) \(sleep.periodTime)")
    }
    
    private func updateSleepDeatil(_ sleep: BDSleep) {
        sleepDetailStack.update(sleep)
    }
    
    /// 设置图表数据: Y轴表示每分钟睡眠状态
    private func updateChartData(_ values: [BDSleepType]?) {
        barChart.highlightValue(nil)
        var entries = [BarChartDataEntry]()
        if let hasValues = values {
            var lastType = BDSleepType.none
            var startIndex: Int = 0
            for (index, type) in hasValues.enumerated() {
                if lastType != type {
                    lastType = type
                    startIndex = index
                }
                let entry = SleepTypeBarEntry.create(x: Double(index), type: type, start: startIndex)
                entries.append(entry)
            }
        } else {
            barChart.data = nil
            barChart.notifyDataSetChanged()
            barChart.setNeedsDisplay()
            return
        }
        
        var set: SleepBarSet!;
        if let hasData = barChart.barData, hasData.dataSetCount > 0 {
            set = hasData.dataSets.first as? SleepBarSet
            set.replaceEntries(entries)
            hasData.notifyDataChanged()
            barChart.notifyDataSetChanged()
        } else {
            set = SleepBarSet(entries: entries, label: "Sleep Day")
            set.drawValuesEnabled = false
            set.colors = Colors.Sleep.items
            
            var dataSets = [BarChartDataSet]();
            dataSets.append(set);
            let chartData = BarChartData(dataSets: dataSets)
            chartData.barWidth = 2
            barChart.data = chartData
            barChart.notifyDataSetChanged()
            barChart.setNeedsDisplay()

        }
        barChart.animate(yAxisDuration: 0.5)
    }
}

// MARK: - DetailStack

class SleepDetailStack: UIStackView {
    
    func update(_ sleep: BDSleep) {
        deepAttr.update(sleep)
        lightAttr.update(sleep)
        wakeAttr.update(sleep)
    }
    
    /// Initialize
    override init(frame: CGRect) {
        super.init(frame: frame)
        initialize()
    }
    
    required init(coder: NSCoder) {
        super.init(coder: coder)
        initialize()
    }
    
    private func initialize() {
        axis = .vertical
        distribution = .fillEqually
        spacing = Constraints.normal
        self.addArrangedSubview(deepAttr)
        self.addArrangedSubview(lightAttr)
        self.addArrangedSubview(wakeAttr)
    }
    
    private lazy var deepAttr: SleepDetailAttr = SleepDetailAttr(.deep)
    private lazy var lightAttr: SleepDetailAttr = SleepDetailAttr(.light)
    private lazy var wakeAttr: SleepDetailAttr = SleepDetailAttr(.wake)
}

class SleepDetailAttr: UIView {
    
    enum SleepDetailAttrType {
        case deep, light, wake
        var title: String {
            switch self {
            case .deep: return SRString.Sleep.deep_sleep.locastr
            case .light: return SRString.Sleep.light_sleep.locastr
            case .wake: return SRString.Sleep.wake.locastr
            }
        }
        var iconColor: UIColor {
            switch self {
            case .deep: return Colors.Sleep.deep
            case .light: return Colors.Sleep.light
            case .wake: return Colors.Sleep.wake
            }
        }
    }
    
    func update(_ sleep: BDSleep) {
        var value: UInt16 = 0
        switch type {
        case .deep: value = sleep.deeplMinutes
        case .light: value = sleep.lightMinutes
        case .wake: value = sleep.wakeMinutes
        }
        valueLabel.text = timeText(Int(value))
    }
    
    private func timeText(_ minute: Int) -> String {
        switch type {
        case .deep, .light: return String(format: "%d %@ %02d %@", minute/60, SRString.Unit.hours.locastr, minute%60, SRString.Unit.minute.locastr)
        case .wake: return String(format: "%d %@", minute, SRString.Unit.minute.locastr)
        }
    }
    
  
    // MARK: initialize
    @available(*, unavailable)
    required init?(coder: NSCoder) {
        fatalError("init(coder:) has not been implemented")
    }
    
    init(_ type: SleepDetailAttrType = .deep) {
        super.init(frame: .zero);
        self.type = type
        initialize();
    }
    
    
    // MARK: - View customer
    private func initialize() {
        [iconView, titleLabel, valueLabel].forEach { view in
            view.translatesAutoresizingMaskIntoConstraints = false
            addSubview(view)
        }
        
        iconView.snp.makeConstraints { (make) in
            make.left.centerY.equalToSuperview()
            make.width.height.equalTo(13)
        }
        
        titleLabel.snp.makeConstraints { (make) in
            make.top.bottom.centerY.equalToSuperview()
            make.height.equalTo(15)
            make.left.equalTo(iconView.snp.right).offset(Constraints.small)
        }
        
        valueLabel.snp.makeConstraints { (make) in
            make.right.centerY.equalToSuperview()
        }
    }
    
    private lazy var type: SleepDetailAttrType = .deep

    
    private lazy var iconView: UIImageView = {
        let view = UIImageView()
        view.backgroundColor = type.iconColor
        return view
    }()
    
    private lazy var titleLabel: UILabel = {
        let label = UILabel(style: Stylesheet.Label.text)
        label.text = type.title
        return label
    }()
    
    private lazy var valueLabel: UILabel = {
        let label = UILabel(style: Stylesheet.Label.text)
        label.text = "0分钟"
        return label
    }()
}

