//
//  ExerciseCell.swift
//  CTFit
//
//  Created by Mac on 2020/6/5.
//  Copyright © 2020 shirajo. All rights reserved.
//

import UIKit
import SnapKit
import TableViewManager
import Charts

final class ExerciseCell: BaseTableViewCell, ZJCellProtocol {

    // MARK: - ZJCellProtocol
    
    var item: ExerciseItem!
    
    typealias ZJCelltemClass = ExerciseItem
    
    override func cellWillAppear() {
        super.cellWillAppear()
        guard item != nil else { return }
        update(item.exerciseItem)
    }
    
    // MARK: - Initialize
    
    override init(style: UITableViewCell.CellStyle, reuseIdentifier: String?) {
        super.init(style: .default, reuseIdentifier: reuseIdentifier)
        backgroundColor = .clear
        addSubviews()
    }
    
    private func addSubviews() {
        [headerView, stackView1, stackView2, stackView3, lineChart].forEach { view in
            view.translatesAutoresizingMaskIntoConstraints = false
            addSubview(view)
        }
        
        headerView.snp.makeConstraints { (make) in
            make.left.top.right.equalToSuperview()
            make.height.equalTo(50)
        }
        
        stackView1.snp.makeConstraints { (make) in
            make.left.equalToSuperview().offset(Constraints.normal);
            make.right.equalToSuperview().offset(-Constraints.normal);
            make.top.equalTo(headerView.snp.bottom)
        }
    
        stackView2.snp.makeConstraints { (make) in
            make.left.equalTo(stackView1.snp.left)
            make.right.equalTo(stackView1.snp.right)
            make.top.equalTo(stackView1.snp.bottom)
        }
        
        stackView3.snp.makeConstraints { (make) in
            make.left.equalTo(stackView1.snp.left)
            make.right.equalTo(stackView1.snp.right)
            make.top.equalTo(stackView2.snp.bottom)
        }
        
        lineChart.snp.makeConstraints { (make) in
            make.left.right.equalToSuperview();
            make.top.equalTo(stackView3.snp.bottom).offset(Constraints.normal);
            make.bottom.equalToSuperview().offset(-Constraints.normal);
            make.height.equalTo(140);
        }
        
        separatorView()
    }
    
    // MARK: - View customer
    
    /// 训练图标
    private lazy var exerciseImageView: UIImageView = {
        let imageView = UIImageView()
        imageView.contentMode = .scaleAspectFit
        imageView.image = BDSportType.walk.image
        return imageView
    }()
    /// 训练名称
    private lazy var exerciseNameLabel: UILabel = {
        let lable = UILabel(style: Stylesheet.Label.text)
        lable.text = BDSportType.walk.text
        return lable
    }()
    /// 训练时间段
    private lazy var exercisePeriodLabel: UILabel = {
        let lable = UILabel(style: Stylesheet.Label.text)
        lable.text = "19:40~00:00"
        return lable
    }()
    /// 时间
    private lazy var durationAttr: ExerciseAttr = { ExerciseAttr()  }()
    /// 步数，距离，卡路里
    private lazy var integrateAttr: ExerciseAttr = { ExerciseAttr() }()
    /// 平均步频 SF = Steps Frequency
    private lazy var averageSFAttr: ExerciseAttr = { ExerciseAttr() }()
    /// 最高步频
    private lazy var highestSFAttr: ExerciseAttr = { ExerciseAttr() }()
    /// 平均心率
    private lazy var averageHRAttr: ExerciseAttr = { ExerciseAttr() }()
    /// 最高心率
    private lazy var highestHRAttr: ExerciseAttr = { ExerciseAttr() }()
    /// 心率图表
    private lazy var lineChart: LineChartView = {
        let chart = LineChartView()
        ChartsUtils.HeartRate.exercise(chart)
        return chart
    }()
    
    private lazy var headerView: UIView = {
        let view = UIView()
        [exerciseImageView, exerciseNameLabel, exercisePeriodLabel].forEach { e in
            e.translatesAutoresizingMaskIntoConstraints = false
            view.addSubview(e)
        }
        exerciseImageView.snp.makeConstraints { (make) in
            make.centerY.equalToSuperview()
            make.left.equalToSuperview().offset(Constraints.normal);
            make.width.height.equalTo(30)
        }
        exerciseNameLabel.snp.makeConstraints { (make) in
            make.centerY.equalToSuperview()
            make.left.equalTo(exerciseImageView.snp.right).offset(Constraints.normal);
        }
        exercisePeriodLabel.snp.makeConstraints { (make) in
            make.centerY.equalToSuperview()
            make.left.equalTo(exerciseNameLabel.snp.right).offset(Constraints.normal);
        }
        return view
    } ()
    
    private lazy var stackView1: UIStackView = {
        let view = stackView()
        view.addArrangedSubview(durationAttr)
        view.addArrangedSubview(integrateAttr)
        return view
    } ()
    
    private lazy var stackView2: UIStackView = {
        let view = stackView()
        view.addArrangedSubview(averageSFAttr)
        view.addArrangedSubview(highestSFAttr)
        return view
    } ()
    private lazy var stackView3: UIStackView = {
        let view = stackView()
        view.addArrangedSubview(averageHRAttr)
        view.addArrangedSubview(highestHRAttr)
        return view
    } ()
    private func stackView() -> UIStackView {
        let stackView = UIStackView()
        stackView.axis = .horizontal
        stackView.distribution = .fillEqually
        stackView.translatesAutoresizingMaskIntoConstraints = false
        return stackView
    }
    
    
    private var separatorColor = Colors.Chart.separator
    private func separatorView() {
        let separator1 = UIView();
        separator1.backgroundColor = separatorColor;
        stackView1.addSubview(separator1);
        separator1.snp.makeConstraints { (make) in
            make.top.left.right.equalToSuperview()
            make.height.equalTo(0.5);
        }
        
        let separator2 = UIView();
        separator2.backgroundColor = separatorColor;
        stackView2.addSubview(separator2);
        separator2.snp.makeConstraints { (make) in
            make.top.left.right.equalToSuperview()
            make.height.equalTo(0.5);
        }
        
        let separator3 = UIView();
        separator3.backgroundColor = separatorColor;
        stackView3.addSubview(separator3);
        separator3.snp.makeConstraints { (make) in
            make.top.left.right.equalToSuperview()
            make.height.equalTo(0.5);
        }
            
        let separator4 = UIView();
        separator4.backgroundColor = separatorColor;
        stackView3.addSubview(separator4);
        separator4.snp.makeConstraints { (make) in
            make.left.right.equalToSuperview()
            make.bottom.equalToSuperview().offset(-0.5)
            make.height.equalTo(0.5);
        }
        
        let separator5 = UIView();
        separator5.backgroundColor = separatorColor;
        addSubview(separator5);
        separator5.snp.makeConstraints { (make) in
            make.centerX.equalToSuperview().offset(-0.5)
            make.top.equalTo(self.stackView1.snp.top)
            make.bottom.equalTo(self.stackView3.snp.bottom)
            make.width.equalTo(0.5);
        }
        
        let separator6 = UIView();
        separator6.backgroundColor = separatorColor;
        addSubview(separator6);
        separator6.snp.makeConstraints { (make) in
            make.left.right.equalToSuperview()
            make.bottom.equalToSuperview().offset(-0.5)
            make.height.equalTo(0.5);
        }
    }
}

// MARK: - Update

extension ExerciseCell {
    
    private var isImperial: Bool { BandSettings.isImperial }

    private func update(_ model: BDExerciseItem) {
        
        // icon + name + period
        exerciseImageView.image = model.type.image
        exerciseNameLabel.text = model.type.text
        exercisePeriodLabel.text = model.periodTime
        
        /// minutes
        durationAttr.title = SRString.Exercise.exercise_time.locastr
        durationAttr.centerValue = " "
        durationAttr.value = String(format: "%d %@", model.duration, SRString.Unit.minute.locastr);

        
        /// steps + distance + calories
        if model.type == .bike {
            setRidingAttr(Int(model.calories));
        } else {
            var distance = BandCalculator.km(device: BDCalculatorType(model.distance))
            if isImperial { distance = BandCalculator.mile(device: BDCalculatorType(model.distance)) }
            setIntegrateAttr(Int(model.steps), Float(distance), Int(model.calories));
        }
        
        /// averageSF + highestSF
        averageSFAttr.title = SRString.Exercise.average_pace_step.locastr
        highestSFAttr.title = SRString.Exercise.highest_pace_step.locastr
        averageSFAttr.value = String(model.avgSpace);
        highestSFAttr.value = String(model.maxSpace);
        
        // averageHR + highestHR
        averageHRAttr.title = SRString.Heart.average_heart_rate.locastr
        highestHRAttr.title = SRString.Heart.hightest_heart_rate.locastr
        averageHRAttr.value = String(model.avgHeart);
        highestHRAttr.value = String(model.maxHeart);
        
        /// heartrate records
        updateChartData(values: model.heartRecords);
    }
    
    /// 设置骑行模式下的数据：骑行不会统计步数，只有燃烧卡路里
    private func setRidingAttr(_ calories: Int) {
        integrateAttr.title = SRString.Exercise.calories_burn.locastr
        integrateAttr.centerValue = ""
        integrateAttr.value = String(format: "%d %@", calories, SRString.Step.unit_calories.locastr);
    }
    
    /// 设置统计的运动数据：步数+距离+卡路里
    private func setIntegrateAttr(_ steps: Int, _ distance: Float, _ calories: Int) {
        integrateAttr.title = String(format: "%d %@", steps, SRString.Step.unit_step.locastr);
        let distanceUnit = isImperial ? SRString.Unit.mile.locastr : SRString.Unit.km.locastr
        integrateAttr.centerValue = String(format: "%.2f %@", distance, distanceUnit);
        integrateAttr.value = String(format: "%d %@", calories, SRString.Step.unit_calories.locastr);
    }

    /// Y轴表示训练时间内每分钟的心率
    private func updateChartData(values: [UInt8]?) {
        lineChart.highlightValue(nil)
        var entries = [ChartDataEntry]();
        if let hasValues = values {
            for (index, value) in hasValues.enumerated() {
                entries.append(ChartDataEntry(x: Double(index), y: Double(value)));
            }
            // 设置心率图表Marker的起始时间'
            if let hasMarker = lineChart.marker as? MarkerUtils.ExerciseHeartRate {
                let model = item.exerciseItem
                hasMarker.startHour = Int(model.startHour);
                hasMarker.startMinute = Int(model.startMinute);
            }
        } else {
            lineChart.data = nil;
            lineChart.notifyDataSetChanged();
            return;
        }
        
        var set: LineChartDataSet!;
        if let hasData = lineChart.data, hasData.dataSetCount > 0 {
            set = hasData.dataSets[0] as? LineChartDataSet;
            set.replaceEntries(entries)
            hasData.notifyDataChanged();
            lineChart.notifyDataSetChanged();
        } else {
            set = LineChartDataSet(entries: entries, label: "Exercise Heartrate");
            set?.valueTextColor = UIColor.clear//设置折线上的字体颜色 - 运动距离页面
            set.mode = .cubicBezier;
            set.drawCirclesEnabled = false;
            set.drawCircleHoleEnabled = false;
            set.formLineWidth = 1;
            set.formSize = 15;
            set.formLineDashLengths = [5, 2.5];
            
            set.colors = [UIColor.green, UIColor.yellow, UIColor.red];
            set.drawFilledEnabled = true // Draw the Gradient
      
            let dataSets = [set];
            let data = LineChartData(dataSets: dataSets as? [IChartDataSet]);
            self.lineChart.data = data;
        }
    }
}

// MARK: - Detail view
private let attrMargin = 5
private let labelHeight = 23
class ExerciseAttr: UIView {
    
    var title: String? {
        get { return titleLabel.text }
        set { titleLabel.text = newValue }
    }
    
    var value: String? {
        get { return valueLabel.text }
        set { valueLabel.text = newValue }
    }
    
    var centerValue: String? {
        get { return centerLabel.text }
        set { centerLabel.text = newValue }
    }
    
    // MARK: - Initialize
    
    @available(*, unavailable)
    required init?(coder aDecoder: NSCoder) {
        fatalError("init(coder:) has not been implemented")
    }
    
    override init(frame: CGRect) {
        super.init(frame: frame);
        addSubviews();
    }
    
    private func addSubviews() {
        [titleLabel, valueLabel].forEach { view in
            view.translatesAutoresizingMaskIntoConstraints = false
            addSubview(view)
        }
        titleLabel.snp.makeConstraints { (make) in
            make.left.right.equalToSuperview()
            make.top.equalToSuperview().offset(attrMargin)
            make.height.equalTo(labelHeight)
        }
        valueLabel.snp.makeConstraints { (make) in
            make.left.right.equalToSuperview()
            make.top.equalTo(self.titleLabel.snp.bottom)
            make.bottom.equalToSuperview().offset(-attrMargin)
            make.height.equalTo(labelHeight)
        }
    }
    
    // MARK: - View customer
    private lazy var titleLabel: UILabel = {
        let label = UILabel(style: Stylesheet.Label.text);
        label.text = "Label"
        return label;
    }()
    private lazy var valueLabel: UILabel = {
        let label = UILabel(style: Stylesheet.Label.text);
        label.text = "Value"
        return label;
    }()
    private lazy var centerLabel: UILabel = {
        let label = UILabel(style: Stylesheet.Label.text);
        [label].forEach { view in
            view.translatesAutoresizingMaskIntoConstraints = false
            addSubview(view)
        }
        label.snp.makeConstraints { (make) in
            make.left.right.equalToSuperview()
            make.top.equalTo(self.titleLabel.snp.bottom)
            make.height.equalTo(labelHeight)
        }
        valueLabel.snp.remakeConstraints { (make) in
            make.left.right.equalToSuperview()
            make.top.equalTo(label.snp.bottom)
            make.bottom.equalToSuperview().offset(-attrMargin)
            make.height.equalTo(labelHeight)
        }
        return label;
    } ()
}
