//
//  StepDayCell.swift
//  CTFit
//
//  Created by Mac on 2020/6/4.
//  Copyright © 2020 shirajo. All rights reserved.
//

import UIKit
import SnapKit
import TableViewManager
import Charts

final class StepDayCell: BaseTableViewCell, ZJCellProtocol {

    // MARK: - ZJCellProtocol
    
    var item: StepDayItem!
    
    typealias ZJCelltemClass = StepDayItem
    
    override func cellWillAppear() {
        super.cellWillAppear()
        guard item != nil else { return }
        update(item.total, item.step)
    }
    
    // MARK: - Initialize

    override init(style: UITableViewCell.CellStyle, reuseIdentifier: String?) {
        super.init(style: .default, reuseIdentifier: reuseIdentifier)
        backgroundColor = .clear
        addSubviews()
    }
    
    // MARK: - View customer
    
    private func addSubviews() {
        [barChart, caloriesCRLabel, stepDetailStack].forEach { view in
            view.translatesAutoresizingMaskIntoConstraints = false
            addSubview(view)
        }
        
        barChart.snp.makeConstraints { (make) in
            make.left.right.equalToSuperview()
            make.top.equalToSuperview().offset(Constraints.small)
            make.height.equalTo(200)
        }
        
        caloriesCRLabel.snp.makeConstraints { (make) in
            make.centerX.equalToSuperview()
            make.top.equalTo(barChart.snp.bottom).offset(Constraints.small)
        }
        
        stepDetailStack.snp.makeConstraints { (make) in
            make.left.right.bottom.equalToSuperview()
            make.top.equalTo(caloriesCRLabel.snp.bottom).offset(Constraints.small)
        }
        
        separatorView()
    }
    
    /// BarChartView
    private lazy var barChart: BarChartView = {
        let chart = BarChartView()
        ChartsUtils.Setp.day(chart)
        return chart
    } ()
    /// 卡路里完成率
    private lazy var caloriesCRLabel: UILabel = {
        let label = UILabel(style: Stylesheet.Label.text)
        label.font = .customFont(ofSize: 13)
        label.attributedText = calorieCRText(0)
        return label
    } ()
    /// 详情Stack
    private lazy var stepDetailStack: StepDetailStack = {
        let stack = StepDetailStack()
        return stack
    } ()
    
    /// Set. separator
    private var separatorColor = Colors.Chart.separator
    private func separatorView() {
        let separator1 = UIView();
        separator1.backgroundColor = separatorColor;
        addSubview(separator1);
        separator1.snp.makeConstraints { (make) in
            make.left.right.top.equalToSuperview()
            make.height.equalTo(0.5)
        }
        
        let separator2 = UIView();
        separator2.backgroundColor = separatorColor;
        stepDetailStack.addSubview(separator2);
        separator2.snp.makeConstraints { (make) in
            make.left.right.top.equalToSuperview()
            make.height.equalTo(0.5)
        }
        
        let separator3 = UIView();
        separator3.backgroundColor = separatorColor;
        addSubview(separator3);
        separator3.snp.makeConstraints { (make) in
            make.left.right.equalToSuperview()
            make.bottom.equalToSuperview().offset(-0.5)
            make.height.equalTo(0.5)
        }
    }
    
}

// MARK: - Update

extension StepDayCell {
    
    private func update(_ total: BDTotal?, _ step: BDStep?) {
        updateChartData(step?.hourRecords)
        updateStepDeatil(total ?? BDTotal())
        caloriesCRLabel.attributedText = calorieCRText(Int(total?.calories ?? 0))
    }
        
    private var goalCalories: CGFloat { CGFloat(BandSettings.goal.calories) * 1000 }
    
    private func calorieCR(_ calories: Int) -> Int {
        guard goalCalories > 0 else { return 0 }
        let percent = CGFloat(calories) / CGFloat(goalCalories)
        return Int(percent*100)
    }
    
    private func calorieCRText(_ calories: Int) -> NSMutableAttributedString {
        let title = SRString.Step.calories_cr.locastr + "："
        let precess = String(format: "%d%%", calorieCR(calories));
        let text = title + precess;
        let attributesString = NSMutableAttributedString(string: text);
        let attributes: [NSAttributedString.Key: Any] = [.foregroundColor: Colors.Base.foreground, .font: UIFont.systemFont(ofSize: 20)]
        attributesString.addAttributes(attributes, range: NSRange.init(location: title.count, length: precess.count));
        return attributesString
    }
    
    private func updateStepDeatil(_ total: BDTotal) {
        stepDetailStack.distanceAttr.value = Int(total.distance)
        stepDetailStack.stepAttr.value = Int(total.steps)
        stepDetailStack.caloriesAttr.value = Int(total.calories/1000)
    }
    
    /// 设置图表数据: Y轴表示每小时内运动步数
    private func updateChartData(_ values: [UInt16]?) {
        barChart.highlightValue(nil)
        var entries: [BarChartDataEntry] = [BarChartDataEntry]();
        
        if let hasValues = values {
            for (index, value) in hasValues.enumerated() {
                entries.append(BarChartDataEntry(x: Double(index), y: Double(value)));
            }
        } else {
            barChart.data = nil;
            barChart.notifyDataSetChanged();
            return;
        }
        
        if let hasData = barChart.barData, hasData.dataSetCount > 0,
            let set = hasData.dataSets.first as? BarChartDataSet {
            set.replaceEntries(entries)
            hasData.notifyDataChanged();
            barChart.notifyDataSetChanged();
        } else {
            let set = BarChartDataSet(entries: entries, label: "Steps");
            set.drawValuesEnabled = false;
            set.setColor(Colors.blue)
            var dataSets = [BarChartDataSet]();
            dataSets.append(set);
            
            let chartData = BarChartData(dataSets: dataSets);
            chartData.barWidth = 0.7
            barChart.data = chartData;
        }
        barChart.animate(yAxisDuration: 0.35);
    }
}


// MARK: - DetailStack

final class StepDetailStack: UIStackView {
    
   lazy var distanceAttr: SportAttr = {
        let item = SportAttr(type: .distance)
        return item
    }()

    lazy var stepAttr: SportAttr = {
        let item = SportAttr(type: .step)
        return item
    }()

    lazy var caloriesAttr: SportAttr = {
        let item = SportAttr(type: .calories)
        return item
    }()
    
    required init(coder: NSCoder) {
        super.init(coder: coder);
        initialize();
    }
    
    override init(frame: CGRect) {
        super.init(frame: frame);
        initialize();
    }

    private func initialize() {
        axis = .horizontal
        distribution = .fillEqually
        self.addArrangedSubview(distanceAttr)
        self.addArrangedSubview(stepAttr)
        self.addArrangedSubview(caloriesAttr)
    }
}


enum SportDataType {
    
    case distance, step, calories
    
    var scale: CGFloat { 0.85 }
    
    var image: UIImage {
        switch self {
        case .distance: return ImageRepo.HomeCircle.distance.scaleImage(scale)
        case .step: return ImageRepo.HomeCircle.steps.scaleImage(scale)
        case .calories: return ImageRepo.HomeCircle.calorie.scaleImage(scale)
        }
    }
    
    private var isImperial: Bool { BandSettings.isImperial }
    
    var unit: String {
        switch self {
        case .distance: return isImperial ? SRString.Unit.mile.locastr : SRString.Unit.km.locastr
        case .step: return SRString.Step.unit_step.locastr
        case .calories: return SRString.Step.unit_calories.locastr
        }
    }
    
    
    func perUnit(style: IntervalStyle = .day) -> String {
        return unit + style.perUnit
    }
}

class SportAttr: UIControl {
    
    private var type: SportDataType = .distance
    
    private var isImperial: Bool { BandSettings.isImperial }

    var value: Int = 0 {
        didSet {
            var valueText = "\(value)"
            if type == .distance {
                var distance = BandCalculator.km(device: BDCalculatorType(value))
                if isImperial { distance = BandCalculator.mile(device: BDCalculatorType(value)) }
                valueText = String(format: "%0.1f", distance)
            }
            valueLabel.text = valueText + " " + type.unit
        }
    }
    
    private lazy var imageView: UIImageView = {
        let view = UIImageView()
        view.contentMode = .scaleAspectFit
        view.image = self.type.image
        return view
    }()
    
    private lazy var valueLabel: UILabel = {
        let label = UILabel(style: Stylesheet.Label.text)
        label.font = .customFont(ofSize: 13)
        label.textAlignment = .center
        return label
    }()

    // MARK: - Initialize
    
    @available(*, unavailable)
    required init?(coder: NSCoder) {
        fatalError("init(coder:) has not been implemented")
    }
    
    init(type: SportDataType = .distance) {
        super.init(frame: .zero)
        self.type = type
        addSubviews()
        setupConstraints()
    }
    
    private func addSubviews() {
        [imageView, valueLabel].forEach { view in
            view.translatesAutoresizingMaskIntoConstraints = false
            addSubview(view)
        }
    }

    private func setupConstraints() {
        imageView.snp.makeConstraints { (make) in
            make.centerX.equalToSuperview()
            make.top.equalToSuperview().offset(Constraints.small)
        }
        
        valueLabel.snp.makeConstraints { (make) in
            make.centerX.equalToSuperview()
            make.top.equalTo(imageView.snp.bottom).offset(Constraints.small)
            make.bottom.equalToSuperview().offset(-Constraints.small)
        }
    }
}
