//
//  SleepIntervalCell.swift
//  CTFit
//
//  Created by Mac on 2020/6/10.
//  Copyright © 2020 shirajo. All rights reserved.
//

import UIKit
import SnapKit
import TableViewManager
import Charts

final class SleepIntervalCell: BaseTableViewCell, ZJCellProtocol {

    // MARK: - ZJCellProtocol
    
    var item: SleepIntervalItem! {
        didSet {
            _ = updateBarChart
        }
    }
    
    typealias ZJCelltemClass = SleepIntervalItem
    
    override func cellWillAppear() {
        super.cellWillAppear()
        guard item != nil else { return }
        update(item.sleepItem)
    }
    
    // MARK: - Initialize
    
    override init(style: UITableViewCell.CellStyle, reuseIdentifier: String?) {
        super.init(style: .default, reuseIdentifier: reuseIdentifier)
        backgroundColor = .clear
        addSubviews()
        setConstraints()
    }
    
    // MARK: - View customer
    
    private lazy var updateBarChart: Bool = {
        switch item.intervalStyle {
        case .week: ChartsUtils.Sleep.week(barChart)
        case .month: ChartsUtils.Sleep.month(barChart)
        case .year: ChartsUtils.Sleep.year(barChart)
        }
        return true
    }()
    
    private lazy var barChart: BarChartView = {
        let chart = BarChartView()
        return chart
    } ()
    
    /// 详情
    private lazy var sleepDetailLabel: UILabel = {
        let label = UILabel(style: Stylesheet.Label.text)
        label.font = .customFont(ofSize: 15)
        label.text = SRString.Sleep.sleep_detail.locastr
        return label
    } ()
    
    /// 睡眠时间段
    private lazy var sleepPeriodLabel: UILabel = {
        let label = UILabel(style: Stylesheet.Label.text)
        label.font = .customFont(ofSize: 12)
        label.text = item?.sleepItem.timePerid
        return label
    } ()
    
    /// 详情Stack
    private lazy var detailStack: SleepIntervalDetailStack = {
        let stack = SleepIntervalDetailStack()
        return stack
    } ()
    
    private func addSubviews() {
        [barChart, sleepDetailLabel, sleepPeriodLabel, detailStack].forEach { view in
            view.translatesAutoresizingMaskIntoConstraints = false
            addSubview(view)
        }
    }
    
    private func setConstraints() {
        barChart.snp.makeConstraints { (make) in
            make.top.equalToSuperview().offset(Constraints.small)
            make.left.right.equalToSuperview()
            make.height.equalTo(200)
        }
        
        sleepDetailLabel.snp.makeConstraints { (make) in
            make.top.equalTo(barChart.snp.bottom).offset(Constraints.large)
            make.centerX.equalToSuperview()
        }
        
        sleepPeriodLabel.snp.makeConstraints { (make) in
            make.top.equalTo(sleepDetailLabel.snp.bottom).offset(0)
            make.centerX.equalToSuperview()
        }
        
        detailStack.snp.makeConstraints { (make) in
            make.left.equalToSuperview().offset(Constraints.large)
            make.right.equalToSuperview().offset(-Constraints.small)
            make.top.equalTo(sleepPeriodLabel.snp.bottom).offset(Constraints.medium)
            make.bottom.equalToSuperview().offset(-Constraints.small)
        }
        
        separatorView()
    }
    
    var separatorColor = Colors.Chart.separator

    fileprivate func separatorView()  {
        let separator1 = UIView();
        separator1.backgroundColor = separatorColor;
        addSubview(separator1);
        separator1.snp.makeConstraints { (make) in
            make.left.right.top.equalToSuperview()
            make.height.equalTo(0.5)
        }
        
        let separator2 = UIView();
        separator2.backgroundColor = separatorColor;
        addSubview(separator2);
        separator2.snp.makeConstraints { (make) in
            make.left.equalToSuperview().offset(Constraints.normal)
            make.right.equalToSuperview().offset(-Constraints.normal)
            make.top.equalTo(sleepPeriodLabel.snp.bottom).offset(2)
            make.height.equalTo(0.5)
        }
    }
    
}

// MARK: - Update

extension SleepIntervalCell {
    
    private func update(_ model: SleepIntervalModel) {
        updateStepDeatil(model)
        if model.intervalStyle == .month {
            ChartsUtils.upateMonthXAxis(barChart, date: model.dateInterval.start)
        }
        updateChartData(model.records)
    }
    
    private func updateStepDeatil(_ model: SleepIntervalModel) {
        sleepPeriodLabel.text = model.timePerid
        detailStack.update(model)
    }
    
    /// 设置图表数据: Y轴表示每小时内运动步数
    private func updateChartData(_ values: [Double?]?) {
        barChart.highlightValue(nil)
        var entries: [BarChartDataEntry] = [BarChartDataEntry]();
        if let hasValues = values {
            for (index, value) in hasValues.enumerated() {
                if value == 0 { continue }
                let val =  Double(value ?? 0)/60
                entries.append(BarChartDataEntry(x: Double(index), y: val));
            }
        } else {
            barChart.data = nil;
            barChart.notifyDataSetChanged();
            return;
        }
        
        if let hasData = barChart.barData, hasData.dataSetCount > 0,
            let set = hasData.dataSets.first as? BarChartDataSet {
            set.replaceEntries(entries)
            hasData.notifyDataChanged();
            barChart.notifyDataSetChanged();
        } else {
            let set = BarChartDataSet(entries: entries, label: "StepsInterval\(item.intervalStyle)");
            set.drawValuesEnabled = false;
            set.setColor(Colors.blue)
            var dataSets = [BarChartDataSet]();
            dataSets.append(set);
            
            let chartData = BarChartData(dataSets: dataSets);
            chartData.barWidth = 0.7
            barChart.data = chartData;
        }
        barChart.animate(yAxisDuration: 0.35);
    }
}


// MARK: - DetailStack

final class SleepIntervalDetailStack: UIView {
    func update(_ model: SleepIntervalModel) {
        let perUnit = model.intervalStyle.dateBarStyle.perUnit
        totalAttr.value = String(format: "%0.1f %@", Float(model.totalMinutes)/60, SRString.Unit.hours.locastr)
        deepAttr.value = String(format: "%0.1f %@", Float(model.deepMinutes)/60, SRString.Unit.hours.locastr)
        averageTotalAttr.value = String(format: "%0.1f %@%@", Float(model.averageTotalMinutes)/60, SRString.Unit.hours.locastr, perUnit)
        averageDeepAttr.value = String(format: "%0.1f %@%@", Float(model.averageDeepMinutes)/60, SRString.Unit.hours.locastr, perUnit)
    }
    
    /// Initialize
    
    @available(*, unavailable)
    required init?(coder aDecoder: NSCoder) {
        fatalError("init(coder:) has not been implemented")
    }
    
    init() {
        super.init(frame: .zero)
        addSubviews()
        setupConstraints()
    }

    /// View customer
    private func addSubviews() {
        [totalAttr, averageTotalAttr, deepAttr, averageDeepAttr].forEach { view in
            view.translatesAutoresizingMaskIntoConstraints = false
            addSubview(view)
        }
    }

    private func setupConstraints() {
        totalAttr.snp.makeConstraints { (make) in
            make.left.top.equalToSuperview()
            make.width.height.equalToSuperview().dividedBy(2)
        }
        
        averageTotalAttr.snp.makeConstraints { (make) in
            make.right.top.equalToSuperview()
            make.width.height.equalToSuperview().dividedBy(2)
        }
        
        deepAttr.snp.makeConstraints { (make) in
            make.left.bottom.equalToSuperview()
            make.width.height.equalToSuperview().dividedBy(2)
        }
        
        averageDeepAttr.snp.makeConstraints { (make) in
            make.right.bottom.equalToSuperview()
            make.width.height.equalToSuperview().dividedBy(2)
        }
    }
    
    private lazy var totalAttr: SleepIntervalAttr = {
        let attr = SleepIntervalAttr()
        attr.title = SRString.Sleep.total_sleep.locastr + ":"
        return attr
    }()

    private lazy var averageTotalAttr: SleepIntervalAttr = {
        let attr = SleepIntervalAttr()
        attr.title = SRString.Sport.average.locastr + ":"
        return attr
    }()
    
    private lazy var deepAttr: SleepIntervalAttr = {
        let attr = SleepIntervalAttr()
        attr.title = SRString.Sleep.total_deep.locastr + ":"
        return attr
    }()
    
    private lazy var averageDeepAttr: SleepIntervalAttr = {
        let attr = SleepIntervalAttr()
        attr.title = SRString.Sport.average.locastr + ":"
        return attr
    }()
}



class SleepIntervalAttr: UIControl {
    
    var title: String = "Sleep" {
        didSet {
            titleLabel.text = title
        }
    }
    
    var value: String = "0 时" {
        didSet {
            valueLabel.text = value
        }
    }
    
    private lazy var titleLabel: UILabel = {
        let label = UILabel(style: Stylesheet.Label.text)
        label.textAlignment = .center
        return label
    }()
    
    private lazy var valueLabel: UILabel = {
        let label = UILabel(style: Stylesheet.Label.text)
        label.textAlignment = .center
        return label
    }()

    // MARK: - Initialize
    
    @available(*, unavailable)
    required init?(coder: NSCoder) {
        fatalError("init(coder:) has not been implemented")
    }
    
    init() {
        super.init(frame: .zero)
        addSubviews()
        setupConstraints()
    }
    
    private func addSubviews() {
        [titleLabel, valueLabel].forEach { view in
            view.translatesAutoresizingMaskIntoConstraints = false
            addSubview(view)
        }
    }

    private func setupConstraints() {
        titleLabel.snp.makeConstraints { (make) in
            make.top.equalToSuperview().offset(Constraints.small)
            make.left.equalToSuperview()
        }
        
        valueLabel.snp.makeConstraints { (make) in
            make.top.equalTo(titleLabel.snp.bottom).offset(Constraints.small/2)
            make.left.equalToSuperview()
            make.bottom.equalToSuperview().offset(-Constraints.small)
        }
    }
}
