//
//  SleepIntervalItem.swift
//  CTFit
//
//  Created by Mac on 2020/6/10.
//  Copyright © 2020 shirajo. All rights reserved.
//


import Foundation
import TableViewManager

final class SleepIntervalItem: BaseTableViewItem {
    
    private(set) var intervalStyle: StatisticalIntervalStyle = .week

    var sleepItem: SleepIntervalModel = SleepIntervalModel()
    
    private override init() {
        super.init()
        selectionStyle = .none
        accessoryType = .none
        isAutoHeight = true
    }
    
    convenience init(with intervalStyle: StatisticalIntervalStyle = .week) {
        self.init()
        self.intervalStyle = intervalStyle
    }
}

class SleepIntervalModel {
    
    var intervalStyle: StatisticalIntervalStyle = .week
    
    var dateInterval: DateInterval = CalendarUtils.calendar.weekInterval(containing: Date())

    /// 总数据
    /// 睡眠时间（分钟）
    var totalMinutes: Int = 0;
    /// 深睡时间（分钟）
    var deepMinutes: Int = 0;
    /// 浅睡时间（分钟）
    var lightMinutes: Int = 0;
    /// 总清醒时间（分钟）
    var wakeMinutes: Int = 0;
    
    /// .day: 960 分钟记录
    /// .week: 周日~周六 7 天记录
    /// .month:  daysOfMonth 天记录
    /// .year 12个月的记录
    var records: [Double?]?
    
    /// 可显示的个数
    var count: Int = 0
    /// 已过去的个数
    var passCount: Int = 0
    /// 是否有效计数：用于计算平均值
    /// 有返回值的元素视为有效：validCount  <= passCount,
    var validCount: Int = 0


    /// 平均值
    var averageTotalMinutes: Int { validCount == 0 ? 0 : totalMinutes / validCount }
    var averageDeepMinutes: Int { validCount == 0 ? 0 : deepMinutes / validCount }
    var averageLightMinutes: Int { validCount == 0 ? 0 : lightMinutes / validCount }
    var averageWakeMinutes: Int { validCount == 0 ? 0 : wakeMinutes / validCount }
    
    /// TimePerid
    var timePerid: String { intervalStyle.timePerid(dateInterval) }
    
    /// sleepArray
    var sleepArray: [BDSleep]

    /// initialize
    init(style intervalStyle: StatisticalIntervalStyle = .week,
         dateInterval: DateInterval = CalendarUtils.calendar.weekInterval(containing: Date()),
         sleepArray: [BDSleep] = []) {
        self.intervalStyle = intervalStyle
        self.dateInterval = dateInterval
        self.sleepArray = sleepArray
        self.validCount = self.sleepArray.count
        if intervalStyle == .year {
            self.count = dateInterval.monthsInterval
            self.passCount = dateInterval.monthPassed
            self.validCount = self.passCount
        } else {
            self.count = dateInterval.daysInterval
            self.passCount = dateInterval.daysPassed
        }
                
        records = [Double?]()
        if intervalStyle == .year {
            parseYear()
        } else {
            parseMonth()
        }
    }
    
    lazy var dateFormatter :DateFormatter = {
        let formatter = DateFormatter();
        formatter.dateFormat = "yyyy-MM-dd";
        return formatter;
    }()
    
    
    func dateStr(_ date: Date) -> String {
        dateFormatter.string(from: date)
    }

    func parseMonth()  {
        for idx in 0..<passCount {
            let date = dateInterval.dateForStart(day: Int(idx))
            let dateStr = self.dateStr(date)
            let sleep: BDSleep = sleepArray.first { (e) -> Bool in (e.dateStr == dateStr) } ?? BDSleep()
            self.deepMinutes  +=  Int(sleep.deeplMinutes)
            self.lightMinutes +=  Int(sleep.lightMinutes)
            self.wakeMinutes  +=  Int(sleep.wakeMinutes)
            self.totalMinutes +=  Int(sleep.totalMinutes)
            self.records!.append(Double(sleep.totalMinutes))
        }
    }
    
    func parseYear()  {
        for idx in 0..<passCount {
            let date = dateInterval.dateForStart(month: Int(idx))
            let monthDateInterval = CalendarUtils.calendar.monthInterval(containing: date)
            let monthTotalArray = sleepArray.filter { (e) -> Bool in
                e.dateStr >= dateStr(monthDateInterval.start) && e.dateStr <= dateStr(monthDateInterval.end)
            }
            var monthTotoalSleep: Double = 0
            monthTotalArray.forEach { (sleep) in
                monthTotoalSleep += Double(sleep.totalMinutes)
                self.deepMinutes  +=  Int(sleep.deeplMinutes)
                self.lightMinutes +=  Int(sleep.lightMinutes)
                self.wakeMinutes  +=  Int(sleep.wakeMinutes)
                self.totalMinutes +=  Int(sleep.totalMinutes)
            }
            self.records!.append(monthTotoalSleep)
        }
    }
}

