//
//  StatisticsExerciseCell.swift
//  CTFit
//
//  Created by Mac on 2020/6/9.
//  Copyright © 2020 shirajo. All rights reserved.
//

import UIKit
import SnapKit
import TableViewManager

final class StatisticsExerciseCell: BaseTableViewCell, ZJCellProtocol {

    // MARK: - ZJCellProtocol
    
    var item: StatisticsExerciseItem!
    
    typealias ZJCelltemClass = StatisticsExerciseItem
    
    override func cellWillAppear() {
        super.cellWillAppear()
        guard item != nil else { return }
        update(item)
    }
    
    // MARK: - Initialize

    override init(style: UITableViewCell.CellStyle, reuseIdentifier: String?) {
        super.init(style: .default, reuseIdentifier: reuseIdentifier)
        backgroundColor = .clear
        addSubviews()
        setConstraints()
    }
    
    //MARK - View customer
    
    private lazy var describeLabel: UILabel = {
        let label = UILabel(style: Stylesheet.Label.text)
        label.textAlignment = .left
        label.font = .customFont(ofSize: 16)
        label.heightAnchor.constraint(equalToConstant: 40).isActive = true;
        label.text = SRString.Exercise.exercise_statistic.locastr + "："
        return label
    } ()
    
    /// 详情Stack
    private lazy var detailStack: StatisticsExerciseStack = {
        let stack = StatisticsExerciseStack()
        return stack
    } ()

    
    private func addSubviews() {
        [describeLabel, detailStack].forEach { view in
            view.translatesAutoresizingMaskIntoConstraints = false
            addSubview(view)
        }
    }
    
    private func setConstraints() {
        describeLabel.snp.makeConstraints { (make) in
            make.top.equalToSuperview()
            make.left.equalToSuperview().offset(Constraints.normal)
            make.right.equalToSuperview().offset(-Constraints.normal)
        }
        
        detailStack.snp.makeConstraints { (make) in
            make.left.equalToSuperview().offset(Constraints.normal)
            make.right.equalToSuperview().offset(-Constraints.normal)
            make.top.equalTo(describeLabel.snp.bottom).offset(Constraints.small)
            make.bottom.equalToSuperview().offset(-Constraints.small)
        }
        
        separatorView()
    }
    
    var separatorColor = Colors.Chart.separator

    fileprivate func separatorView()  {
        let separator1 = UIView();
        separator1.backgroundColor = separatorColor;
        describeLabel.addSubview(separator1);
        separator1.snp.makeConstraints { (make) in
            make.left.right.equalToSuperview()
            make.bottom.equalToSuperview().offset(-0.5)
            make.height.equalTo(0.5)
        }
        
        let separator2 = UIView();
        separator2.backgroundColor = separatorColor;
        addSubview(separator2);
        separator2.snp.makeConstraints { (make) in
            make.left.right.equalToSuperview()
            make.bottom.equalToSuperview().offset(-0.5)
            make.height.equalTo(0.5)
        }
    }
}

// MARK: - update

extension StatisticsExerciseCell {
    
    private func update(_ item: StatisticsExerciseItem) {
        detailStack.model = item.statisticExerciseModel
    }
    
}


// MARK: - ExerciseStack

class StatisticsExerciseStack: UIStackView {
    
    var model: StatisticExerciseModel! {
        didSet {
            update()
        }
    }
    
    var items: [StatisticsExerciseAttr] = [StatisticsExerciseAttr]()
    
    // MARK: - initialize

    required init(coder: NSCoder) {
        super.init(coder: coder);
        initialize()

    }

    private override init(frame: CGRect) {
        super.init(frame: frame);
        initialize()
    }
    

    private func initialize() {
        tag = 0
        axis = .vertical
        distribution = .fillEqually
        StatisticExerciseModel.items.forEach { (type) in
            let attr = StatisticsExerciseAttr()
            items.append(attr)
            addArrangedSubview(attr)
        }
    }
    
    private func update() {
        let exerciseArray = StatisticExerciseModel.items.sorted { (type1, type2) -> Bool in
            model.records[type1] ?? 0 > model.records[type2] ?? 0
        }
        for idx in 0..<exerciseArray.count {
            let type = exerciseArray[idx]
            let attr = items[idx]
            attr.type = type
            attr.count = model.records[type] ?? 0
            let progress =  model.count > 0 ? Float(model.records[type] ?? 0)/Float(model.count) : 0.0
            attr.progressView.setProgress(progress, animated: (tag==1))
            attr.progressView.progressTintColor = Colors.Chart.exerciseGroup[(idx%Colors.Chart.exerciseGroup.count)]
        }
        tag = 1
    }
}

class StatisticsExerciseAttr: UIControl {
    
    var type: BDSportType = .walk {
        didSet {
            nameLabel.text = type.text
        }
    }
    
    var count: Int = 0 {
        didSet {
            numLabel.text = "：x\(count)"
        }
    }
    
    lazy var progressView: UIProgressView = {
        let view = UIProgressView()
        view.trackTintColor = Colors.Chart.starNormal
        view.progressTintColor = Colors.blue
        view.progress = 0.0;
        view.borderCorner = 4;
        //通过变形改变进度条高度（ 横向宽度不变，纵向高度变成默认的2倍）
        view.transform = CGAffineTransform(scaleX: 1.0, y: 2.0);
        return view
    }()

    private lazy var nameLabel: UILabel = {
        let label = UILabel(style: Stylesheet.Label.text)
        label.textAlignment = .right
        label.font = .customFont(ofSize: 15)
        label.numberOfLines = 1
        label.widthAnchor.constraint(equalToConstant: StatisticExerciseModel.maxLen).isActive = true;
        return label
    }()
    
    private lazy var numLabel: UILabel = {
        let label = UILabel(style: Stylesheet.Label.text)
        label.textAlignment = .left
        label.font = .customFont(ofSize: 14)
        label.numberOfLines = 1
        label.heightAnchor.constraint(equalToConstant: 30).isActive = true;
        label.widthAnchor.constraint(equalToConstant: 48).isActive = true;
        return label
    }()
    

    // MARK: - Initialize
    
    @available(*, unavailable)
    required init?(coder: NSCoder) {
        fatalError("init(coder:) has not been implemented")
    }
    
    init(type: BDSportType = .walk, count: Int = 0, progress: Float = 0 ) {
        super.init(frame: .zero)
        addSubviews()
        setupConstraints()
    }
    
    private func addSubviews() {
        [nameLabel, numLabel, progressView].forEach { view in
            view.translatesAutoresizingMaskIntoConstraints = false
            addSubview(view)
        }
    }

    private func setupConstraints() {
        nameLabel.snp.makeConstraints { (make) in
            make.left.centerY.equalToSuperview()
        }
        
        numLabel.snp.makeConstraints { (make) in
            make.top.bottom.equalToSuperview()
            make.left.equalTo(nameLabel.snp.right)
        }
        
        progressView.snp.makeConstraints { (make) in
            make.centerY.right.equalToSuperview()
            make.left.equalTo(numLabel.snp.right).offset(4)
        }
    }
    
    
    fileprivate var separatorColor = Colors.Chart.separator

    fileprivate func separatorView()  {
        let separator1 = UIView();
        separator1.backgroundColor = separatorColor;
        addSubview(separator1);
        separator1.snp.makeConstraints { (make) in
            make.left.right.equalToSuperview()
            make.bottom.equalToSuperview().offset(-0.5)
            make.height.equalTo(0.5)
        }
    }
}
