//
//  StepIntervalCell.swift
//  CTFit
//
//  Created by Mac on 2020/6/4.
//  Copyright © 2020 shirajo. All rights reserved.
//

import UIKit
import SnapKit
import TableViewManager
import Charts

final class StepIntervalCell: BaseTableViewCell, ZJCellProtocol {

    // MARK: - ZJCellProtocol
    var item: StepIntervalItem! {
        didSet {
            detailView.intervalStyle = item.intervalStyle
            _ = updateBarChart
        }
    }
    typealias ZJCelltemClass = StepIntervalItem
    override func cellWillAppear() {
        super.cellWillAppear()
        guard item != nil else { return }
        update(item.model)
    }
    

    // MARK: - Initiaize

    override init(style: UITableViewCell.CellStyle, reuseIdentifier: String?) {
        super.init(style: .default, reuseIdentifier: reuseIdentifier)
        backgroundColor = .clear
        addSubviews()
        setConstraints()
    }
    
    // MARK: - View customer
    
    private lazy var updateBarChart: Bool = {
        switch item.intervalStyle {
        case .week: ChartsUtils.Setp.week(barChart)
        case .month: ChartsUtils.Setp.month(barChart)
        case .year: ChartsUtils.Setp.year(barChart)
        }
        return true
    }()
    
    private lazy var barChart: BarChartView = {
        let chart = BarChartView()
        return chart
    } ()
    
    /// 详情Stack
    private lazy var detailView: StepIntervalDetailView = {
        let stack = StepIntervalDetailView()
        return stack
    } ()
    
    /// 达标率
    private lazy var crAttr: StepIntervalCRAttr = {
        let attr = StepIntervalCRAttr()
        return attr
    } ()
    
    /// 运动状况（评星分级）
    private lazy var gradingAttr: StepIntervalGradingAttr = {
        let attr = StepIntervalGradingAttr()
        return attr
    } ()
    
    /// 获得奖杯
    private lazy var trophyAttr: StepIntervalTrophyAttr = {
        let attr = StepIntervalTrophyAttr()
        return attr
    } ()
    
    private func addSubviews() {
        [barChart, detailView, crAttr, gradingAttr, trophyAttr].forEach { view in
            view.translatesAutoresizingMaskIntoConstraints = false
            addSubview(view)
        }
    }
    
    private func setConstraints() {
        barChart.snp.makeConstraints { (make) in
            make.top.equalToSuperview().offset(Constraints.small)
            make.left.right.equalToSuperview()
            make.height.equalTo(200)
        }
        
        detailView.snp.makeConstraints { (make) in
            make.left.right.equalToSuperview()
            make.top.equalTo(barChart.snp.bottom).offset(Constraints.normal)
        }
        
        crAttr.snp.makeConstraints { (make) in
            make.left.right.equalToSuperview()
            make.top.equalTo(detailView.snp.bottom)
        }
        
        gradingAttr.snp.makeConstraints { (make) in
            make.left.right.equalToSuperview()
            make.top.equalTo(crAttr.snp.bottom)
        }
        
        trophyAttr.snp.makeConstraints { (make) in
            make.left.right.bottom.equalToSuperview()
            make.top.equalTo(gradingAttr.snp.bottom)
        }
        
        separatorView()
    }
    
    var separatorColor = Colors.Chart.separator

    fileprivate func separatorView()  {
        let separator1 = UIView();
        separator1.backgroundColor = separatorColor;
        addSubview(separator1);
        separator1.snp.makeConstraints { (make) in
            make.left.right.top.equalToSuperview()
            make.height.equalTo(0.5)
        }
        
        let separator2 = UIView();
        separator2.backgroundColor = separatorColor;
        addSubview(separator2);
        separator2.snp.makeConstraints { (make) in
            make.left.right.equalToSuperview()
            make.bottom.equalToSuperview().offset(-0.5)
            make.height.equalTo(0.5)
        }
        
        let separator3 = UIView();
        separator3.backgroundColor = separatorColor;
        detailView.addSubview(separator3);
        separator3.snp.makeConstraints { (make) in
            make.left.right.top.equalToSuperview()
            make.height.equalTo(0.5)
        }
    }
}

// MARK: - Update

extension StepIntervalCell {
    
    private func update(_ model: StepIntervalModel) {
        updateStepDeatil(model)
        crAttr.caloriesStandardReach = model.standardReach
        gradingAttr.caloriesStandardReach = model.standardReach
        trophyAttr.trophyNum = model.trophyNum
        if model.intervalStyle == .month {
            ChartsUtils.upateMonthXAxis(barChart, date: model.dateInterval.start)
        }
        updateChartData(model.records)
    }
    
    private func updateStepDeatil(_ model: StepIntervalModel) {
        detailView.update(model)
    }
    
    private var isImperial: Bool { BandSettings.isImperial }

    /// 设置图表数据: Y轴表示每小时内运动步数
    private func updateChartData(_ values: [Double?]?) {
        barChart.highlightValue(nil)
        var entries: [BarChartDataEntry] = [BarChartDataEntry]();
        
        if let hasValues = values {
            for (index, value) in hasValues.enumerated() {
                if value == 0 { continue }
                var distance = BandCalculator.km(device: value ?? 0)
                if isImperial { distance = BandCalculator.mile(km: distance) }
                entries.append(BarChartDataEntry(x: Double(index), y: distance));
            }
        } else {
            barChart.data = nil;
            barChart.notifyDataSetChanged();
            return;
        }
        
        if let hasData = barChart.barData, hasData.dataSetCount > 0,
            let set = hasData.dataSets.first as? BarChartDataSet {
            set.replaceEntries(entries)
            hasData.notifyDataChanged();
            barChart.notifyDataSetChanged();
        } else {
            let set = BarChartDataSet(entries: entries, label: "StepsInterval\(item.intervalStyle)");
            set.drawValuesEnabled = false;
            set.setColor(Colors.blue)
            var dataSets = [BarChartDataSet]();
            dataSets.append(set);
            
            let chartData = BarChartData(dataSets: dataSets);
            chartData.barWidth = 0.7
            barChart.data = chartData;
        }
        barChart.animate(yAxisDuration: 0.35);
    }
}


// MARK: - 计步属性视图
final class StepIntervalDetailView: UIView {
    
    func update(_ model: StepIntervalModel) {
        distanceAttr.totalValue = Int(model.distance)
        stepAttr.totalValue = Int(model.steps)
        caloriesAttr.totalValue = Int(model.calories)
        
        distanceAttr.intervalValue = Int(model.averageDistance)
        stepAttr.intervalValue = Int(model.averageSteps)
        caloriesAttr.intervalValue = Int(model.averageCalories)
    }
    
    var intervalStyle: StatisticalIntervalStyle = .week {
        didSet {
            guard intervalStyle != oldValue else { return }
            distanceAttr.intervalStyle = intervalStyle
            stepAttr.intervalStyle = intervalStyle
            caloriesAttr.intervalStyle = intervalStyle
        }
    }
    
    /// Initialize
    
    @available(*, unavailable)
    required init?(coder aDecoder: NSCoder) {
        fatalError("init(coder:) has not been implemented")
    }
    
    init() {
        super.init(frame: .zero)
        addSubviews()
        setupConstraints()
    }

    /// View customer
    
    private func addSubviews() {
        [distanceAttr, stepAttr, caloriesAttr].forEach { view in
            view.translatesAutoresizingMaskIntoConstraints = false
            addSubview(view)
        }
    }

    private func setupConstraints() {
        distanceAttr.snp.makeConstraints { (make) in
            make.left.top.bottom.equalToSuperview()
            make.width.equalToSuperview().dividedBy(3)
        }
        
        stepAttr.snp.makeConstraints { (make) in
            make.centerX.top.bottom.equalToSuperview()
            make.width.equalToSuperview().dividedBy(3)
        }
        
        caloriesAttr.snp.makeConstraints { (make) in
            make.right.top.bottom.equalToSuperview()
            make.width.equalToSuperview().dividedBy(3)
        }
    }
    
    private lazy var distanceAttr: StepIntervalAttr = {
        let item = StepIntervalAttr(type: .distance)
        return item
    }()

    private lazy var stepAttr: StepIntervalAttr = {
        let item = StepIntervalAttr(type: .step)
        return item
    }()

    private lazy var caloriesAttr: StepIntervalAttr = {
        let item = StepIntervalAttr(type: .calories)
        return item
    }()
}


// MARK: - 计步属性项
class StepIntervalAttr: UIControl {
    
    var intervalStyle: StatisticalIntervalStyle = .week
    
    var totalValue: Int = 0 {
        didSet {
            var valueText = "\(totalValue)"
            if type == .distance {
                valueText = String(format: "%0.1f", BandCalculator.km(device: BDCalculatorType(totalValue)))
            }
            if type == .calories {
                valueText = "\(Int(totalValue/1000))"
            }
            totalValueLabel.text = valueText + " " + unit
        }
    }
    
    var intervalValue: Int = 0 {
        didSet {
            var valueText = "\(intervalValue)"
            if type == .distance {
                var disntance = BandCalculator.km(device: BDCalculatorType(intervalValue))
                if isImperial { disntance = BandCalculator.mile(device: BDCalculatorType(intervalValue)) }
                valueText = String(format: "%0.1f", disntance)
            }
            if type == .calories {
                valueText = "\(Int(intervalValue/1000))"
            }
            intervalValueLabel.text = valueText + " " + perUnit
        }
    }
    
    // MARK: - Initialize
    
    @available(*, unavailable)
    required init?(coder: NSCoder) {
        fatalError("init(coder:) has not been implemented")
    }
    
    init(type: SportDataType = .distance) {
        super.init(frame: .zero)
        self.type = type
        addSubviews()
        setupConstraints()
    }
    
    // MARK: - View customer
    
    private func addSubviews() {
        [imageView, totalValueLabel, intervalValueLabel].forEach { view in
            view.translatesAutoresizingMaskIntoConstraints = false
            addSubview(view)
        }
    }

    private func setupConstraints() {
        imageView.snp.makeConstraints { (make) in
            make.centerX.equalToSuperview()
            make.top.equalToSuperview().offset(Constraints.small)
        }
        
        totalValueLabel.snp.makeConstraints { (make) in
            make.centerX.equalToSuperview()
            make.top.equalTo(imageView.snp.bottom).offset(Constraints.small)
        }
        
        intervalValueLabel.snp.makeConstraints { (make) in
            make.centerX.bottom.equalToSuperview()
            make.top.equalTo(totalValueLabel.snp.bottom)
            make.bottom.equalToSuperview().offset(-Constraints.small)
        }
    }
    
    private var type: SportDataType = .distance
    
    private var unit: String { type.unit }

    private var perUnit: String { type.perUnit(style: intervalStyle.dateBarStyle) }
    
    private var isImperial: Bool { BandSettings.isImperial }
    
    private lazy var imageView: UIImageView = {
        let view = UIImageView()
        view.contentMode = .scaleAspectFit
        view.image = self.type.image
        return view
    }()
    
    private lazy var totalValueLabel: UILabel = {
        let label = UILabel(style: Stylesheet.Label.text)
        label.font = .customFont(ofSize: 13)
        label.textAlignment = .center
        return label
    }()
    
    private lazy var intervalValueLabel: UILabel = {
        let label = UILabel(style: Stylesheet.Label.text)
        label.font = .customFont(ofSize: 13)
        label.textAlignment = .center
        return label
    }()
}

// MARK: - 达标完成率
class StepIntervalCRAttr: UIControl {

    var caloriesStandardReach: Double = 0 {
        didSet { detailLabel.text = "\(Int(caloriesStandardReach * 100)) %"}
    }
    

    // MARK: - Initialize
    @available(*, unavailable)
    required init?(coder: NSCoder) {
        fatalError("init(coder:) has not been implemented")
    }
    
    init(type: SportDataType = .distance) {
        super.init(frame: .zero)
        addSubviews()
    }
    
    // MARK: - View customer
    
    private func addSubviews() {
        heightAnchor.constraint(equalToConstant: 40).isActive = true;

        [titleLabel, detailLabel].forEach { view in
            view.translatesAutoresizingMaskIntoConstraints = false
            addSubview(view)
        }

        titleLabel.snp.makeConstraints { (make) in
            make.centerY.equalToSuperview()
            make.left.equalToSuperview().offset(Constraints.normal)
        }
        
        detailLabel.snp.makeConstraints { (make) in
            make.centerY.equalToSuperview()
            make.left.equalTo(titleLabel.snp.right).offset(Constraints.small)
        }
        
        separatorView()
    }
    
    var separatorColor = Colors.Chart.separator

    fileprivate func separatorView()  {
        let separator1 = UIView();
        separator1.backgroundColor = separatorColor;
        addSubview(separator1);
        separator1.snp.makeConstraints { (make) in
            make.left.right.top.equalToSuperview()
            make.height.equalTo(0.5)
        }
    }
    
    private lazy var titleLabel: UILabel = {
        let label = UILabel(style: Stylesheet.Label.text)
        label.font = .customFont(ofSize: 16)
        label.textAlignment = .left
        label.text = SRString.Step.standard_reach_rate.locastr + "："
        return label
    }()
    
    private lazy var detailLabel: UILabel = {
        let label = UILabel(style: Stylesheet.Label.text)
        label.font = .customFont(ofSize: 16)
        label.textAlignment = .left
        return label
    }()
}

// MARK: - 运动状况（评分分级）
class StepIntervalGradingAttr: UIControl {
    var caloriesStandardReach: Double = 0 {
        didSet { starRateView.score(CGFloat(caloriesStandardReach * 5))}
    }
    
    // MARK: - Initialize
    @available(*, unavailable)
    required init?(coder: NSCoder) {
        fatalError("init(coder:) has not been implemented")
    }
    
    init(type: SportDataType = .distance) {
        super.init(frame: .zero)
        addSubviews()
    }
    
    // MARK: - View customer
    
    private func addSubviews() {
        heightAnchor.constraint(equalToConstant: 40).isActive = true;

        [titleLabel, starRateView].forEach { view in
            view.translatesAutoresizingMaskIntoConstraints = false
            addSubview(view)
        }

        titleLabel.snp.makeConstraints { (make) in
            make.centerY.equalToSuperview()
            make.left.equalToSuperview().offset(Constraints.normal)
        }
        
        starRateView.snp.makeConstraints { (make) in
            make.centerY.equalToSuperview()
            make.width.equalTo(96)
            make.height.equalTo(16)
            make.left.equalTo(titleLabel.snp.right).offset(Constraints.small)
        }
        
        separatorView()
    }
    
    var separatorColor = Colors.Chart.separator

    fileprivate func separatorView()  {
        let separator1 = UIView();
        separator1.backgroundColor = separatorColor;
        addSubview(separator1);
        separator1.snp.makeConstraints { (make) in
            make.left.right.top.equalToSuperview()
            make.height.equalTo(0.5)
        }
    }
    
    private lazy var titleLabel: UILabel = {
        let label = UILabel(style: Stylesheet.Label.text)
        label.font = .customFont(ofSize: 16)
        label.textAlignment = .left
        label.text = SRString.Step.movement.locastr + "："
        return label
    }()
    
    private lazy var starRateView: StarRateView = {
        let view = StarRateView(frame: CGRect(x: 0, y: 0, width: 96, height: 16), totalStarCount: 5, currentStarCount: 0, starSpace: 4)
        return view
    }()
}

// MARK: - 奖杯
class StepIntervalTrophyAttr: UIControl {
    
    var trophyNum: Int = 0 {
        didSet { countLabel.text = "x \(trophyNum)"}
    }
    
    // MARK: - Initialize
    @available(*, unavailable)
    required init?(coder: NSCoder) {
        fatalError("init(coder:) has not been implemented")
    }
    
    init(type: SportDataType = .distance) {
        super.init(frame: .zero)
        addSubviews()
    }
    
    // MARK: - View customer
    
    private func addSubviews() {
        heightAnchor.constraint(equalToConstant: 40).isActive = true;

        [titleLabel, iconView, countLabel].forEach { view in
            view.translatesAutoresizingMaskIntoConstraints = false
            addSubview(view)
        }

        titleLabel.snp.makeConstraints { (make) in
            make.centerY.equalToSuperview()
            make.left.equalToSuperview().offset(Constraints.normal)
        }
        
        iconView.snp.makeConstraints { (make) in
            make.centerY.equalToSuperview()
            make.left.equalTo(titleLabel.snp.right).offset(Constraints.small)
        }
        
        countLabel.snp.makeConstraints { (make) in
            make.centerY.equalToSuperview()
            make.left.equalTo(iconView.snp.right).offset(Constraints.small)
        }
        
        separatorView()
    }
    
    var separatorColor = Colors.Chart.separator

    fileprivate func separatorView()  {
        let separator1 = UIView();
        separator1.backgroundColor = separatorColor;
        addSubview(separator1);
        separator1.snp.makeConstraints { (make) in
            make.left.right.top.equalToSuperview()
            make.height.equalTo(0.5)
        }
    }
    
    private lazy var titleLabel: UILabel = {
        let label = UILabel(style: Stylesheet.Label.text)
        label.font = .customFont(ofSize: 16)
        label.textAlignment = .left
        label.text = SRString.Step.trophy.locastr + "："
        return label
    }()
    
    private lazy var iconView: UIImageView = {
        let view = UIImageView(image: ImageRepo.Step.trophy.reSizeImage(CGSize(width: 18, height: 18)))
        return view
    }()
    
    private lazy var countLabel: UILabel = {
        let label = UILabel(style: Stylesheet.Label.text)
        label.font = .customFont(ofSize: 16)
        label.textAlignment = .left
        label.text = "x 0"
        return label
    }()
}
