//
//  StepIntervalItem.swift
//  CTFit
//
//  Created by Mac on 2020/6/9.
//  Copyright © 2020 shirajo. All rights reserved.
//

import Foundation
import TableViewManager

final class StepIntervalItem: BaseTableViewItem {
    
    private(set) var intervalStyle: StatisticalIntervalStyle = .week

    var model: StepIntervalModel = StepIntervalModel()
    
    private override init() {
        super.init()
        selectionStyle = .none
        accessoryType = .none
        isAutoHeight = true
    }
    
    convenience init(with intervalStyle: StatisticalIntervalStyle = .week) {
        self.init()
        self.intervalStyle = intervalStyle
    }
}


class StepIntervalModel {
    
    var intervalStyle: StatisticalIntervalStyle = .week
    
    var dateInterval: DateInterval = CalendarUtils.calendar.weekInterval(containing: Date())

    /// 总值
    var steps: Double = 0
    var distance: Double = 0
    var calories: Double = 0
    
    /// 距离记录数据
    /// .day: 24 小时记录
    /// .week: 周日~周六 7 天记录
    /// .month:  daysOfMonth 天记录
    /// .year 12个月的记录
    var records: [Double?]?

    /// 可显示的个数
    var count: Int = 0
    /// 已过去的个数
    var passCount: Int = 0
    /// 是否有效计数：用于计算平均值
    /// 有返回值的元素视为有效：validCount  <= passCount, 
    var validCount: Int = 0

    
    /// 使用卡路里值计算（骑行运动没有步数）
    /// 达标次数
    var standardReachNum: Int = 0
    /// 达标率
    var standardReach: Double { (Double(totoalArray.count) * goalCalories == 0) ? 0 : calories / (Double(totoalArray.count) * goalCalories)}
    /// 奖杯数( 每100%奖励一个 )
    var trophyNum: Int { Int(standardReach) }
    
    /// 平均值
    var averageDistance: Double { validCount == 0 ? 0 : distance / Double(validCount) }
    var averageSteps: Double { validCount == 0 ? 0 : steps / Double(validCount) }
    var averageCalories: Double { validCount == 0 ? 0 : calories / Double(validCount) }
    
    /// totoalArray
    var totoalArray: [BDTotal]

    /// initialize
    init(style intervalStyle: StatisticalIntervalStyle = .week,
         dateInterval: DateInterval = CalendarUtils.calendar.weekInterval(containing: Date()),
         totoalArray: [BDTotal] = []) {
        self.intervalStyle = intervalStyle
        self.dateInterval = dateInterval
        self.totoalArray = totoalArray
        self.validCount = self.totoalArray.count
        if intervalStyle == .year {
            self.count = dateInterval.monthsInterval
            self.passCount = dateInterval.monthPassed
            self.validCount = dateInterval.monthPassed
        } else {
            self.count = dateInterval.daysInterval
            self.passCount = dateInterval.daysPassed
        }
        
        records = [Double?]()
        if intervalStyle == .year {
            parseYear()
        } else {
            parseMonth()
        }
    }
    
    var goalSteps: Int { Int(BandSettings.goal.step) }

    var goalCalories: Double { Double(BandSettings.goal.calories) * 1000 }

    lazy var dateFormatter :DateFormatter = {
        let formatter = DateFormatter();
        formatter.dateFormat = "yyyy-MM-dd";
        return formatter;
    }()
    
    
    func dateStr(_ date: Date) -> String {
        dateFormatter.string(from: date)
    }

    func parseMonth()  {
        for idx in 0..<passCount {
            let date = dateInterval.dateForStart(day: Int(idx))
            let dateStr = self.dateStr(date)
            let total: BDTotal = totoalArray.first { (e) -> Bool in (e.dateStr == dateStr) } ?? BDTotal()
            self.steps +=  Double(total.steps)
            self.distance += Double(total.distance)
            self.calories += Double(total.calories)
            if total.calories >= goalCalories { self.standardReachNum += 1 }
            self.records!.append(Double(total.distance))
        }
    }
    
    func parseYear()  {
        for idx in 0..<passCount {
            let date = dateInterval.dateForStart(month: Int(idx))
            let monthDateInterval = CalendarUtils.calendar.monthInterval(containing: date)
            let monthTotalArray = totoalArray.filter { (e) -> Bool in
                e.dateStr >= dateStr(monthDateInterval.start) && e.dateStr <= dateStr(monthDateInterval.end)
            }
            var monthDistance: Double = 0
            monthTotalArray.forEach { (total) in
                monthDistance += Double(total.distance)
                self.steps +=  Double(total.steps)
                self.distance += Double(total.distance)
                self.calories += Double(total.calories)
                if total.calories >= goalCalories { self.standardReachNum += 1 }
            }            
            self.records!.append(monthDistance)
        }
    }
}
