//
//  RecordPageView.swift
//  CTFit
//
//  Created by Mac on 2020/6/8.
//  Copyright © 2020 shirajo. All rights reserved.
//

import UIKit

class RecordPageView: UIView {
    
    /// 页面
    var viewControllers: [UIViewController] = [UIViewController]() {
        didSet {
            switchToIndex(0, animated: false)
            segmentControl.selectedSegmentIndex = 0;
        }
    }
    
    /// 选中位置
    var selectIndex: Int? = nil
    
    
    // MARK: - Initialize
    @available(*, unavailable)
    required init?(coder aDecoder: NSCoder) {
        fatalError("init(coder:) has not been implemented")
    }
    
    private init() {
        super.init(frame: .zero)
        addSubviews()
        setupConstraints()
    }
    
    convenience init(superController: UIViewController) {
        self.init()
        superController.addChild(pageController)
    }

    // MARK: - Private
    
    private lazy var segmentControl: UISegmentedControl = {
        let items = [SRString.Base.day.locastr,
                     SRString.Base.week.locastr,
                     SRString.Base.month.locastr,
                     SRString.Base.year.locastr]
        let segment = UISegmentedControl(items: items);
        segment.backgroundColor = Colors.Chart.segmentGround
        segment.tintColor = Colors.Base.background
        segment.setTitleTextAttributes([.foregroundColor: Colors.Base.foreground, .font: UIFont.customFont(ofSize: 14)], for: .normal);
        segment.setTitleTextAttributes([.foregroundColor: Colors.Base.foreground, .font: UIFont.customFont(ofSize: 14)], for: .selected);
        if #available(iOS 13.0, *) {
            segment.selectedSegmentTintColor = Colors.Base.background
        } else {
            segment.clipsToBounds = true
            segment.layer.cornerRadius = 6
        }
        segment.addTarget(self, action: #selector(segmentValueChanged(sender:)), for: .valueChanged);
        return segment
    }()
    
    private lazy var pageController: UIPageViewController = {
        let controller = UIPageViewController(transitionStyle: .scroll, navigationOrientation: .horizontal, options: [.interPageSpacing: 0])
        return controller
    }()
    

    // MARK:  method
    private func addSubviews() {
        self.pageController.dataSource = self
        self.pageController.delegate = self
        [pageController.view, segmentControl].forEach { view in
            view!.translatesAutoresizingMaskIntoConstraints = false
            addSubview(view!)
        }
    }
    
    private func setupConstraints() {
        segmentControl.snp.makeConstraints { (make) in
            make.top.equalToSuperview().offset(Constraints.small)
            make.left.equalToSuperview().offset(Constraints.small)
            make.right.equalToSuperview().offset(-Constraints.small)
            make.height.equalTo(30)
        }
        
        pageController.view.snp.makeConstraints { (make) in
            // make.top.equalTo(segmentControl.snp.bottom)
            make.left.right.bottom.top.equalToSuperview()
        }
    }
    
    private func switchToIndex(_ index: Int, animated: Bool) {
        guard index >= 0, index < viewControllers.count else { return }
        let direction: UIPageViewController.NavigationDirection =  (index >= selectIndex ?? 0) ? .forward : .reverse
        if !animated { selectIndex = index }
        if animated { segmentControl.isUserInteractionEnabled = false }
        pageController.setViewControllers([viewControllers[index]], direction: direction, animated: animated) { [weak self](_) in
            guard let strong = self else { return }
            strong.selectIndex = index
            if animated { strong.segmentControl.isUserInteractionEnabled = true }
        }
    }

    @objc fileprivate func segmentValueChanged(sender: Any) {
        let index = segmentControl.selectedSegmentIndex;
        switchToIndex(index, animated: false)
    }

}

// MARK: - UIPageViewControllerDataSource

extension RecordPageView: UIPageViewControllerDataSource {
    func pageViewController(_ pageViewController: UIPageViewController, viewControllerBefore viewController: UIViewController) -> UIViewController? {
        guard let index = viewControllers.firstIndex(of: viewController), index > 0 else { return nil }
        return viewControllers[index-1]
    }
    
    func pageViewController(_ pageViewController: UIPageViewController, viewControllerAfter viewController: UIViewController) -> UIViewController? {
        guard let index = viewControllers.firstIndex(of: viewController), index < viewControllers.count-1 else { return nil }
        return viewControllers[index+1]
    }
}

// MARK: - UIPageViewControllerDelegate

extension RecordPageView: UIPageViewControllerDelegate {
    
    func pageViewController(_ pageViewController: UIPageViewController, didFinishAnimating finished: Bool, previousViewControllers: [UIViewController], transitionCompleted completed: Bool) {
        
        if completed {
            guard let viewController = pageViewController.viewControllers?.first,
                let index = viewControllers.firstIndex(of: viewController) else { return }
            // UILog.v("index: \(index), viewController:\(viewControllers[index])")
            segmentControl.selectedSegmentIndex = index;
            selectIndex = index;
        }
    }
}
