//
//  NotifyTypeView.swift
//  CTFit
//
//  Created by Mac on 2020/9/29.
//  Copyright © 2020 jpaxh. All rights reserved.
//


import UIKit
import RxSwift
import MJRefresh
import TableViewManager

class NotifyTypeView: UIView {
    
    @available(*, unavailable)
    required init?(coder aDecoder: NSCoder) {
        fatalError("init(coder:) has not been implemented")
    }
    
    private init() {
        super.init(frame: .zero)
        self.backgroundColor = groundColor
    }
    
    convenience init(with settingsInfo: inout BandSettings,
         leftTap: @escaping ButtonTapEvent, rightTap: @escaping ButtonTapEvent) {
        self.init()
        self.settingsInfo = settingsInfo
        self.leftTap = leftTap
        self.rightTap = rightTap
        titleLabel.text = SRString.Device.notify_message_type.locastr
        addSubviews()
        setupConstraints()
        borderCorner = 12
    }
    
    override func layoutSubviews() {
        super.layoutSubviews()
    }
    
    override func draw(_ rect: CGRect) {
        let path = UIBezierPath(rect: CGRect(x: 0, y: 44, width: rect.size.width, height: 0.5))
        Colors.Cell.separator.withAlphaComponent(0.9).setFill()
        path.fill(with: .normal, alpha: 1)
        path.close()
    }
    
    private func addSubviews() {
        [titleLabel, circleChecker, buttonsStack].forEach { view in
            view.translatesAutoresizingMaskIntoConstraints = false
            addSubview(view)
        }
    }

    private func setupConstraints() {
        titleLabel.snp.makeConstraints { (make) in
            make.left.right.top.equalToSuperview()
            make.height.equalTo(44)
        }
        
        let width = 250
        circleChecker.snp.makeConstraints { (make) in
            make.left.equalToSuperview().offset(Constraints.large)
            make.right.equalToSuperview().offset(-Constraints.large)
            make.top.equalTo(titleLabel.snp.bottom).offset(Constraints.medium)
            make.width.height.equalTo(width)
        }
        
        buttonsStack.snp.makeConstraints { (make) in
            make.left.equalToSuperview().offset(Constraints.medium)
            make.right.equalToSuperview().offset(-Constraints.medium)
            make.top.equalTo(circleChecker.snp.bottom).offset(Constraints.normal)
            make.bottom.equalToSuperview().offset(-Constraints.normal)
        }    
    }
    
    // MARK: - Field
    typealias Anything = () -> Void
    typealias ButtonTapEvent = (_ sender: UIButton) -> Void
    private var leftTap: ButtonTapEvent? = nil
    private var rightTap: ButtonTapEvent? = nil

    private let disposeBag = DisposeBag()
     
    private var settingsInfo: BandSettings!
    
    private let groundColor = Colors.darkMode(Colors.dark, Colors.white)
    
    private lazy var circleChecker: CircleChecker = {
        let view = CircleChecker()
        view.wihteList = settingsInfo.ancsList
        return view
    } ()
    
    private lazy var titleLabel: UILabel = {
        let label = UILabel(style: Stylesheet.Label.naviTitle)
        label.backgroundColor = Colors.darkMode(Colors.dark, Colors.blue)
        label.textColor = Colors.white
        return label
    } ()

    private lazy var buttonsStack: UIStackView = {
        let stack = UIStackView();
        stack.axis = .horizontal;
        stack.alignment = .center;
        stack.distribution = .fillEqually;
        stack.spacing = Constraints.normal
        let leftButton = createButton(SRString.Base.cancel.locastr, leftTap)
        let rightButton = createButton(SRString.Base.ok.locastr) { [weak self] (sender) in
            guard let strong = self else { return }
            strong.settingsInfo.ancsList = strong.circleChecker.wihteList
            strong.rightTap?(sender)
        }

        leftButton.backgroundColor = Colors.darkMode(Colors.darkGray, Colors.lightGray)
        leftButton.setTitleColor(Colors.darkMode(Colors.lightGray, Colors.white), for: .normal)
        rightButton.backgroundColor = Colors.darkMode(Colors.darkGray, Colors.blue)
        rightButton.setTitleColor(Colors.darkMode(Colors.white, Colors.white), for: .normal)
        stack.addArrangedSubview(leftButton)
        stack.addArrangedSubview(rightButton)
        return stack
    } ()

    private func createButton(_ text: String?, _ tap: ButtonTapEvent?) ->UIButton {
        let buttom = UIButton(type: .system)
        buttom.apply(Stylesheet.View.dialogButton)
        buttom.setTitle(text, for: .normal)
        guard let _ = text else { return buttom }
        buttom.isUserInteractionEnabled = true
        buttom.rx.tap.subscribe(onNext: { _ in
            if let hasTap = tap { hasTap(buttom) }
        }).disposed(by: disposeBag)
        return buttom
    }
}
