//
//  RemindDatasource.swift
//  CTFit
//
//  Created by Mac on 2020/9/22.
//  Copyright © 2020 jpaxh. All rights reserved.
//


import Foundation

enum SRRemindType {
    case call, message, sedentary, water
    case heart, brightScrren
    case nodisturb, target
    var text: String {
        switch self {
        case .call:
            switch BandConnect.protocolType {
            case .series_z: return SRString.Device.reminder_incoming_period.locastr
            case .series_s: return SRString.Device.reminder_incoming.locastr
            }
        case .message:
            switch BandConnect.protocolType {
            case .series_z: return SRString.Device.notify_message_period.locastr
            case .series_s: return SRString.Device.notify_message.locastr
            }
        case .sedentary: return SRString.SportSet.reminder_sedentary.locastr
        case .water: return SRString.Device.drinkwater_reminder.locastr
        case .heart: return SRString.Device.heart_monitor.locastr
        case .brightScrren: return SRString.Device.roll_brightscrren.locastr
        case .nodisturb: return SRString.Device.no_distrub.locastr
        case .target: return SRString.SportSet.reminder_target.locastr
        }
    }
}

final class RemindDatasource {
    
    func setOnFeatureValueChanged(_ closrue: @escaping SwitchItem.ValueChangedClosure) {
        self.onFeatureValueChanged = closrue
    }
    
    init(with type: SRRemindType, settingsInfo: inout BandSettings) {
        self.type = type
        self.settingsInfo = settingsInfo
    }
    
    var remind: BDRemind  {
        switch type {
        case .call: return settingsInfo.callRemind
        case .message: return settingsInfo.messageRemind
        case .sedentary: return settingsInfo.sedentaryRemind
        case .water: return settingsInfo.waterRemind
        case .heart: return settingsInfo.restingHeartRate
        case .brightScrren: return settingsInfo.rollBrightScrren
        case .nodisturb: return settingsInfo.noDistrub
        case .target: return settingsInfo.targetRemind
        }
    }
    
    // MARK: - items lazy
    lazy var featureItem: SwitchItem = {
        let item = SwitchItem(with: .feature_enable, isOn: remind.enable)
        item.setOnValueChanged { [weak self] (enable) in
            guard let strong = self else { return }
            strong.remind.enable = enable
            if let closure = strong.onFeatureValueChanged { closure(enable) }
        }
        return item
    }()
 
    lazy var timePeriodItems: [TimeItem]? = {
        guard remind.type.hasTime else { return nil}
        let startTimeItem = TimeItem(with: .start, hour: Int(remind.startHour), minute: 0, houly: settingsInfo.format.hourly)
        let endTimeItem = TimeItem(with: .end, hour: Int(remind.endHour), minute: 59, houly: settingsInfo.format.hourly)
        startTimeItem.setOnValueChanged { [weak self] (item) in
            guard let strong = self else { return }
            strong.remind.startHour = UInt8(item.hour)
        }
        endTimeItem.setOnValueChanged { [weak self] (item) in
            guard let strong = self else { return }
            strong.remind.endHour = UInt8(item.hour)
        }
        return [startTimeItem, endTimeItem]
    } ()
    
    lazy var vibrateItem: VibrateItem? = {
        guard remind.type.hasVibrate else { return nil}
        
        let item = VibrateItem(with: remind.vibrate)
        item.setValueChanged { [weak self] (vibrate)in
            guard let strong = self else { return }
            strong.remind.vibrate = vibrate
        }
        return item
    } ()

    lazy var intervalItem: SettingPickerItem? = {
        if type == .sedentary { return SettingPickerItem(with: .intervalSedentary, settingsInfo: &settingsInfo) }
        if type == .water { return SettingPickerItem(with: .intervalWater, settingsInfo: &settingsInfo) }
        return nil
    } ()
    
    
    // MARK: - private field
    private var onFeatureValueChanged: SwitchItem.ValueChangedClosure? = nil
    fileprivate var type: SRRemindType = .call
    fileprivate var settingsInfo: BandSettings!
}

