//
//  RemindViewController.swift
//  CTFit
//
//  Created by Mac on 2020/9/22.
//  Copyright © 2020 jpaxh. All rights reserved.
//

import Foundation
import TableViewManager

class RemindViewController: BaseViewController, CustomView {
    
    typealias ViewClass = BaseTableView
    
    deinit {
        onDidChanged()
    }
    
    override init() {
        super.init()
        hidesBottomBarWhenPushed = true
    }
    
    convenience init(with type: SRRemindType, settingsInfo: inout BandSettings) {
        self.init()
        navigationItem.title = type.text
        datasource = RemindDatasource(with: type, settingsInfo: &settingsInfo)
    }
    
    override func loadView() {
        super.loadView()
        view = ViewClass()
    }

    override func viewDidLoad() {
        super.viewDidLoad()
        setupTableView()
    }
    
    override func viewWillAppear(_ animated: Bool) {
        super.viewWillAppear(animated)
    }

    override func initNavigationBar() {
        super.initNavigationBar()
    }
    
    // MARK: - tableViewManger
    private var datasource: RemindDatasource!
    
    private lazy var tableViewManger: ZJTableViewManager = {
        let manager = ZJTableViewManager(tableView: customView.tableView)
        manager.register(SwitchCell.self, SwitchItem.self)
        manager.register(PickerCell.self, PickerItem.self)
        manager.register(TimeCell.self, TimeItem.self)
        manager.register(VibrateCell.self, VibrateItem.self)
        manager.register(SettingPickerCell.self, SettingPickerItem.self)
        manager.add(section: featureSection);
        if datasource.featureItem.isOn {
            if let section = vibrateSection { manager.add(section: section) }
            if let section = intervalSection { manager.add(section: section) }
            if let section = timePeroidSection { manager.add(section: section) }
        }
        return manager
    } ()
    
    private lazy var featureSection: ZJTableViewSection = {
        let section = ZJTableViewSection()
        section.headerHeight = Constraints.normal; section.footerHeight = 0.01;
        section.add(item: datasource.featureItem)
        return section
    } ()
    
    private lazy var timePeroidSection: ZJTableViewSection? = {
        guard let items = datasource.timePeriodItems, items.count>0 else { return nil}
        let section = BaseTableViewSection(customHeader: SRString.Period.allow_period.locastr)
        items.forEach { (item) in
            item.addSubPickerItem(to: section)
            section.add(item: item)
        }
        return section
    } ()
    
    private lazy var vibrateSection: ZJTableViewSection? = {
        guard let item = datasource.vibrateItem else { return nil}
        let section = ZJTableViewSection()
        section.headerHeight = Constraints.normal; section.footerHeight = 0.01;
        section.add(item: item)
        return section
    } ()
    
    private lazy var intervalSection: ZJTableViewSection? = {
        guard let item = datasource.intervalItem else { return nil}
        let section = ZJTableViewSection()
        section.headerHeight = Constraints.normal; section.footerHeight = 0.01;
        item.addSubPickerItem(to: section)
        section.add(item: item)
        return section
    } ()

    private func setupTableView() {
        tableViewManger.reload();
        datasource.setOnFeatureValueChanged { [weak self] (enable) in
            guard let strong = self else { return }
            strong.didChangedFeature(enable)
        }
    }
    
    private func didChangedFeature(_ enable: Bool) {
        onDidChanged()
        if enable {
            if let section = vibrateSection { tableViewManger.add(section: section) }
            if let section = intervalSection { tableViewManger.add(section: section) }
            if let section = timePeroidSection { tableViewManger.add(section: section) }
            var sections = [Int]()
            if let index = vibrateSection?.index { sections.append(index) }
            if let index = intervalSection?.index { sections.append(index) }
            if let index = timePeroidSection?.index { sections.append(index) }
            customView.tableView.insertSections(sections, animationStyle: .fade)

        } else {
            var sections = [Int]()
            if let index = vibrateSection?.index { sections.append(index) }
            if let index = intervalSection?.index { sections.append(index) }
            if let index = timePeroidSection?.index { sections.append(index) }
            if let section = vibrateSection { tableViewManger.remove(section: section) }
            if let section = intervalSection { tableViewManger.remove(section: section) }
            if let section = timePeroidSection { tableViewManger.remove(section: section) }
            customView.tableView.deleteSections(sections, animationStyle: .fade)
        }
    }
}
 
