//
//  SingleDatasource.swift
//  CTFit
//
//  Created by Mac on 2020/9/16.
//  Copyright © 2020 jpaxh. All rights reserved.
//


import Foundation
import RxSwift

enum SingleType {
    case gender, hand_type
    case hourly, system, tempUnitType, dateformatType, brightScreenTimeType
    var text: String {
        switch self {
        case .gender: return SRString.Personal.gender.locastr
        case .hand_type: return SRString.Personal.hand_type.locastr
        case .hourly: return SRString.Device.hourly_set.locastr
        case .system: return SRString.Device.system_unit.locastr
        case .tempUnitType: return SRString.Device.temp_unit.locastr
        case .dateformatType: return SRString.Device.dateformat.locastr
        case .brightScreenTimeType: return SRString.Device.bright_screen_time.locastr
        }
    }
}

final class SingleDatasource {
    
    var items: [SingleItem] = [SingleItem]()
        
    // MARK: - Initialize
    init(with type: SingleType, settingsInfo: inout BandSettings) {
        self.type = type
        self.settingsInfo = settingsInfo
        loadItems()
        
    }
    
    private func loadItems() {
        switch type {
        case .gender:
            BDGenderType.collection.forEach { (type) in
                let item = SingleItem(with: type, settingsInfo: &settingsInfo)
                if settingsInfo.user.gender == type { selectItem = item; selectItem?.accessoryType = .checkmark}
                items.append(item)
            }
        case .hand_type:
            BDHandType.collection.forEach { (type) in
                let item = SingleItem(with: type, settingsInfo: &settingsInfo)
                if settingsInfo.user.handType == type { selectItem = item; selectItem?.accessoryType = .checkmark}
                items.append(item)
            }
        case .hourly:
            BDHourlyType.collection.forEach { (type) in
                let item = SingleItem(with: type, settingsInfo: &settingsInfo)
                if settingsInfo.format.hourly == type { selectItem = item; selectItem?.accessoryType = .checkmark}
                items.append(item)
            }
        case .system:
            BDSystemType.collection.forEach { (type) in
                let item = SingleItem(with: type, settingsInfo: &settingsInfo)
                if settingsInfo.format.system == type { selectItem = item; selectItem?.accessoryType = .checkmark}
                items.append(item)
            }
        case .tempUnitType:
            BDTempUnitsType.collection.forEach { (type) in
                let item = SingleItem(with: type, settingsInfo: &settingsInfo)
                if settingsInfo.format.tempUnit == type { selectItem = item; selectItem?.accessoryType = .checkmark}
                items.append(item)
            }
        case .dateformatType:
            BDDateformatType.collection.forEach { (type) in
                let item = SingleItem(with: type, settingsInfo: &settingsInfo)
                if settingsInfo.format.dateFormat == type { selectItem = item; selectItem?.accessoryType = .checkmark}
                items.append(item)
            }
        case .brightScreenTimeType:
            BDBrightScreenTimeType.collection.forEach { (type) in
                let item = SingleItem(with: type, settingsInfo: &settingsInfo)
                if settingsInfo.format.brightScreenTime == type { selectItem = item; selectItem?.accessoryType = .checkmark}
                items.append(item)
            }
        }
        
        SingleItem.observable.observeOn(MainScheduler.instance).subscribe(onNext: { [weak self] item in
            guard let strong = self else { return }
            strong.selectItem = item
        }).disposed(by: disposeBag)
    }
           
    
    // MARK: - private field
    private var type: SingleType = .gender
    private var settingsInfo: BandSettings!
    private let disposeBag = DisposeBag()
    private var selectItem: SingleItem?  {
        didSet {
            guard oldValue != selectItem else { return }
            guard let newItem = selectItem else { return }
            newItem.accessoryType = .checkmark
            newItem.reload()
            guard let oldItem = oldValue  else { return }
            oldItem.accessoryType = .none
            oldItem.reload()
        }
    }
}
