//
//  SingleView.swift
//  CTFit
//
//  Created by Mac on 2020/9/22.
//  Copyright © 2020 jpaxh. All rights reserved.
//

import UIKit
import MJRefresh
import RxSwift
import TableViewManager

class SingleView: UIView {
    
    @available(*, unavailable)
    required init?(coder aDecoder: NSCoder) {
        fatalError("init(coder:) has not been implemented")
    }
    
    private init() {
        super.init(frame: .zero)
        self.backgroundColor = groundColor
    }
    
    convenience init(with type: SingleType, settingsInfo: inout BandSettings,
         leftTap: @escaping ButtonTapEvent, rightTap: @escaping ButtonTapEvent) {
        self.init()
        
        self.leftTap = leftTap
        self.rightTap = rightTap
        titleLabel.text = type.text
        datasource = SingleDatasource(with: type, settingsInfo: &settingsInfo)
        addSubviews()
        setupConstraints()
        tableViewManger.reload()
        borderCorner = 12
    }
    
    override func layoutSubviews() {
        super.layoutSubviews()
        if autoBounces {
            tableView.bounces = tableView.contentSize.height > tableView.height+2 ? true : false
        }
    }
    
    private func addSubviews() {
        [titleLabel, tableView, buttonsStack].forEach { view in
            view.translatesAutoresizingMaskIntoConstraints = false
            addSubview(view)
        }
    }

    private func setupConstraints() {
        titleLabel.snp.makeConstraints { (make) in
            make.left.right.top.equalToSuperview()
            make.height.equalTo(44)
        }
        
        let count = min(datasource.items.count, 3)
        let height = count * 44 + 1
        var width = UIDevice.scrWidth*0.85
        if UIDevice.idiom == .pad { width = UIDevice.scrWidth*0.6 }
        tableView.snp.makeConstraints { (make) in
            make.left.right.equalToSuperview()
            make.top.equalTo(titleLabel.snp.bottom)
            make.width.equalTo(width)
            make.height.equalTo(height)
        }
        buttonsStack.snp.makeConstraints { (make) in
            make.left.equalToSuperview().offset(Constraints.medium)
            make.right.equalToSuperview().offset(-Constraints.medium)
            make.top.equalTo(tableView.snp.bottom).offset(Constraints.large)
            make.bottom.equalToSuperview().offset(-Constraints.normal)
        }
    }
    
    // MARK: - Field
    typealias Anything = () -> Void
    typealias ButtonTapEvent = (_ sender: UIButton) -> Void
    private var leftTap: ButtonTapEvent? = nil
    private var rightTap: ButtonTapEvent? = nil

    private var options: [String] = [String]()

    private let disposeBag = DisposeBag()
    
    private var style: UITableView.Style = .grouped
    
    private var autoBounces: Bool = true
    
    private let groundColor = Colors.darkMode(Colors.dark, Colors.white)
    
    private lazy var tableView: UITableView = {
        let tableView = UITableView(frame: .zero, style: style)
        tableView.estimatedRowHeight = 44
        tableView.estimatedSectionHeaderHeight = 0;
        tableView.estimatedSectionFooterHeight = 0;
        tableView.backgroundColor = groundColor
        tableView.clipsToBounds = true
        tableView.showsVerticalScrollIndicator = false
        tableView.showsHorizontalScrollIndicator = false
        tableView.separatorInset = .zero
        return tableView
    }()

    private lazy var mj_header: MJRefreshHeader = {
        let header = MJRefreshNormalHeader();
        header.lastUpdatedTimeLabel?.isHidden = true;
        header.setTitle(SRString.Searched.search_device.locastr, for: .idle)
        header.setTitle(SRString.Searched.search_device.locastr, for: .pulling)
        header.setTitle(SRString.Searched.searching.locastr, for: .refreshing)
        return header;
    }()
    
    private lazy var titleLabel: UILabel = {
        let label = UILabel(style: Stylesheet.Label.naviTitle)
        label.backgroundColor = Colors.darkMode(Colors.dark, Colors.blue)
        label.textColor = Colors.white
        return label
    } ()

    private lazy var buttonsStack: UIStackView = {
        let stack = UIStackView();
        stack.axis = .horizontal;
        stack.alignment = .center;
        stack.distribution = .fillEqually;
        stack.spacing = Constraints.normal
        let leftButton = createButton(SRString.Base.cancel.locastr, leftTap)
        let rightButton = createButton(SRString.Base.ok.locastr, rightTap)
        leftButton.backgroundColor = Colors.darkMode(Colors.darkGray, Colors.lightGray)
        leftButton.setTitleColor(Colors.darkMode(Colors.lightGray, Colors.white), for: .normal)
        rightButton.backgroundColor = Colors.darkMode(Colors.darkGray, Colors.blue)
        rightButton.setTitleColor(Colors.darkMode(Colors.white, Colors.white), for: .normal)
        stack.addArrangedSubview(leftButton)
        stack.addArrangedSubview(rightButton)
        return stack
    } ()

    private func createButton(_ text: String?, _ tap: ButtonTapEvent?) ->UIButton {
        let buttom = UIButton(type: .system)
        buttom.apply(Stylesheet.View.dialogButton)
        buttom.setTitle(text, for: .normal)
        guard let _ = text else { return buttom }
        buttom.isUserInteractionEnabled = true
        buttom.rx.tap.subscribe(onNext: { _ in
            if let hasTap = tap { hasTap(buttom) }
        }).disposed(by: disposeBag)
        return buttom
    }
    
    // MARK: - tableViewManger
    
    private var datasource: SingleDatasource!
    
    private lazy var tableViewManger: ZJTableViewManager = {
        let manager = ZJTableViewManager(tableView: tableView)
        manager.register(SingleDialogCell.self, SingleItem.self)
        manager.add(section: section);
        return manager
    } ()
    
    private lazy var section: ZJTableViewSection = {
        let section = BaseTableViewSection()
        section.headerHeight = 0.01
        datasource.items.forEach { (e) in
            e.cellHeight = 44
            section.add(item: e)
        }
        return section
    } ()
    
    // MARK: - methods
    
    func setRefreshHeader(_ action: Anything? = nil) {
        autoBounces = false
        tableView.mj_header = mj_header;
        mj_header.rx.refreshing.asDriver()
            .drive(onNext: { _ in (action ?? {})() })
            .disposed(by: disposeBag)
        mj_header.beginRefreshing()
    }

    func endRefreshing() {
        mj_header.endRefreshing()
    }

    func setTableView(dataSource: UITableViewDataSource, delegate: UITableViewDelegate) {
        self.tableView.dataSource = dataSource
        self.tableView.delegate = delegate
    }

    func register(cellType: UITableViewCell.Type, forCellReuseIdentifier identifier: String) {
        tableView.register(cellType, forCellReuseIdentifier: identifier)
    }

    func indexPath(for cell: UITableViewCell) -> IndexPath? {
        return tableView.indexPath(for: cell)
    }

    func refreshTableView() {
        DispatchQueue.main.async { [weak self] in
            self?.tableView.reloadData()
        }
    }
}
