//
//  DeviceSettingItem.swift
//  CTFit
//
//  Created by Mac on 2020/9/17.
//  Copyright © 2020 jpaxh. All rights reserved.
//

import UIKit
import TableViewManager

enum DeviceSettingItemType {
    case hourly, system_unit, temp_unit, dateformat, brightscreen_time
    case incoming_reminder
    case notify_message, notify_message_type
    case heart_monitor,roll_brightscrren
    case drinkwater_reminder
    case no_disturb
    case findband

    var text: String {
        switch self {
        case .hourly: return SRString.Device.hourly_set.locastr
        case .system_unit: return SRString.Device.system_unit.locastr
        case .temp_unit: return SRString.Device.temp_unit.locastr
        case .dateformat: return SRString.Device.dateformat.locastr
        case .brightscreen_time: return SRString.Device.bright_screen_time.locastr
        case .incoming_reminder:
            switch BandConnect.protocolType {
            case .series_z: return SRString.Device.reminder_incoming_period.locastr
            case .series_s: return SRString.Device.reminder_incoming.locastr
            }
        case .notify_message:
            switch BandConnect.protocolType {
            case .series_z: return SRString.Device.notify_message_period.locastr
            case .series_s: return SRString.Device.notify_message.locastr
            }
        case .notify_message_type: return SRString.Device.notify_message_type.locastr
        case .heart_monitor: return SRString.Device.heart_monitor.locastr
        case .roll_brightscrren: return SRString.Device.roll_brightscrren.locastr
        case .drinkwater_reminder: return SRString.Device.drinkwater_reminder.locastr
        case .no_disturb: return SRString.Device.no_distrub.locastr
        case .findband: return SRString.Connected.findband.locastr
        }
    }
    
    static let collection: [DeviceSettingItemType] = [.hourly, .system_unit, .temp_unit, .dateformat, .brightscreen_time,
                                                      incoming_reminder, notify_message, notify_message_type,
                                                      heart_monitor, roll_brightscrren, drinkwater_reminder,
                                                      no_disturb,
                                                      findband, ]
    
    var singleType: SingleType {
        switch self {
        case .hourly: return .hourly
        case .system_unit: return .system
        case .temp_unit: return .tempUnitType
        case .dateformat: return .dateformatType
        case .brightscreen_time: return .brightScreenTimeType
        default: return .hourly
        }
        
    }
}

final class DeviceSettingItem: BaseTableViewItem {
    
    // MARK: - Cell source
    var text: String? { type.text }
    var detailText: String? {
        switch type {
        case .hourly: return settingsInfo.format.hourly.text
        case .system_unit: return settingsInfo.format.system.text
        case .temp_unit: return settingsInfo.format.tempUnit.text
        case .dateformat: return settingsInfo.format.dateFormat.text
        case .brightscreen_time: return settingsInfo.format.brightScreenTime.text
        case .incoming_reminder: return settingsInfo.callRemind.detailText(hourly: settingsInfo.format.hourly)
        case .notify_message: return settingsInfo.messageRemind.detailText(hourly: settingsInfo.format.hourly)
        case .heart_monitor: return settingsInfo.restingHeartRate.detailText(hourly: settingsInfo.format.hourly)
        case .roll_brightscrren: return settingsInfo.rollBrightScrren.detailText(hourly: settingsInfo.format.hourly)
        case .drinkwater_reminder: return settingsInfo.waterRemind.detailText(hourly: settingsInfo.format.hourly)
        case .no_disturb: return settingsInfo.noDistrub.detailText(hourly: settingsInfo.format.hourly)
        default: return ""
        }
    }

    
    // MARK: - Initialize
    override init() {
        super.init()
        cellHeight = 55
        selectionStyle = .default
        accessoryType = .none
    }

    convenience init(_ type: DeviceSettingItemType, settingsInfo: inout BandSettings) {
        self.init()
        self.type = type
        self.settingsInfo = settingsInfo
        onHandler()
    }
    
    // MARK: - Field
    private var type: DeviceSettingItemType = .system_unit
    private var settingsInfo: BandSettings!
    private lazy var dialog: Dialog = Dialog(with: .normal)
}

extension DeviceSettingItem {
    
    private func onHandler(){
        setSelectionHandler { [weak self] (item) in
            guard let strong = self else { return }
            strong.didSelectedHandler()
        }
    }
    
    private func didSelectedHandler() {
        switch type {
        case .hourly, .system_unit, .temp_unit, .dateformat, .brightscreen_time: showDialog()
        case .incoming_reminder, .notify_message, .heart_monitor, .roll_brightscrren, .drinkwater_reminder, .no_disturb: toDetailController()
        case .notify_message_type: showNotifyTypeDialog()
        case .findband: BandCommand.findBand()
        }
    }
    
    private func toDetailController() {
        var detailType: SRRemindType? = nil
        switch type {
        case .incoming_reminder: detailType = .call
        case .notify_message: detailType = .message
        case .heart_monitor: detailType = .heart
        case .roll_brightscrren: detailType = .brightScrren
        case .drinkwater_reminder: detailType = .water
        case .no_disturb: detailType = .nodisturb
        default : break
        }
        guard let type = detailType else { return }
        push(RemindViewController(with: type, settingsInfo: &settingsInfo))
    }
    
    private func push(_ vc: BaseViewController) {
        Helper.currentVc?.navigationController?.pushViewController(vc, animated: true)
        vc.onDidChangedClosure { [weak self] (_) in
            guard let strong = self else { return }
            strong.reload()
        }
    }
    
    /// Dialog
    private func showDialog() {
        var info: BandSettings = settingsInfo.copy()
        let leftTap: SingleView.ButtonTapEvent = { [weak self] (sender) in
            guard let strong = self else { return }
            strong.dialog.dismiss(true) { _ in }
        }
        
        let rightTap: SingleView.ButtonTapEvent = { [weak self] (sender) in
            guard let strong = self else { return }
            switch strong.type {
            case .hourly: strong.settingsInfo.format.hourly = info.format.hourly
            case .system_unit: strong.settingsInfo.format.system = info.format.system
            case .temp_unit: strong.settingsInfo.format.tempUnit = info.format.tempUnit
            case .dateformat: strong.settingsInfo.format.dateFormat = info.format.dateFormat
            case .brightscreen_time: strong.settingsInfo.format.brightScreenTime = info.format.brightScreenTime
            default : break
            }
            if strong.type == .hourly {
                strong.reloadTableView()
            } else {
                strong.reload()
            }
            strong.dialog.dismiss(true) { _ in }
        }
        let singleView = SingleView(with: type.singleType, settingsInfo: &info, leftTap: leftTap, rightTap: rightTap)
        dialog.pop(singleView, completed: { _ in })
    }
    
    private func showNotifyTypeDialog() {
        var info: BandSettings = settingsInfo.copy()

        let leftTap: SingleView.ButtonTapEvent = { [weak self] (sender) in
            guard let strong = self else { return }
            strong.dialog.dismiss(true) { _ in }
        }
        
        let rightTap: SingleView.ButtonTapEvent = { [weak self] (sender) in
            guard let strong = self else { return }
            strong.settingsInfo.ancsList = info.ancsList
            strong.dialog.dismiss(true) { _ in }
        }
        let notifyTypeView = NotifyTypeView(with: &info, leftTap: leftTap, rightTap: rightTap)

        dialog.pop(notifyTypeView, completed: { _ in }) 
    }
}
