//
//  PersonalHeaderView.swift
//  CTFit
//
//  Created by Mac on 2020/9/15.
//  Copyright © 2020 jpaxh. All rights reserved.
//

import UIKit
import RxSwift
import RxCocoa
import SnapKit
 
let PersonalHeaderIdentifier = "PersonalHeaderIdentifier"
 
class PersonalHeaderView: UITableViewHeaderFooterView {
    
    // MARK: - Initialize
    @available(*, unavailable)
    required init?(coder aDecoder: NSCoder) {
        fatalError("init(coder:) has not been implemented")
    }
    
    override func prepareForReuse() {
        super.prepareForReuse()
    }
    
    fileprivate override init(reuseIdentifier: String?) {
        super.init(reuseIdentifier: PersonalHeaderIdentifier)
        contentView.backgroundColor = Colors.darkMode(Colors.black_less, Colors.blue)
    }
    
    convenience init(with settingsInfo: inout BandSettings) {
        self.init(reuseIdentifier:nil)
        self.settingsInfo = settingsInfo
        setupConstraints()
    }
    
    // MARK: - View customer
    private let disposeBag = DisposeBag()
    
    private var settingsInfo: BandSettings!

    fileprivate func setupConstraints() {
        [portraitButton].forEach { view in
            view.translatesAutoresizingMaskIntoConstraints = false
            addSubview(view)
        }
        portraitButton.snp.makeConstraints { (make) in
            make.centerX.centerY.equalToSuperview()
        }
    }
    
    fileprivate lazy var portraitButton: UIButton = {
        let button = UIButton(style: Stylesheet.View.portraitButton)
        if let hasPortraitPath = settingsInfo.user.portraitPath,
            let docmentPath = documentPath() as NSString?,
            let image = UIImage(contentsOfFile: docmentPath.appendingPathComponent(hasPortraitPath)) {
            button.setImage(image, for: .normal)
        } else {
            button.setImage(ImageRepo.Settings.ic_portrait, for: .normal)
        }
        
        button.rx.tap.subscribe(onNext: {[weak self] _ in
            guard let strong = self else { return }
            strong.showSheet()
        }).disposed(by: disposeBag)
        
        return button
    } ()
}


// MARK: - Handler actions

extension PersonalHeaderView {
    /// - 选择图片
    private func showSheet() {
        AlertManager.sheet(title: nil, message: nil, takePictureHander: { [weak self] (_) in
            guard let strong = self else { return }
            strong.presentCamera()
        }, albumHander: { [weak self] (_) in
            guard let strong = self else { return }
            strong.presentLibrary()
        }, cancelHander: { (_) in
            
        })

    }
    /// - 相册
    private func presentLibrary() {
        let vc = SRLibraryViewController()
        vc.onDidPickerPhoto { [weak self] (image) in
            guard let strong = self else { return }
            strong.save(with: image) { }
            vc.navigationController?.dismiss(animated: true, completion: nil)
        }
       
        let cameraNavivc = BaseNavigationController(rootViewController: vc)
        cameraNavivc.modalPresentationStyle = .fullScreen
        guard let currentVc = Helper.currentVc as? PersonalViewController else { return }
        currentVc.present(cameraNavivc, animated: true, completion: nil)
    }
    
    /// - 拍照
    private func presentCamera() {
        let vc = SRCameraViewController()
        vc.onDidCapturePhoto { [weak self] (image) in
            guard let strong = self else { return }
            strong.save(with: image) { }
            vc.navigationController?.dismiss(animated: true, completion: nil)
        }
       
        let cameraNavivc = BaseNavigationController(rootViewController: vc)
        cameraNavivc.modalPresentationStyle = .fullScreen
        guard let currentVc = Helper.currentVc as? PersonalViewController else { return }
        currentVc.present(cameraNavivc, animated: true, completion: nil)
    }
}

// MARK: - Save portrait image
extension PersonalHeaderView {

    fileprivate static let kPortraitName = "portrait.png"
    
    fileprivate func documentPath() -> String? {
        return NSSearchPathForDirectoriesInDomains(.documentDirectory, .userDomainMask, true).first;
    }
    
    fileprivate func save(with image: UIImage, cmtp: @escaping (() -> Void)) {
        DispatchQueue.global().async { [weak self] in
            guard let strong = self else { return }
            let result = strong.save(image: image)
            if result {
                DispatchQueue.main.async {
                    strong.portraitButton.setImage(image, for: .normal)
                    if strong.settingsInfo.user.portraitPath != PersonalHeaderView.kPortraitName {
                        strong.settingsInfo.user.portraitPath = PersonalHeaderView.kPortraitName
                        let info = BandSettings.extract()
                        info.user.portraitPath = PersonalHeaderView.kPortraitName
                        info.saveToCache()
                    }
                    cmtp()
                }
            }
        }
    }
    
    fileprivate func save(image: UIImage) -> Bool {
        guard let docmentPath = documentPath() as NSString? else { return false; }
        let imagePath = docmentPath.appendingPathComponent(PersonalHeaderView.kPortraitName)
        UILog.v("imagePath: \(imagePath)");
        
        let fileManager = FileManager.default;
        if fileManager.fileExists(atPath: imagePath) {
            do {
                try fileManager.removeItem(atPath: imagePath);
            } catch _ { UILog.v("failed to remove file");
                return false
            }
        }
        
        var data = image.pngData()
        if data == nil { data = image.jpegData(compressionQuality: 0.8) }
        guard let hasData = data else { UILog.v("Representation Image failed"); return false }
        
        do {
            try hasData.write(to: URL(fileURLWithPath: imagePath));
            return true;
        } catch _{
            UILog.v("Failed to write image");
        }
        return false;
    }
    
}
