//
//  File.swift
//  CTFit
//
//  Created by Mac on 2020/9/14.
//  Copyright © 2020 jpaxh. All rights reserved.
//

import Foundation
import TableViewManager

enum PersonalItemType {
    case gender, height, weight, hand_type, mmhg_reference
    var text: String {
        switch self {
        case .gender: return SRString.Personal.gender.locastr
        case .height: return SRString.Personal.height.locastr
        case .weight: return SRString.Personal.weight.locastr
        case .hand_type: return SRString.Personal.hand_type.locastr
        case .mmhg_reference: return SRString.BP.mmhg_reference.locastr
        }
    }
    static let collection: [PersonalItemType] = [.gender, .height, .weight, .hand_type, .mmhg_reference]
}

final class PersonalItem: BaseTableViewItem {
    
    // MARK: - Cell source
    var text: String { type.text }
    var detailText: String {
        switch type {
        case .gender: return settingsInfo.user.gender.text
        case .height: return lengthText(settingsInfo.user.height, settingsInfo.format.system)
        case .weight: return weightText(settingsInfo.user.weight, settingsInfo.format.system)
        case .hand_type: return settingsInfo.user.handType.text
        case .mmhg_reference: return mmhgText()
        }
    }
    
    
    // MARK: - Initialize
    override init() {
        super.init()
        accessoryType = .disclosureIndicator
    }
    
    convenience init(with type: PersonalItemType, settingsInfo: inout BandSettings) {
        self.init()
        self.type = type
        self.settingsInfo = settingsInfo
        onHandler()
    }
    
    // MARK: - Private
    private var type: PersonalItemType = .gender
    private var settingsInfo: BandSettings = BandSettings()
}


extension PersonalItem {
    private func lengthText(_ value: UInt8, _ system: BDSystemType) -> String {
        if system == .metric {
            return "\(value)" + SRString.Unit.cm.locastr
        } else {
            let tuples = BandCalculator.imperial(cm: Int(value))
            return "\(tuples.foot)" + SRString.Unit.foot.locastr + "\(tuples.inch)" + SRString.Unit.inch.locastr
        }
    }
    
    private func weightText(_ value: UInt8, _ system: BDSystemType) -> String {
        if system == .metric {
            return "\(value)" + SRString.Unit.kg.locastr
        } else {
            return "\(lround(BandCalculator.lb(kg: BDCalculatorType(value))))" + SRString.Unit.lb.locastr
        }
    }
    
    private func mmhgText() -> String {
        return "\(settingsInfo.user.dbpReference)/\(settingsInfo.user.sbpReference)" + SRString.BP.mmhg.locastr
    }
}

extension PersonalItem {
    private func onHandler() {
        setSelectionHandler { [weak self] (item) in
            guard let strong = self else { return }
            strong.didSelectedHandler()
        }
    }
    
    private func didSelectedHandler() {
        switch type {
        case .gender: push(SingleViewController(with: .gender, settingsInfo: &settingsInfo))
        case .hand_type: push(SingleViewController(with: .hand_type, settingsInfo: &settingsInfo))
        case .height: push(PickersViewController(with: .height, settingsInfo: &settingsInfo))
        case .weight: push(PickersViewController(with: .weight, settingsInfo: &settingsInfo))
        case .mmhg_reference: push(PickersViewController(with: .mmhg_reference, settingsInfo: &settingsInfo))
        }
    }
    
    fileprivate func push(_ vc: BaseViewController) {
        Helper.currentVc?.navigationController?.pushViewController(vc, animated: true)
        vc.onDidChangedClosure { [weak self] (_) in
            guard let strong = self else { return }
            strong.reload()
        }
    }
}
