//
//  SportSettingItem.swift
//  CTFit
//
//  Created by Mac on 2020/9/24.
//  Copyright © 2020 jpaxh. All rights reserved.
//


import UIKit
import TableViewManager

enum SportSettingItemType {
    case strideWalk, strideRun, goalStep, goalCalories, goalDistance, targetRemind
    case heartHigh, heartLow
    case sedentaryRemind

    var text: String {
        switch self {
        case .strideWalk: return SRString.SportSet.stride_walk.locastr
        case .strideRun: return SRString.SportSet.stride_run.locastr
        case .goalStep: return SRString.SportSet.goal_steps.locastr
        case .goalCalories: return SRString.SportSet.goal_calories.locastr
        case .goalDistance: return SRString.SportSet.goal_distance.locastr
        case .targetRemind: return SRString.SportSet.reminder_target.locastr
        case .heartHigh: return SRString.SportSet.reminder_heart_high.locastr
        case .heartLow: return SRString.SportSet.reminder_heart_low.locastr
        case .sedentaryRemind: return SRString.SportSet.reminder_sedentary.locastr
        }
    }
    
    static var collection: [SportSettingItemType] = {
        switch BandConnect.protocolType {
        case .series_z: return [.strideWalk, .strideRun, .goalStep, .goalCalories, .goalDistance,
                                .heartHigh, .heartLow, .sedentaryRemind]
        case .series_s: return [.goalStep, .goalCalories, .goalDistance,
                                .targetRemind, .sedentaryRemind]
        }
    } ()
}

final class SportSettingItem: BaseTableViewItem {
    
    // MARK: - Cell source
    var text: String? { type.text }
    var detailText: String? {
        switch type {
        case .strideWalk: return lengthText(settingsInfo.user.strideWalk, settingsInfo.format.system)
        case .strideRun: return lengthText(settingsInfo.user.strideRun, settingsInfo.format.system)
        case .goalStep: return stepText(settingsInfo.goal.step)
        case .goalCalories: return caloriesText(settingsInfo.goal.calories)
        case .goalDistance: return distanceText(settingsInfo.goal.distance, settingsInfo.format.system)
        case .targetRemind: return settingsInfo.targetRemind.detailText(hourly: settingsInfo.format.hourly)
        case .heartHigh: return settingsInfo.heartRateAlarm.detailText(isHigh: true)
        case .heartLow: return settingsInfo.heartRateAlarm.detailText(isHigh: false)
        case .sedentaryRemind: return settingsInfo.sedentaryRemind.detailText(hourly: settingsInfo.format.hourly)
        }
    }

    
    // MARK: - Initialize
    override init() {
        super.init()
        cellHeight = 55
        selectionStyle = .default
        accessoryType = .none
    }

    convenience init(_ type: SportSettingItemType, settingsInfo: inout BandSettings) {
        self.init()
        self.type = type
        self.settingsInfo = settingsInfo
        onHandler()
    }
    
    // MARK: - Field
    private var type: SportSettingItemType = .strideRun
    private var settingsInfo: BandSettings!
}

extension SportSettingItem {
    
    private func onHandler(){
        setSelectionHandler { [weak self] (item) in
            guard let strong = self else { return }
            strong.didSelectedHandler()
        }
    }
    
    private func didSelectedHandler() {
        switch type {
        case .strideWalk: push(PickersViewController(with: .strideWalk, settingsInfo: &settingsInfo))
        case .strideRun: push(PickersViewController(with: .strideRun, settingsInfo: &settingsInfo))
        case .goalStep: push(GoalAlarmViewController(with: .step, settingsInfo: &settingsInfo))
        case .goalCalories: push(GoalAlarmViewController(with: .calories, settingsInfo: &settingsInfo))
        case .goalDistance: push(GoalAlarmViewController(with: .distance, settingsInfo: &settingsInfo))
        case .targetRemind: push(RemindViewController(with: .target, settingsInfo: &settingsInfo))
        case .heartHigh: push(HeartAlarmViewController(with: .high, settingsInfo: &settingsInfo))
        case .heartLow: push(HeartAlarmViewController(with: .low, settingsInfo: &settingsInfo))
        case .sedentaryRemind: push(RemindViewController(with: .sedentary, settingsInfo: &settingsInfo))
        }
    }
    
    private func push(_ vc: BaseViewController) {
        Helper.currentVc?.navigationController?.pushViewController(vc, animated: true)
        vc.onDidChangedClosure { [weak self] (_) in
            guard let strong = self else { return }
            strong.reload()
        }
    }
    
}


extension SportSettingItem {
    private func lengthText(_ value: UInt8, _ system: BDSystemType) -> String {
        if system == .metric {
            return "\(value)" + SRString.Unit.cm.locastr
        } else {
            let tuples = BandCalculator.imperial(cm: Int(value))
            return "\(tuples.foot)" + SRString.Unit.foot.locastr + "\(tuples.inch)" + SRString.Unit.inch.locastr
        }
    }
    
    private func stepText(_ value: UInt32) -> String {
        return "\(value)" + SRString.Step.unit_step.locastr
    }
    
    private func caloriesText(_ value: UInt32) -> String {
        return "\(value)" + SRString.Step.unit_calories.locastr
    }
    
    private func distanceText(_ value: UInt32, _ system: BDSystemType) -> String {
        if system == .metric {
            return "\(lround(BandCalculator.km(goal: BDCalculatorType(value))))" + SRString.Unit.km.locastr
        } else {
            return "\(lround(BandCalculator.mile(goal: BDCalculatorType(value))))" + SRString.Unit.mile.locastr
        }
    }
    
    private func heartRateText(_ value: UInt8) -> String {
        return "\(value)" + SRString.Heart.bpm.locastr
    }
    
    private func intervalText(_ value: UInt8) -> String {
        return "\(value)" + SRString.Unit.seconds.locastr
    }
}
