//
//  SettingPickerItem.swift
//  CTFit
//
//  Created by Mac on 2020/9/23.
//  Copyright © 2020 jpaxh. All rights reserved.
//


import Foundation
import TableViewManager

enum SettingPickerItemType {
    case height, weight, sbpReference, dbpReference
    case strideWalk, strideRun
    
    case goalStep, goalCalories, goalDistance
    case heartHigh, heartLow
    case intervalWater, intervalSedentary

    var text: String {
        switch self {
        case .height: return SRString.Personal.height.locastr
        case .weight: return SRString.Personal.weight.locastr
        case .sbpReference: return SRString.BP.systolic_pressure.locastr
        case .dbpReference: return SRString.BP.diastolic_pressure.locastr
        case .strideWalk: return SRString.SportSet.stride_walk.locastr
        case .strideRun: return SRString.SportSet.stride_run.locastr
            
        case .goalStep: return SRString.SportSet.goal_steps.locastr
        case .goalCalories: return SRString.SportSet.goal_calories.locastr
        case .goalDistance: return SRString.SportSet.goal_distance.locastr
            
        case .heartHigh: return SRString.Heart.hightest_heart_rate.locastr
        case .heartLow: return SRString.Heart.lowest_heart_rate.locastr
        case .intervalWater, .intervalSedentary: return SRString.Device.remind_interval.locastr

        }
    }
}

final class SettingPickerItem: BaseExpandTreeCellItem {
    func setValueChanged(_ closure: @escaping ValueChangedClosure) {
        self.didChangedClosure = closure
    }
    
    // MARK: - Cell source
    var text: String { type.text }
    var detailText: String {
        switch type {
        case .height: return lengthText(settingsInfo.user.height, settingsInfo.format.system)
        case .weight: return weightText(settingsInfo.user.weight, settingsInfo.format.system)
        case .sbpReference: return mmhgText(settingsInfo.user.sbpReference)
        case .dbpReference: return mmhgText(settingsInfo.user.dbpReference)
        case .strideWalk: return lengthText(settingsInfo.user.strideWalk, settingsInfo.format.system)
        case .strideRun: return lengthText(settingsInfo.user.strideRun, settingsInfo.format.system)
            
        case .goalStep: return stepText(settingsInfo.goal.step)
        case .goalCalories: return caloriesText(settingsInfo.goal.calories)
        case .goalDistance: return distanceText(settingsInfo.goal.distance, settingsInfo.format.system)
        
        case .heartHigh: return heartRateText(settingsInfo.heartRateAlarm.heartRateHigh)
        case .heartLow: return heartRateText(settingsInfo.heartRateAlarm.heartRateLow)
        case .intervalWater: return intervalText(settingsInfo.waterRemind.interval)
        case .intervalSedentary: return intervalText(settingsInfo.sedentaryRemind.interval)
        }
    }
    
    // MARK: - Initialize
    override init() {
        super.init()
        accessoryType = .none
    }
    
    convenience init(with type: SettingPickerItemType, settingsInfo: inout BandSettings) {
        self.init()
        self.type = type
        self.settingsInfo = settingsInfo
    }
    
    // MARK: - private field
    private var type: SettingPickerItemType = .height
    private var settingsInfo: BandSettings!
    typealias ValueChangedClosure = () -> Void
    private var didChangedClosure: ValueChangedClosure? = nil
    private func onDidChanged() { if let closure = didChangedClosure { closure() } }
    
    
    // MARK: - Sub picker item
    
    func addSubPickerItem(to section: ZJTableViewSection) {
        addSub(item: pickerItem, section: section)
    }
    
    private lazy var pickerItem: PickerItem = {
        let item = PickerItem(components: components, selectRows: selectRows)
        item.setItemSelectedClosure { [weak self] (text, component) in
            guard let strong = self else { return }
            strong.onItemSelected(text, component)
        }
        return item
    } ()
    
    private lazy var components: [[String]] = {
        switch type {
        case .height: return PickerOptionsUitls.components(.height, settingsInfo.format.system)
        case .weight: return PickerOptionsUitls.components(.weight, settingsInfo.format.system)
        case .sbpReference: return PickerOptionsUitls.components(.highestHeartRate)
        case .dbpReference: return PickerOptionsUitls.components(.lowestHeartRate)
        case .strideWalk: return PickerOptionsUitls.components(.strideWalk, settingsInfo.format.system)
        case .strideRun: return PickerOptionsUitls.components(.strideRun, settingsInfo.format.system)
            
        case .goalStep: return PickerOptionsUitls.components(.goalSteps)
        case .goalCalories: return PickerOptionsUitls.components(.goalCalories)
        case .goalDistance: return PickerOptionsUitls.components(.goalDistance, settingsInfo.format.system)
        
        case .heartHigh: return PickerOptionsUitls.components(.highestHeartRate)
        case .heartLow: return PickerOptionsUitls.components(.lowestHeartRate)
        case .intervalWater, .intervalSedentary: return PickerOptionsUitls.components(.interval)
        }
    } ()
    
    private lazy var selectRows: [String] = {
        switch type {
        case .height, .strideWalk, .strideRun:
            var value = settingsInfo.user.height
            if type == .strideWalk { value = settingsInfo.user.strideWalk }
            if type == .strideRun { value = settingsInfo.user.strideRun }
           
            switch settingsInfo.format.system {
            case .metric:
                if type == .height, value > 240 { value = 240 }
                return ["\(value)"]
            case .imperial:
                let tuples = BandCalculator.imperial(cm: Int(value))
                return ["\(tuples.foot)", "\(tuples.inch)"]
            }
        case .weight:
            switch settingsInfo.format.system {
            case .metric: return ["\(settingsInfo.user.weight)"]
            case .imperial:  return ["\(lround(BandCalculator.lb(kg: BDCalculatorType(settingsInfo.user.weight))))"]
            }
            
        case .sbpReference: return ["\(settingsInfo.user.sbpReference)"]
        case .dbpReference: return ["\(settingsInfo.user.dbpReference)"]
            
        case .goalStep: return ["\(settingsInfo.goal.step)"]
        case .goalCalories: return ["\(settingsInfo.goal.calories)"]
        case .goalDistance:
            switch settingsInfo.format.system {
            case .metric: return ["\(lround(BandCalculator.km(goal: BDCalculatorType(settingsInfo.goal.distance))))"]
            case .imperial: return ["\(lround(BandCalculator.mile(goal: BDCalculatorType(settingsInfo.goal.distance))))"]
            }
        case .heartHigh: return ["\(settingsInfo.heartRateAlarm.heartRateHigh)"]
        case .heartLow: return ["\(settingsInfo.heartRateAlarm.heartRateLow)"]
        case .intervalWater: return ["\(settingsInfo.waterRemind.interval)"]
        case .intervalSedentary: return ["\(settingsInfo.sedentaryRemind.interval)"]
        }
    } ()
    
    private func onItemSelected(_ text: String, _ component: Int) {
        guard let value = Int(text) else { return }
        switch type {
        case .height, .strideWalk, .strideRun:
            guard var value = Int(text) else { return }
            
            var selValue: UInt8 = 0
            if type == .height { selValue = settingsInfo.user.height }
            if type == .strideWalk { selValue = settingsInfo.user.strideWalk }
            if type == .strideRun { selValue = settingsInfo.user.strideRun }
            
            if settingsInfo.format.system == .imperial {
                var tuples = BandCalculator.imperial(cm: Int(selValue))
                if component == 0 { tuples.foot = Int(value) } else
                if component == 1 { tuples.inch = Int(value) }
                value = BandCalculator.cm(imperial: tuples)
            }
            
            if type == .height { settingsInfo.user.height = UInt8(value) }
            if type == .strideWalk { settingsInfo.user.strideWalk = UInt8(value) }
            if type == .strideRun { settingsInfo.user.strideRun = UInt8(value) }
            
        case .weight:
            var result = value
            if settingsInfo.format.system == .imperial { result = lround(BandCalculator.kg(lb: BDCalculatorType(value))) }
            settingsInfo.user.weight = UInt8(result)
        case .sbpReference: settingsInfo.user.sbpReference = UInt8(value)
        case .dbpReference: settingsInfo.user.dbpReference = UInt8(value)
        
        case .goalStep: settingsInfo.goal.step = UInt32(value)
        case .goalCalories: settingsInfo.goal.calories = UInt32(value)
        case .goalDistance:
            var result = BandCalculator.goal(km: Double(value))
            if settingsInfo.format.system == .imperial {
                result = BandCalculator.goal(mile: Double(value))
            }
            settingsInfo.goal.distance = UInt32(min(lround(result), 25000))
        case .heartHigh: settingsInfo.heartRateAlarm.heartRateHigh = UInt8(value)
        case .heartLow: settingsInfo.heartRateAlarm.heartRateLow = UInt8(value)
        case .intervalWater: settingsInfo.waterRemind.interval = UInt8(value)
        case .intervalSedentary: settingsInfo.sedentaryRemind.interval = UInt8(value)
        }
        reload()
    }
}


extension SettingPickerItem {
    
    private func lengthText(_ value: UInt8, _ system: BDSystemType) -> String {
        if system == .metric {
            return "\(value)" + SRString.Unit.cm.locastr
        } else {
            let tuples = BandCalculator.imperial(cm: Int(value))
            return "\(tuples.foot)" + SRString.Unit.foot.locastr + "\(tuples.inch)" + SRString.Unit.inch.locastr
        }
    }
    
    private func weightText(_ value: UInt8, _ system: BDSystemType) -> String {
        if system == .metric {
            return "\(value)" + SRString.Unit.kg.locastr
        } else {
            return "\(lround(BandCalculator.lb(kg: BDCalculatorType(value))))" + SRString.Unit.lb.locastr
        }
    }
    
    private func mmhgText(_ value: UInt8) -> String {
        return "\(value)" + SRString.BP.mmhg.locastr
    }
    
    private func stepText(_ value: UInt32) -> String {
        return "\(value)" + SRString.Step.unit_step.locastr
    }
    
    private func caloriesText(_ value: UInt32) -> String {
        return "\(value)" + SRString.Step.unit_calories.locastr
    }
    
    private func distanceText(_ value: UInt32, _ system: BDSystemType) -> String {
        if system == .metric {
            return "\(lround(BandCalculator.km(goal: BDCalculatorType(value))))" + SRString.Unit.km.locastr
        } else {
            return "\(lround(BandCalculator.mile(goal: BDCalculatorType(value))))" + SRString.Unit.mile.locastr
        }
    }
    
    private func heartRateText(_ value: UInt8) -> String {
        return "\(value)" + SRString.Heart.bpm.locastr
    }
    
    private func intervalText(_ value: UInt8) -> String {
        return "\(value)" + SRString.Unit.minute.locastr
    }
}
