//
//  SettingsItem.swift
//  CTFit
//
//  Created by Mac on 2020/9/10.
//  Copyright © 2020 jpaxh. All rights reserved.
//

import UIKit
import TableViewManager

enum SettingsItemType {
    case personal, camera, device, sport, clock, application, faq
    var text: String {
        switch self {
        case .personal: return SRString.Settings.personal.locastr
        case .camera: return SRString.Connected.take_picture.locastr
        case .device: return SRString.Settings.device.locastr
        case .sport: return SRString.Settings.sport.locastr
        case .clock: return SRString.Settings.clock.locastr
        case .application: return SRString.Settings.application.locastr
        case .faq: return SRString.Settings.faq.locastr
        }
    }
    
    var icon: UIImage? {
        switch self {
        case .personal: return ImageRepo.Settings.personal
        case .camera: return ImageRepo.Settings.camera
        case .device: return ImageRepo.Settings.device
        case .sport: return ImageRepo.Settings.sport
        case .clock: return ImageRepo.Settings.clock
        case .application: return ImageRepo.Settings.application
        case .faq: return ImageRepo.Settings.faq
        }
    }
    
    static let collection: [SettingsItemType] = [.personal, .camera, .device, .sport, .clock, .application, .faq]
    
}

final class SettingsItem: BaseTableViewItem {
    
    
    // MARK: - Cell source
    var icon: UIImage? { type.icon?.reSizeImage(CGSize(width: 48, height: 48)) }
    var text: String? { type.text }
    
    // MARK: - Initialize
    override init() {
        super.init()
        cellHeight = 75
        selectionStyle = .default
        accessoryType = .none
    }
    
    convenience init(_ type: SettingsItemType) {
        self.init()
        self.type = type
        onHandler()
    }
    
    // MARK: - Field
    var type: SettingsItemType = .device
}

extension SettingsItem {
    private func onHandler(){
        setSelectionHandler { [weak self] (item) in
            guard let strong = self else { return }
            strong.didSelectedHandler()
        }
    }
    
    private func didSelectedHandler() {
        switch type {
        case .personal: push(PersonalViewController())
        case .device: push(DeviceSettingViewController())
        case .sport: push(SportSettingViewController())
        case .clock: push(ClockSettingViewController())
        case .faq: push(FaqViewController())
        case .application: push(ApplicationViewController())
        case .camera: presentCamera()
        }
    }
    
    private func push(_ vc: BaseViewController) {
        guard sureReaded() else { return }
        guard let currentVc = Helper.currentVc as? SettingsViewController else { return }
        currentVc.navigationController?.pushViewController(vc, animated: true)
    }
    
    private func presentCamera() {
        let cameraNavivc = BaseNavigationController(rootViewController: SRCameraViewController(type: .device))
        cameraNavivc.modalPresentationStyle = .fullScreen
        guard let currentVc = Helper.currentVc as? SettingsViewController else { return }
        currentVc.present(cameraNavivc, animated: true, completion: nil)
    }
    
    func sureReaded() -> Bool {
        let collection: [SettingsItemType] = [.personal, .device, .sport, .clock]
        if collection.contains(type), !BandConnect.isConnected {
            ToastUtils.showToast(error: SRString.Base.no_connected_device.locastr)
            return false
        }
        switch BandConnect.protocolType {
        case .series_z: return sureReadedZ()
        case .series_s: return sureReadedS()
        }
    }
    
    func sureReadedZ() -> Bool {
        let collection: [SettingsItemType] = [.personal, .device, .sport, ]
        if collection.contains(type) {
            if BDReaded.firmware, BDReaded.userInfo {
                let feature = BandData.firmwareFeature
                if !feature.waterRemind, !feature.dateformat_detect_brightScrren  { return true }
                if feature.waterRemind || feature.dateformat_detect_brightScrren, BDReaded.drinkInfo { return true }
            }
            ToastUtils.showToast(error: SRString.Base.sure_readed_alert.locastr)
            return false
        }
        
        if type == .clock {
            if BDReaded.clockInfo { return true}
            ToastUtils.showToast(error: SRString.Base.sure_readed_alert.locastr)
            return false
        }
        return true
    }
    
    func sureReadedS() -> Bool {
        let collection: [SettingsItemType] = [.personal, .device, .sport, ]
        if collection.contains(type) {
            if BDReaded.hybridInfo { return true }
            ToastUtils.showToast(error: SRString.Base.sure_readed_alert.locastr)
            return false
        }
        if type == .clock {
            if BDReaded.clockInfo { return true}
            ToastUtils.showToast(error: SRString.Base.sure_readed_alert.locastr)
            return false
        }
        return true
    }
}
