//
//  TimeItem.swift
//  CTFit
//
//  Created by Mac on 2020/6/1.
//  Copyright © 2020 shirajo. All rights reserved.
//

import Foundation
import TableViewManager

enum TimeItemType {
    case start, end
    var text: String {
        switch self {
        case .start: return SRString.Period.start_time.locastr
        case .end: return SRString.Period.end_time.locastr
        }
    }
}

final class TimeItem: BaseExpandTreeCellItem {
    func setOnValueChanged(_ closure: @escaping ValueChangedClosure) {
        self.didChangedClosure = closure
    }
    
    // MARK: - Cell sources
    var text: String { type.text }
    var detailText: String { CalendarUtils.stringFormat(hour, minute, hourly) }
    
    // MARK: - Initialize
    convenience init(with type: TimeItemType, hour: Int, minute: Int, houly: BDHourlyType = .h24) {
        self.init()
        self.type = type
        self.hour = hour
        self.minute = minute
        self.hourly = houly

    }
    
    // MARK: -  field
    var hour: Int = 0
    var minute: Int = 0
    private var type: TimeItemType = .start
    private var hourly: BDHourlyType = .h24
    
    typealias ValueChangedClosure = (_ item: TimeItem) -> Void
    private var didChangedClosure: ValueChangedClosure? = nil
    private func onDidChanged() { if let closure = didChangedClosure { closure(self) } }

    // MARK: - Sub picker item
    func addSubPickerItem(to section: ZJTableViewSection) {
        addSub(item: pickerItem, section: section)
    }
    private lazy var pickerItem: PickerItem = {
        let item = PickerItem(components: components, selectRows: selectRows)
        item.setItemSelectedClosure { [weak self] (text, component) in
            guard let strong = self else { return }
            strong.onItemSelected(text, component)
        }
        return item
    } ()
    
    private lazy var components: [[String]] = {
        if type == .start { return PickerOptionsUitls.components(.timeStart) }
        return PickerOptionsUitls.components(.timeEnd)
    } ()
    
    private lazy var selectRows: [String] = {
        switch type {
        case .start: return ["\(hour)", "\(0)"]
        case .end: return ["\(hour)", "\(59)"]
        }
    } ()
    
    private func onItemSelected(_ text: String, _ component: Int) {
        guard let value = Int(text) else { return }
        if component == 0 { hour = value}
        if component == 1 { minute = value}
        reload()
        onDidChanged()
    }
}
