//
//  SportHistoryCell.swift
//  CTFit
//
//  Created by Mac on 2020/6/20.
//  Copyright © 2020 shirajo. All rights reserved.
//

import UIKit
import TableViewManager

final class SportHistoryCell: BaseTableViewCell, ZJCellProtocol {

    // MARK: - ZJCellProtocol
    
    var item: SportHistoryItem!
    
    typealias ZJCelltemClass = SportHistoryItem
    
    override func cellWillAppear() {
        super.cellWillAppear()
        guard item != nil else { return }
        update(item.record)
    }
    
    // MARK: - Initialize
    override init(style: UITableViewCell.CellStyle, reuseIdentifier: String?) {
        super.init(style: .subtitle, reuseIdentifier: reuseIdentifier)
        self.textLabel?.font = .customFont(ofSize: 15)
        self.textLabel?.textColor = Colors.Base.foreground
        self.detailTextLabel?.textColor = Colors.Base.foreground
        self.textLabel?.numberOfLines = 1
        self.detailTextLabel?.numberOfLines = 1
    }
}

extension SportHistoryCell {
    
    private func update(_ record: SportRecorder) {
        let type: BDSportType =  BDSportType.parse(UInt8(record.sportType)) ?? .normal
        self.imageView?.image = type.image.reSizeImage(CGSize(width: 29, height: 29))
        self.textLabel?.text = record.startTimeText + durationText(record.duration)
        self.detailTextLabel?.text = distanceText(m: Int(record.distance))
    }
    
    private var isImperial: Bool { BandSettings.isImperial }
    
    private var distanceUnit: String {
        (isImperial ? SRString.Unit.mile.locastr:SRString.Unit.km.locastr)
    }
    
    private func distanceText(m: Int) -> String {
        if m < 1000 {
            return SRString.Sport.distance.locastr + String(format: " %d ", m) + SRString.Unit.m.locastr
        } else {
            let value = (isImperial ? BandCalculator.mile(km: Double(m)/1000.0) : Double(m)/1000.0)
            return SRString.Sport.distance.locastr + String(format: " %0.1f ", value) + distanceUnit
        }
    }
    
    private func durationText(_ duration: Int) -> String {
        let h = duration/3600
        let m = (duration/60)%60
        if h>0 {
            return "\(h)" + SRString.Unit.hour.locastr + "\(m)" + SRString.Unit.min.locastr
        } else {
            return "\(m)" + SRString.Unit.minute.locastr

        }
    }
}
