//
//  SportDetailChartCell.swift
//  CTFit
//
//  Created by Mac on 2020/10/12.
//  Copyright © 2020 jpaxh. All rights reserved.
//


import UIKit
import TableViewManager
import Charts

final class SportDetailChartCell: BaseTableViewCell, ZJCellProtocol {

    // MARK: - ZJCellProtocol
    
    var item: SportDetailChartItem!
    
    typealias ZJCelltemClass = SportDetailChartItem
    
    override func cellWillAppear() {
        super.cellWillAppear()
        guard item != nil else { return }
        update(item)
    }
    
    // MARK: - Initialize
    
    override init(style: UITableViewCell.CellStyle, reuseIdentifier: String?) {
        super.init(style: .default, reuseIdentifier: reuseIdentifier)
        backgroundColor = Colors.sportRecordBack
        addSubviews()
    }
    
    // MARK: - View customer
    
    private lazy var iconViw: UIImageView = {
        let view = UIImageView(image: ImageRepo.Sport.realtime_speed)
        return view
    } ()
    
    private lazy var titleLabel: UILabel = {
        let label = UILabel(style: Stylesheet.Label.text)
        label.textColor = Colors.white
        label.font = .customFont(ofSize: 17)
        label.text = SRString.Sport.realtime_speed.locastr + "(\(speedUnit()))"
        return label
    }()
    
    private lazy var headerView: UIView = {
        let view = UIView()
        
        [iconViw, titleLabel].forEach { e in
            e.translatesAutoresizingMaskIntoConstraints = false
            view.addSubview(e)
        }
        
        iconViw.snp.makeConstraints { (make) in
            make.left.centerY.equalToSuperview()
        }
        
        titleLabel.snp.makeConstraints { (make) in
            make.centerY.equalToSuperview()
            make.left.equalToSuperview().offset(Constraints.large)
        }

        return view
    }()
        
    private lazy var lineChart: LineChartView = {
        let chart = LineChartView()
        ChartsUtils.Sport.speed(chart)
        return chart
    } ()
    
    private func addSubviews() {
        [headerView, lineChart].forEach { view in
            view.translatesAutoresizingMaskIntoConstraints = false
            addSubview(view)
        }
        
        headerView.snp.makeConstraints { (make) in
            make.top.right.equalToSuperview()
            make.leading.equalTo(self.snp.leadingMargin)
            make.height.equalTo(64)
        }
        
        lineChart.snp.makeConstraints { (make) in
            make.leading.trailing.equalToSuperview()
            make.top.equalTo(headerView.snp.bottom)
            make.bottom.equalToSuperview().offset(-Constraints.medium)
            make.height.equalTo(140)
        }
    }
}

extension SportDetailChartCell {
    
    private var isImperial: Bool { BandSettings.isImperial }
    
    private func parse(item: SportDetailChartItem, closure: @escaping(_ fastSpeed: Double, _ values: [Double]?, _ points: [SportPoint]) -> Void) {
        DispatchQueue.global().async { [weak self] in
            guard let strong = self else { return }
            let record = item.record
            var points: [SportPoint] = []
            record.tracks.forEach { points += $0.points  }

            var fastSpeed: Double = 0
            var values: [Double] = [Double]()
            points.forEach { (point) in
                if point.speed > fastSpeed { fastSpeed = point.speed }
                var speed = point.speed * 3.6
                if strong.isImperial { speed = BandCalculator.mile(km: speed)}
                values.append(speed)
            }
            fastSpeed *= 3.6
            if strong.isImperial { fastSpeed = BandCalculator.mile(km: fastSpeed)}
            DispatchQueue.main.async {
                closure(fastSpeed, values, points)
            }
        }
    }

    private func update(_ item: SportDetailChartItem) {
        titleLabel.text = SRString.Sport.realtime_speed.locastr + "(\(speedUnit()))"
        self.parse(item: item, closure: { [weak self] (fastSpeed, values, points) in
            guard let strong = self else { return }
            strong.updateChart(fastSpeed, values, points)
        })
    }
    
    private func updateChart(_ fastSpeed: Double, _ values: [Double]?, _ points: [SportPoint]) {
        if fastSpeed > 25 {
            lineChart.leftAxis.axisMaximum = 50
        } else if fastSpeed > 50 {
            lineChart.leftAxis.axisMaximum = 100
        }
        
        lineChart.highlightValue(nil)
        let marker = MarkerUtils.SportSpeed(items: points);  marker.chartView = lineChart
        lineChart.marker = marker;
        updateChartData(values: values)
    }

    
    private func speedUnit() -> String {
        return isImperial ? SRString.Unit.mile_ph.locastr : SRString.Unit.km_ph.locastr
    }
    
    /// Y轴表示实时速度
    private func updateChartData(values: [Double]?) {
        var entries = [ChartDataEntry]();
        if let hasValues = values {
            for (index, value) in hasValues.enumerated() {
                entries.append(ChartDataEntry(x: Double(index), y: Double(value)));
            }
        } else {
            lineChart.data = nil;
            lineChart.notifyDataSetChanged();
            return;
        }

        var set: LineChartDataSet!;
        if let hasData = lineChart.data, hasData.dataSetCount > 0 {
            set = hasData.dataSets[0] as? LineChartDataSet;
            set.replaceEntries(entries)
            hasData.notifyDataChanged();
            lineChart.notifyDataSetChanged();
        } else {
            set = LineChartDataSet(entries: entries, label: "Sport Speed");
            set?.valueTextColor = UIColor.clear//设置折线上的字体颜色 - 运动距离页面
            set.mode = .cubicBezier;
            set.drawCirclesEnabled = false;
            set.drawCircleHoleEnabled = false;
            set.formLineWidth = 1;
            set.formSize = 15;
            set.formLineDashLengths = [5, 2.5];
            
            set.colors = [Colors.orange, UIColor.yellow, UIColor.red];
            let gradientColors = [ChartColorTemplates.colorFromString("#00FBAB2D").cgColor,
                                  ChartColorTemplates.colorFromString("#FFFBAB2D").cgColor]
            let gradient = CGGradient(colorsSpace: nil, colors: gradientColors as CFArray, locations: nil)!
            set?.fillAlpha = 1
            set?.fill = Fill(linearGradient: gradient, angle: 90) //.linearGradient(gradient, angle: 90)
            
            set.drawFilledEnabled = true // Draw the Gradient
            set.drawFilledEnabled = true // Draw the Gradient
      
            let dataSets = [set];
            let data = LineChartData(dataSets: dataSets as? [IChartDataSet]);
            self.lineChart.data = data;
        }
    }
}


