//
//  SportDetailMoreCell.swift
//  CTFit
//
//  Created by Mac on 2020/10/12.
//  Copyright © 2020 jpaxh. All rights reserved.
//

import UIKit
import TableViewManager

final class SportDetailMoreCell: BaseTableViewCell, ZJCellProtocol {

    // MARK: - ZJCellProtocol
    
    var item: SportDetailMoreItem!
    
    typealias ZJCelltemClass = SportDetailMoreItem
    
    override func cellWillAppear() {
        super.cellWillAppear()
        guard item != nil else { return }
        update(item)
    }
    
    // MARK: - Initialize
    
    override init(style: UITableViewCell.CellStyle, reuseIdentifier: String?) {
        super.init(style: .default, reuseIdentifier: reuseIdentifier)
        backgroundColor = Colors.sportRecordBack
        addSubviews()
    }
    
    // MARK: - View customer
    
    private lazy var iconViw: UIImageView = {
        let view = UIImageView(image: ImageRepo.Sport.pace_speed)
        return view
    } ()
    
    private lazy var titleLabel: UILabel = {
        let label = UILabel(style: Stylesheet.Label.text)
        label.textColor = Colors.white
        label.font = .customFont(ofSize: 17)
        label.text = SRString.Sport.pace_speed.locastr
        return label
    }()
    
    private lazy var headerView: UIView = {
        let view = UIView()
        
        [iconViw, titleLabel].forEach { e in
            e.translatesAutoresizingMaskIntoConstraints = false
            view.addSubview(e)
        }
        
        iconViw.snp.makeConstraints { (make) in
            make.left.centerY.equalToSuperview()
        }
        
        titleLabel.snp.makeConstraints { (make) in
            make.centerY.equalToSuperview()
            make.left.equalToSuperview().offset(Constraints.large)
        }

        return view
    }()
        
    private lazy var distanceAttr: SportDetailMoreAttr = {
        let attr = SportDetailMoreAttr()
        attr.title = distanceUnit()
        return attr
    } ()
    
    private lazy var durationAttr: SportDetailMoreAttr = {
        let attr = SportDetailMoreAttr()
        attr.title = SRString.Sport.use_time.locastr
        return attr
    }()
    
    private lazy var paceAttr: SportDetailMoreAttr = {
        let attr = SportDetailMoreAttr()
        attr.title = SRString.Sport.pace_speed.locastr
        return attr
    }()
    
    private lazy var stackView: UIStackView = {
        let view = UIStackView(arrangedSubviews: [distanceAttr, durationAttr, paceAttr])
        view.axis = .horizontal;
        view.distribution = .fillEqually;
        view.alignment = .center;
        view.spacing = 0;
        return view
    } ()
    
    private func addSubviews() {
        [headerView, stackView].forEach { view in
            view.translatesAutoresizingMaskIntoConstraints = false
            addSubview(view)
        }
        
        headerView.snp.makeConstraints { (make) in
            make.top.right.equalToSuperview()
            make.leading.equalTo(self.snp.leadingMargin)
            make.height.equalTo(64)
        }
        
        stackView.snp.makeConstraints { (make) in
            make.leading.trailing.equalToSuperview()
            make.top.equalTo(headerView.snp.bottom)
            make.bottom.equalToSuperview().offset(-Constraints.medium)

        }
    }
}

extension SportDetailMoreCell {
    private func update(_ item: SportDetailMoreItem) {
        distanceAttr.title = distanceUnit()
        distanceAttr.text = distanceText(item.record.distance)
        durationAttr.text = item.record.durationText
        paceAttr.text = paceText(item.record.averageSpeend)

    }
    
    private var isImperial: Bool { BandSettings.isImperial }

    private func distanceUnit() -> String {
        return isImperial ? SRString.Unit.mile.locastr : SRString.Unit.km.locastr
    }
    
    private func distanceText(_ m: Double) -> String {
        if m < 1 { return "0"}
        var distance = m / 1000
        if isImperial { distance = BandCalculator.mile(km: distance)}
        return String(format: "%0.2f", distance)
    }
    
    private func paceText(_ speedInMps: Double) -> String {
        guard speedInMps > 0 else { return "00'00\"" }
        /// 1km 耗时多少秒
        var paceInSeconds = Int(1000 / speedInMps);
        /// 英制处理
        if isImperial {
            /// 1英里 耗时多少秒
            paceInSeconds = Int(1609 / speedInMps)
        }
        /// 分钟
        let paceMins: Int = paceInSeconds / 60
        /// 秒
        let paceSeconds: Int = paceInSeconds % 60
        let text = String(format: "%02d'%02d\"", paceMins, paceSeconds)
        return text
    }
}

class SportDetailMoreAttr: UIStackView {
    
    var title: String? {
        didSet { titleLabel.text = title }
    }
    
    var text: String? {
        didSet { valueLabel.text = text }
    }
    
    private lazy var titleLabel: UILabel = {
        let label = UILabel(style: Stylesheet.Label.description);
        label.textAlignment = .center
        label.font = .customFont(ofSize: 17)
        label.text = "--"
        return label;
    }()
    
    private lazy var valueLabel: UILabel = {
        let label = UILabel(style: Stylesheet.Label.text);
        label.textAlignment = .center
        label.textColor = Colors.white
        label.font = .customFont(ofSize: 17)
        label.text = "--"
        return label;
    }()
    
    // MARK: - initialize
    override init(frame: CGRect) {
        super.init(frame: frame);
        initialize();
    }
    
    required init(coder: NSCoder) {
        super.init(coder: coder);
        initialize();
    }
    
    fileprivate func initialize() {
        loadSubviews();
    }
    
    fileprivate func loadSubviews() {
        axis = .vertical;
        distribution = .fillEqually
        spacing = Constraints.small;
        addArrangedSubview(titleLabel);
        addArrangedSubview(valueLabel);
    }
}
