//
//  SportDetailPaceCell.swift
//  CTFit
//
//  Created by Mac on 2020/10/12.
//  Copyright © 2020 jpaxh. All rights reserved.
//


import UIKit
import TableViewManager

final class SportDetailPaceCell: BaseTableViewCell, ZJCellProtocol {

    // MARK: - ZJCellProtocol
    
    var item: SportDetailPaceItem!
    
    typealias ZJCelltemClass = SportDetailPaceItem
    
    override func cellWillAppear() {
        super.cellWillAppear()
        guard item != nil else { return }
        update(item)
    }
    
    // MARK: - Initialize
    
    override init(style: UITableViewCell.CellStyle, reuseIdentifier: String?) {
        super.init(style: .default, reuseIdentifier: reuseIdentifier)
        backgroundColor = Colors.sportRecordBack
        addSubviews()
    }
    
    // MARK: - View customer
        
    private lazy var paceAttrFast: SportDetailPaceAttr = {
        let attr = SportDetailPaceAttr()
        attr.title = SRString.Sport.pace_speed_fastest.locastr
        return attr
    } ()
    
    private lazy var paceAttrAverage: SportDetailPaceAttr = {
        let attr = SportDetailPaceAttr()
        attr.title = SRString.Sport.pace_speed_average.locastr
        return attr
    }()
    
    private lazy var paceAttrSlowest: SportDetailPaceAttr = {
        let attr = SportDetailPaceAttr()
        attr.title = SRString.Sport.pace_speed_slowest.locastr
        return attr
    }()
    
    private lazy var stackView: UIStackView = {
        let view = UIStackView(arrangedSubviews: [paceAttrFast, paceAttrAverage, paceAttrSlowest])
        view.axis = .horizontal;
        view.distribution = .fillEqually;
        view.alignment = .center;
        view.spacing = 2;
        return view
    } ()
    
    private func addSubviews() {
        [stackView].forEach { view in
            view.translatesAutoresizingMaskIntoConstraints = false
            addSubview(view)
        }
        
        stackView.snp.makeConstraints { (make) in
            make.centerY.equalToSuperview()
            make.leading.trailing.equalToSuperview()
        }
    }
}

extension SportDetailPaceCell {
    private func update(_ item: SportDetailPaceItem) {
        paceAttrFast.speed = item.record.fastSpeed
        paceAttrAverage.speed = item.record.averageSpeend
        paceAttrSlowest.speed = item.record.slowestSpeed
    }
}

class SportDetailPaceAttr: UIStackView {
    
    private var isImperial: Bool { BandSettings.isImperial }

    var speed: Double = 0 {
        didSet { paceLabel.text = paceText(speed) }
    }
    
    var title: String? {
        didSet { titleLabel.text = title }
    }
       
    private lazy var paceLabel: UILabel = {
        let label = UILabel(style: Stylesheet.Label.text);
        label.font = .boldCustomFont(ofSize: 30)
        label.textColor = UIColor.white;
        label.text = "00'00\""
        return label;
    }()
    
    private lazy var titleLabel: UILabel = {
        let label = UILabel(style: Stylesheet.Label.description);
        label.textAlignment = .center
        label.font = .customFont(ofSize: 17)
        label.text = "--"
        return label;
    }()
    
    
    // MARK: - initialize
    override init(frame: CGRect) {
        super.init(frame: frame);
        initialize();
    }
    
    required init(coder: NSCoder) {
        super.init(coder: coder);
        initialize();
    }
    
    fileprivate func initialize() {
        loadSubviews();
    }
    
    fileprivate func loadSubviews() {
        axis = .vertical;
        distribution = .fillEqually
        spacing = 0;
        addArrangedSubview(paceLabel);
        addArrangedSubview(titleLabel);
    }
    
    /// spped (m/s)
    fileprivate func paceText(_ speedInMps: Double) -> String {
        
        guard speedInMps > 0 else { return "00'00\"" }
        
        /// 1km 耗时多少秒
        var paceInSeconds = Int(1000 / speedInMps);
        /// 英制处理
        if isImperial {
            /// 1英里 耗时多少秒
            paceInSeconds = Int(1609 / speedInMps)
        }
        /// 分钟
        let paceMins: Int = paceInSeconds / 60
        /// 秒
        let paceSeconds: Int = paceInSeconds % 60
        let text = String(format: "%02d'%02d\"", paceMins, paceSeconds)
        return text
    }
}
