//
//  SportTrackView.swift
//  CTFit
//
//  Created by Mac on 2020/6/23.
//  Copyright © 2020 shirajo. All rights reserved.
//

import UIKit
import RxSwift

class SportTrackView: UIView {
    
    func onViewDidLoad() { }
    func onViewDidAppear() { }
    func onViewDidDisappear() { }
    
    // MARK: - Initialize

    @available(*, unavailable)
    required init?(coder aDecoder: NSCoder) {
        fatalError("init(coder:) has not been implemented")
    }
    
    init(_ sportRecord: SportRecorder) {
        super.init(frame: .zero)
        recorder = sportRecord
        addSubviews()
        updateInfo()
        observers()
    }
    

    // MARK: - View customer
    
    private var recorder: SportRecorder!
    
    private lazy var mapView: MapView = MapView(isResultTrack: true)
    
    private lazy var dateView: SportDateView = {
        let view = SportDateView()
        view.sportType = BDSportType.parse(UInt8(truncatingIfNeeded: recorder.sportType))
        view.sportDate = recorder.startTimeTextTrack()
        return view
    } ()
    
    private lazy var bottomView: TrackBottomView = {
        let view = TrackBottomView()
        view.layer.masksToBounds = true
        view.layer.cornerRadius = 16
        view.layer.shadowOpacity = 0.5;
        view.layer.shadowOffset = CGSize(width: 0, height: 0);
        view.layer.shadowRadius = 5.0;
        return view
    } ()
    
    private func addSubviews() {
        [mapView, dateView, bottomView].forEach { view in
            view.translatesAutoresizingMaskIntoConstraints = false
            addSubview(view)
        }

        let margin: CGFloat = 12
        mapView.snp.makeConstraints { (make) in
            make.edges.equalToSuperview().inset(UIEdgeInsets(top: 0, left: 0, bottom: 0, right: 0))
        }
        dateView.snp.makeConstraints { (make) in
            make.left.top.equalToSuperview().offset(margin)
        }
        bottomView.snp.makeConstraints { (make) in
            make.left.equalToSuperview().offset(margin)
            make.right.equalToSuperview().offset(-margin)
            make.bottom.equalToSuperview().offset(UIDevice.bottomLayoutGuide > 0 ? -Constraints.medium : -margin)
        }
    }
    
    // MARK: - updateInfo
    
    private func updateInfo() {
        recorder.tracks.forEach { (track) in
            track.getOverlay { [weak self] (overlay) in
                guard let strong = self else { return }
                if let hasOverlay = overlay {
                    strong.mapView.addOverlay(hasOverlay)
                    if track.distance > 1000 {
                        let padding = UIEdgeInsets(top: 30, left: 10, bottom: 180, right: 10)
                        strong.mapView.setVisibleMapRect(hasOverlay.boundingMapRect, edgePadding: padding, animated: true)
                    } else {
                        if let centerCoordinate = track.centerCoordinate {
                            strong.mapView.setRegion(coordinate: centerCoordinate)
                        }
                    }
                }
            }
            annotation(track: track)
        }
        bottomView.setRecorder(recorder)
    }
    
    private func annotation(track: SportTrack) {
        guard track.points.count>0 else { return }
        let firstPoint = track.firstPoint!
        let lastPoint = track.lastPoint!
        annotation(point: firstPoint)
        annotation(point: lastPoint, isLast: true)
    }
    
    private func annotation(point: SportPoint, isLast: Bool = false) {
        point.compatible { [weak self] (location) in
            guard let strong = self else { return }
            let annotation = MapAnnotationPoint(lat: location.latitude, lon: location.longitude);
            annotation.image = isLast ? ImageRepo.Sport.track_end : ImageRepo.Sport.track_start
            
            let date = Date(timeIntervalSince1970: TimeInterval(point.time));
            let formatter = DateFormatter();
            formatter.dateStyle = .none;
            formatter.timeStyle = .medium;
            annotation.title = formatter.string(from: date);
            
            formatter.dateStyle = .medium;
            formatter.timeStyle = .none;
            annotation.subtitle = formatter.string(from: date);
            strong.mapView.addAnnotation(annotation)
        }
    }
    
    private let disposeBag = DisposeBag()

    private func observers() {
        BandSettings.observable.subscribe(onNext: { [weak self] (options) in
            guard let strong = self else { return }
            guard options.contains(.system) else { return }
            strong.bottomView.setRecorder(strong.recorder)
        }).disposed(by: disposeBag)
    }
}


class SportDateView: UIView {
    
    let iconHeight: CGFloat = 25
    
    /// Date of tracking record
    var sportDate: String? {
        didSet { timeLabel.text = sportDate }
    }

    var sportType: BDSportType? {
        didSet { iconView.image = (sportType ?? .normal).image.reSizeImage(CGSize(width: iconHeight, height: iconHeight)) }
    }
    
    override init(frame: CGRect) {
        super.init(frame: frame);
        self.addSubviews();
    }
    
    required init?(coder aDecoder: NSCoder) {
        super.init(coder: aDecoder);
    }
    
    private func addSubviews() {
        self.backgroundColor = Colors.sportRecordBack;
        self.borderCorner = 12;
        
        [iconView, timeLabel].forEach { view in
            view.translatesAutoresizingMaskIntoConstraints = false
            addSubview(view)
        }

        iconView.snp.makeConstraints { (make) in
            make.leading.equalToSuperview().offset(8)
            make.top.equalToSuperview().offset(8)
            make.bottom.equalToSuperview().offset(-8)
        }
        
        timeLabel.snp.makeConstraints { (make) in
            make.centerY.equalToSuperview()
            make.trailing.equalToSuperview().offset(-8)
            make.leading.equalTo(self.iconView.snp.trailing).offset(8)
        }
    }
    
    private var iconView: UIImageView =  {
        let view = UIImageView()
        return view
    } ()
     
    private var timeLabel: UILabel = {
        let label = UILabel(style: Stylesheet.Label.text)
        label.textColor = Colors.white
        return label
    } ()
}
