//
//  TrackBottomView.swift
//  CTFit
//
//  Created by Mac on 2020/6/23.
//  Copyright © 2020 shirajo. All rights reserved.
//

import UIKit

class TrackBottomView: UIView {
    
    private var isImperial: Bool { BandSettings.isImperial }

    private func distanceUnit() -> String {
        return isImperial ? SRString.Unit.mile.locastr : SRString.Unit.km.locastr
    }
    
    private func distanceText(_ m: Double) -> String {
        if m < 1 { return "0"}
        var distance = m / 1000
        if isImperial { distance = BandCalculator.mile(km: distance)}
        return String(format: "%0.2f", distance)
    }
    
    func setRecorder(_ record: SportRecorder) {
        distanceAttr.value = distanceText(record.distance)
        distanceAttr.unit = distanceUnit()
        timeAttr.value = record.durationText
        caloriesAttr.value = "0"
        paceAttrFast.speed = record.fastSpeed
        paceAttrAverage.speed = record.averageSpeend
        paceAttrSlowest.speed = record.slowestSpeed
    }
    
    // MARK: - Initialize

    @available(*, unavailable)
    required init?(coder aDecoder: NSCoder) {
        fatalError("init(coder:) has not been implemented")
    }
    
    init() {
        super.init(frame: .zero)
        backgroundColor = Colors.sportRecordBack
        addSubviews()
    }
    
    private func addSubviews() {
        [moreStackView, gradientLinear, paceStackView].forEach { view in
            view.translatesAutoresizingMaskIntoConstraints = false
            addSubview(view)
        }
        
        let horizontalSpacing = Constraints.normal
        let verticalSpacing = 12
        moreStackView.snp.makeConstraints { (make) in
            make.left.top.equalToSuperview().offset(verticalSpacing)
            make.right.equalToSuperview().offset(-horizontalSpacing)
        }
        
        gradientLinear.snp.makeConstraints { (make) in
            make.left.equalToSuperview().offset(horizontalSpacing)
            make.right.equalToSuperview().offset(-horizontalSpacing)
            make.top.equalTo(moreStackView.snp.bottom).offset(verticalSpacing)
            make.height.equalTo(GradientLinear.kLinearHeight)
        }
        
        paceStackView.snp.makeConstraints { (make) in
            make.left.equalToSuperview().offset(horizontalSpacing)
            make.right.equalToSuperview().offset(-horizontalSpacing)
            make.top.equalTo(gradientLinear.snp.bottom).offset(verticalSpacing)
            make.right.bottom.equalToSuperview().offset(-verticalSpacing)
        }
    }


    // MARK: - Filed
    
    private lazy var gradientLinear: GradientLinear = {
        let view = GradientLinear();
        return view
    } ()
    
    /// more statcks

    private lazy var distanceAttr: TrackDetailMoreAttr = {
        let attr = TrackDetailMoreAttr()
        attr.title = SRString.Sport.distance.locastr
        attr.unit = distanceUnit()
        attr.value = "0.00"
        return attr
    } ()
    
    private lazy var timeAttr: TrackDetailMoreAttr = {
        let attr = TrackDetailMoreAttr()
        attr.title = SRString.Sport.use_time.locastr
        attr.unit = SRString.Sport.unit_time.locastr
        attr.value = "00:00:00"
        return attr
    } ()
    
    private lazy var caloriesAttr: TrackDetailMoreAttr = {
        let attr = TrackDetailMoreAttr()
        attr.title = SRString.Sport.consume.locastr
        attr.unit = SRString.Step.unit_calories.locastr
        attr.value = "0"
        return attr
    } ()
    
    
    private lazy var moreStackView: UIView = {
        let view = UIView()
        [distanceAttr, timeAttr].forEach { v in
            view.translatesAutoresizingMaskIntoConstraints = false
            view.addSubview(v)
        }
        distanceAttr.snp.makeConstraints { (make) in
            make.left.top.bottom.equalToSuperview()
            make.width.equalToSuperview().multipliedBy(0.5)
        }
        timeAttr.snp.makeConstraints { (make) in
            make.right.top.bottom.equalToSuperview()
            make.width.equalToSuperview().multipliedBy(0.5)
        }
        return view
    } ()
    
    
    /// pace statcks
    private lazy var paceAttrFast: TrackDetailPaceAttr = {
        let attr = TrackDetailPaceAttr()
        attr.title = SRString.Sport.fastest.locastr
        return attr
    } ()
    
    private lazy var paceAttrAverage: TrackDetailPaceAttr = {
        let attr = TrackDetailPaceAttr()
        attr.title = SRString.Sport.average.locastr
        return attr
    }()
    
    private lazy var paceAttrSlowest: TrackDetailPaceAttr = {
        let attr = TrackDetailPaceAttr()
        attr.title = SRString.Sport.slowest.locastr
        return attr
    }()
    
    
    private lazy var paceStackView: UIView = {
        let view = UIView()
        [paceAttrFast, paceAttrAverage, paceAttrSlowest].forEach { v in
            view.translatesAutoresizingMaskIntoConstraints = false
            view.addSubview(v)
        }
        paceAttrFast.snp.makeConstraints { (make) in
            make.left.top.bottom.equalToSuperview()
        }
        paceAttrAverage.snp.makeConstraints { (make) in
            make.centerX.top.bottom.equalToSuperview()
        }
        paceAttrSlowest.snp.makeConstraints { (make) in
            make.right.top.bottom.equalToSuperview()
        }
        return view
    } ()
}


// MARK: - DetailAttr

class TrackDetailMoreAttr: UIStackView {
    
    var title: String? {
        get { return titleLabel.text }
        set { titleLabel.text = newValue }
    }
    
    var value: String? {
        get { return valueLabel.text }
        set { valueLabel.text = newValue }
    }
    
    var unit: String? {
        get { return unitLabel.text }
        set { unitLabel.text = newValue }
    }
    
    private lazy var titleLabel: UILabel = {
        let label = UILabel(style: Stylesheet.Label.description);
        label.font = .customFont(ofSize: 14)
        return label;
    }()
    
    private lazy var valueLabel: UILabel = {
        let label = UILabel(style: Stylesheet.Label.description);
        label.textAlignment = .center
        label.textColor = Colors.white
        label.font = .customFont(ofSize: 14)
        label.text = "0"
        return label;
    }()
    
    private lazy var unitLabel: UILabel = {
        let label = UILabel(style: Stylesheet.Label.description);
        label.textAlignment = .center
        label.font = .customFont(ofSize: 14)
        label.text = "--"
        return label;
    }()
    
    // MARK: initialize
    
    override init(frame: CGRect) {
        super.init(frame: frame);
        initialize();
    }
    
    required init(coder: NSCoder) {
        super.init(coder: coder);
        initialize();
    }
    
    fileprivate func initialize() {
        loadSubviews();
    }
    
    fileprivate func loadSubviews() {
        axis = .vertical;
        distribution = .fill
        spacing = 4;
        addArrangedSubview(titleLabel);
        addArrangedSubview(valueLabel);
        addArrangedSubview(unitLabel);
    }
}

class TrackDetailPaceAttr: UIStackView {
    
    private var isImperial: Bool { BandSettings.format.system == .imperial }

    var title: String? {
        didSet { titleLabel.text = title }
    }
    
    var speed: Double = 0 {
        didSet { paceLabel.text = paceText(speed) }
    }
       
    private lazy var titleLabel: UILabel = {
        let label = UILabel(style: Stylesheet.Label.description);
        label.textAlignment = .center
        label.font = .customFont(ofSize: 14)
        label.text = "--"
        return label;
    }()
    
    private lazy var paceLabel: UILabel = {
        let label = UILabel(style: Stylesheet.Label.description);
        label.textColor = UIColor.white;
        label.font = .customFont(ofSize: 14)
        label.text = "00'00\""
        return label;
    }()
    
    
    // MARK: - initialize
    override init(frame: CGRect) {
        super.init(frame: frame);
        initialize();
    }
    
    required init(coder: NSCoder) {
        super.init(coder: coder);
        initialize();
    }
    
    fileprivate func initialize() {
        loadSubviews();
    }
    
    fileprivate func loadSubviews() {
        axis = .vertical;
        distribution = .fill
        spacing = 4;
        addArrangedSubview(titleLabel);
        addArrangedSubview(paceLabel);
    }
    
    /// spped (m/s)
    fileprivate func paceText(_ speedInMps: Double) -> String {
        guard speedInMps > 0 else { return "00'00\"" }
        
        /// 1km 耗时多少秒
        var paceInSeconds = Int(1000 / speedInMps);
        /// 英制处理
        if isImperial {
            /// 1英里 耗时多少秒
            paceInSeconds = Int(1609 / speedInMps)
        }
        /// 分钟
        let paceMins: Int = paceInSeconds / 60
        /// 秒
        let paceSeconds: Int = paceInSeconds % 60
        let text = String(format: "%02d'%02d\"", paceMins, paceSeconds)
        return text
    }
}

fileprivate class GradientLinear: UIView {
    
    public static var kLinearHeight: CGFloat = 12
    
    
    var gradientLayer: CAGradientLayer {
        get { return self.layer as! CAGradientLayer }
    }
    
    // MARK: - Initialize
    override open class var layerClass: AnyClass {
        return CAGradientLayer.self;
    }
    
    override init(frame: CGRect) {
        super.init(frame: frame);
        initialize();
    }
    
    required init?(coder aDecoder: NSCoder) {
        super.init(coder: aDecoder);
    }
    
    func initialize() {
        gradientLayer.locations = [0, 1];
        gradientLayer.startPoint = CGPoint(x: 0, y: 0.5);
        gradientLayer.endPoint = CGPoint(x: 1, y: 0.5);
        gradientLayer.colors = [UIColor.green.cgColor, UIColor.orange.cgColor];
        gradientLayer.cornerRadius = CGFloat(GradientLinear.kLinearHeight / 2);
    }
}
