//
//  SportView.swift
//  CTFit
//
//  Created by Mac on 2020/6/11.
//  Copyright © 2020 shirajo. All rights reserved.
//

import UIKit
import MapKit
import NEUtils

class MapView: MKMapView {
    
    var exercising: Bool = false
    
    /// 当前轨迹覆盖
    var currentSegmentOverlay: MKOverlay?
    
    var tapMapScale: ((_ sender: UIButton) ->Void)? {
        didSet { mapInfo.tapMapScale = tapMapScale }
    }
    
    func setRegion(coordinate: CLLocationCoordinate2D) {
        JZLocationConverter.default.wgs84ToGcj02(coordinate, result: { [weak self] (location) in
            guard let strong = self else { return }
            let regin = MKCoordinateRegion(center: location, span: strong.span)
            strong.setRegion(regin, animated: false)
        })
    }
    
    func startSportTracking() {
        setCenter(false)
    }
    
    func stopSportTracking() {
        setCenter(false)
    }
    
    /// 根据航向信息更新航向箭头
    func updateHeading(_ heading: CLHeading) {
        guard !isResultTrack else { return }
        headingImageView?.isHidden = false
        let rotation = CGFloat(heading.trueHeading/180 * Double.pi)
        headingImageView?.transform = CGAffineTransform(rotationAngle: rotation)
    }
    
    /// 更新轨迹线覆盖
    func updateCurrentSegmentOverlay(_ overlay: MKOverlay?) {
        if self.currentSegmentOverlay != nil {
            self.removeOverlay(self.currentSegmentOverlay!)
        }
        if let hasOverlay = overlay {
            self.addOverlay(hasOverlay);
            self.currentSegmentOverlay = hasOverlay
        }
    }
    
    /// 增加轨迹线覆盖
    func addTrackOverlay(_ overlay: MKOverlay?) {
        if let hasOverlay = overlay {
            self.addOverlay(hasOverlay);
            self.setCenter(hasOverlay.coordinate, animated: false);
        }
    }
    
    /// 移除轨迹线覆盖. 以及位置坐标
    func clear() {
        if let hasOverlay = self.currentSegmentOverlay {
            self.removeOverlay(hasOverlay)
            self.tempCoordinates = nil;
        }
    }
    
    // MARK: - Initialize

    required init?(coder: NSCoder) {
        fatalError("init(coder:) has not been implemented")
    }
    
    init(isResultTrack: Bool = false) {
        super.init(frame: .zero)
        userLightMode()
        mapType = .standard
        if #available(iOS 11.0, *) { mapType = .mutedStandard }
        isZoomEnabled = true
        isRotateEnabled = false
        showsCompass = false
        isPitchEnabled = false
        initializeRegion()
        
        self.isResultTrack = isResultTrack
        if isResultTrack {
            showsUserLocation = false
            userTrackingMode = .none
        } else {
            showsUserLocation = true
            userTrackingMode = .followWithHeading
            initialize()
        }
        delegate = self
    }
    
    override func addSubview(_ view: UIView) {
        super.addSubview(view)
        if "\(view.classForCoder)" == "MKAttributionLabel" {
            let text = "Legal"
            let range = NSRange.init(location: 0, length: text.count)
            let attr = NSMutableAttributedString(string: text)
            attr.addAttribute(.font, value: UIFont.mediumCustomFont(ofSize: 10), range: range)
            attr.addAttribute(.underlineStyle, value: 1, range: range)
            view.setValue(attr, forKey: "_strokeText")
            view.setValue(attr, forKey: "_innerText")
        }
    }
    
    private func initialize()  {
        mapInfo.tapUserLocation = { [weak self] (sender) in
            guard let strong = self else { return }
            if sender.customState == .normal {
                strong.setCenter(true)
                sender.customState = .selected
            } else {
                sender.customState = .normal
            }
        }
    }
    
    
    // MARK: - private
    /// 路径详情
    private var isResultTrack = false
    /// 是否自动移动到用户位置到中心点 （+开启运动）
    private var isFirstAutoCenter: Bool = true
    /// 是手势移动地图
    private var isGestures: Bool = false
    /// 位置坐标
    private var tempCoordinates: [CLLocationCoordinate2D]?
    /// 用于显示在MKMapViewDelegate上初始化的标题的箭头图像
    private var headingImageView: UIImageView!
    /// 地图附加信息
    private lazy var mapInfo: MapInfoView = MapInfoView(self)
    /// 跨度
    private lazy var span: MKCoordinateSpan = { MKCoordinateSpan(latitudeDelta: 0.01, longitudeDelta: 0.01) }()
    
    /// 初始化地区
    private func initializeRegion() {
        isGestures = false
        if let hasLocation =  userLocation.location {
            let regin = MKCoordinateRegion(center: hasLocation.coordinate, span: span)
            setRegion(regin, animated: false)
            return
        }
        let regin = MKCoordinateRegion(center: LocationHelper.latestLocation, span: span)
        setRegion(regin, animated: false)
    }
    
    /// 更新信号显示
    func updateSignal(_ accuracy: Double) {
        guard !isResultTrack else { return }
        mapInfo.updateSignal(accuracy)
    }
    
    private func setCenter(_ animated: Bool){
        guard !isResultTrack else { return }
        if mapInfo.userLocationState == .normal {
            mapInfo.userLocationState = .selected
        }

        isGestures = false
        if let hasLocation =  userLocation.location {
            if LocationHelper.latestLocation !=  hasLocation.coordinate {
                setCenter(hasLocation.coordinate, animated: animated)
            }
        } else {
            setCenter(LocationHelper.latestLocation, animated: animated)
        }
    }
}


// MARK: - MKMapViewDelegate

extension MapView: MKMapViewDelegate {
    
    /// 添加标记
    func mapView(_ mapView: MKMapView, viewFor annotation: MKAnnotation) -> MKAnnotationView? {
        if annotation.isKind(of: MapAnnotationPoint.self) {
            let pointAnnotation = annotation as! MapAnnotationPoint
            let annotationView = MKAnnotationView(annotation: pointAnnotation, reuseIdentifier: "PointAnnotation");
            annotationView.canShowCallout = true;
            annotationView.centerOffset = CGPoint(x: 0, y: -8)
            annotationView.image = pointAnnotation.image
            return annotationView;
        }
        return nil
    }
    
    /// 绘制运动轨迹
    func mapView(_ mapView: MKMapView, rendererFor overlay: MKOverlay) -> MKOverlayRenderer {
        /// 光栅图像块覆盖的数据源
        if overlay.isKind(of: MKTileOverlay.self) {
            return MKTileOverlayRenderer(overlay: overlay)
        }
        /// 轨迹线覆盖
        if overlay is MKPolyline {
            let pr = MKPolylineRenderer(overlay: overlay)
            pr.strokeColor = Colors.blue.withAlphaComponent(0.8)
            pr.lineWidth = 3
            return pr
        }
        return MKOverlayRenderer()
    }
    
    func mapView(_ mapView: MKMapView, didAdd views: [MKAnnotationView]) {
        guard !isResultTrack else { return }
        guard let mapView = mapView as? MapView, mapView.headingImageView==nil else { return }
        for view in views {
            let annotationView = view as MKAnnotationView
            // 为定位位置添加箭头图标，用于指示方向.
            if annotationView.annotation!.isKind(of: MKUserLocation.self) {
                if mapView.headingImageView == nil {
                   let image = ImageRepo.Sport.location_arrow
                    mapView.headingImageView = UIImageView(image: image)
                    mapView.headingImageView!.frame = CGRect(x: (annotationView.frame.size.width - image.size.width)/2, y: (annotationView.frame.size.height - image.size.height)/2, width: image.size.width, height: image.size.height)
                    annotationView.insertSubview(mapView.headingImageView!, at: 0)
                    mapView.headingImageView?.isHidden = false
                    break
                }
            }
        }
    }
    
    func mapView(_ mapView: MKMapView, didUpdate userLocation: MKUserLocation) {
        guard !isResultTrack else { return }
        if let hasLocation = userLocation.location {
            if !exercising { updateSignal(hasLocation.horizontalAccuracy) }
            if mapInfo.userLocationState == .selected || isFirstAutoCenter {
                setCenter(isFirstAutoCenter ? false : true )
                isFirstAutoCenter = false
            }
            /// 地图返回的 userLocation 不需要判断中国区域转换
            LocationHelper.latestLocation = hasLocation.coordinate
        }
    }
    
    func mapView(_ mapView: MKMapView, regionWillChangeAnimated animated: Bool) {
        guard !isResultTrack else { return }
        if isGestures { mapInfo.userLocationState = .normal }
        isGestures = true
    }
    
    override func touchesBegan(_ touches: Set<UITouch>, with event: UIEvent?) {
        guard !isResultTrack else { return }
        if mapInfo.userLocationState == .selected {
            mapInfo.userLocationState = .normal
        }
    }

}

// MARK: - MapAnnotationPoint

class MapAnnotationPoint: NSObject, MKAnnotation {
    
    fileprivate var _coordinate: CLLocationCoordinate2D
    
    var title: String?
    
    var subtitle: String?
    
    var image: UIImage?
    
    init(lat: Double, lon: Double) {
        _coordinate = CLLocationCoordinate2D(latitude: lat, longitude: lon);
        super.init();
    }
    
    var coordinate: CLLocationCoordinate2D {
        get { return _coordinate; }
        set { _coordinate = newValue; }
    }
}

 
// MARK: - MapInfoView

import RxSwift
import RxCocoa

class MapInfoView {
    
    var tapUserLocation: ((_ sender: CustomButton) -> Void)?
    
    var tapMapScale: ((_ sender: UIButton) -> Void)?
    
    var userLocationState: CustomButton.CustomState {
        get { userLocationButton.customState }
        set { userLocationButton.customState = newValue}
    }

    /// 更新信号图像
    /// - signal:  gps信号在范围内计数 [0-6]
    func updateSignal(_ accuracy: Double) {
        if accuracy < 0 {
            self.signalView.isHidden = true
        } else {
            self.signalView.isHidden = false
            self.signalView.image = GpsSingal.imageParse(accuracy)
        }
    }
    
    init(_ mapView: MKMapView) {
        addSubviews(mapView);
        setupConstraints(mapView);
    }
    
    // MARK: - private
    
    private let disposeBag = DisposeBag()
    
    /// gps信号
    private lazy var signalView: UIImageView = {
        let view = UIImageView()
        view.isHidden = true
        return view
    } ()
    
    /// 将用户位置移动到地图中心
    private var userLocationButton: CustomButton = {
        let button = CustomButton();
        button.setCustomImage(ImageRepo.Sport.location_user_normal, for: .normal)
        button.setCustomImage(ImageRepo.Sport.location_user_highlight, for: .selected)
        return button
    } ()
    
    /// 展开或收缩 MapView Frame 隐藏头部/底部视图
    private lazy var scaleButton: UIButton = {
        let button = UIButton();
        button.setImage(ImageRepo.Sport.map_scale, for: .normal);
        return button
    } ()
    
    
    private func addSubviews(_ mapView: MKMapView) {
        [signalView, scaleButton, userLocationButton].forEach { view in
            view.translatesAutoresizingMaskIntoConstraints = false
            mapView.addSubview(view)
        }
        userLocationButton.rx.tap.subscribe(onNext: { [weak self] in
            guard let strong = self else { return }
            guard !LocationHelper.showNoLocationPermissionAlert() else { return }
            guard let closure = strong.tapUserLocation else { return }
            closure(strong.userLocationButton)
        }).disposed(by: disposeBag)
        
        scaleButton.rx.tap.subscribe(onNext: { [weak self] in
            guard let strong = self else { return }
            if let closure = strong.tapMapScale { closure(strong.scaleButton) }
        }).disposed(by: disposeBag)
    }
    
    private func setupConstraints(_ mapView: MKMapView) {
        self.signalView.snp.makeConstraints { (make) in
            make.leading.top.equalToSuperview().offset(8);
        }
        
        self.scaleButton.snp.makeConstraints { (make) in
            make.trailing.bottom.equalToSuperview().offset(-12);
        }
        
        self.userLocationButton.snp.makeConstraints { (make) in
            make.leading.equalToSuperview().offset(8)
            make.bottom.equalToSuperview().offset(-20)
            make.width.height.equalTo(44)
        }
    }

}


// MARK: - GpsSingal

/// 信号精度的上限阈值(以米为单位)。
enum GpsSingal {
    
    case step1, step2, step3, step4, step5, step6
    
    private var value: Double {
        switch self {
        case .step1: return 6.0
        case .step2: return 11.0
        case .step3: return 31.0
        case .step4: return 51.0
        case .step5: return 101.0
        case .step6: return 201.0
        }
    }
    
    static let collection: [GpsSingal] = [.step1, .step2, .step3, .step4, .step5, .step6]
    
    static func indexParse(_ accuracy: Double) -> Int {
        for (idx, element) in collection.reversed().enumerated() {
            if accuracy > element.value { return idx }
        }
        return collection.count
    }
    
    static func imageParse(_ accuracy: Double) -> UIImage {
        let images: [UIImage] = [ImageRepo.Sport.agps_signal6, ImageRepo.Sport.agps_signal5, ImageRepo.Sport.agps_signal4, ImageRepo.Sport.agps_signal3, ImageRepo.Sport.agps_signal2, ImageRepo.Sport.agps_signal1, ImageRepo.Sport.agps_signal0]
        for (idx, element) in collection.enumerated() {
            if accuracy <= element.value { return images[idx] }
        }
        return images.last!
    }
}
