//
//  SportBottomView.swift
//  CTFit
//
//  Created by Mac on 2020/6/11.
//  Copyright © 2020 shirajo. All rights reserved.
//

import UIKit
import RxSwift
import RxCocoa
import Toast

class SportBottomView: UIView {
    
    // MARK: - Initialize
    @available(*, unavailable)
    required init?(coder aDecoder: NSCoder) {
        fatalError("init(coder:) has not been implemented")
    }
    
    init() {
        super.init(frame: .zero)
        backgroundColor = Colors.Base.background
        addSubviews()
        clear()
        observers()
    }
    
    private func addSubviews() {
        addLongPressGesture()

        [backgroundImageView, historyButton, distanceAttr, actionButton, voiceButton, timeAttr, speedAttr].forEach { view in
            view.translatesAutoresizingMaskIntoConstraints = false
            addSubview(view)
        }

        backgroundImageView.snp.makeConstraints { (make) in
            make.edges.equalToSuperview().inset(UIEdgeInsets(top: 0, left: 0, bottom: 0, right: 0))
        }
        
        historyButton.snp.makeConstraints { (make) in
            make.top.equalToSuperview().offset(Constraints.small)
            make.left.equalToSuperview().offset(Constraints.small)
            make.height.greaterThanOrEqualTo(44)
            make.width.lessThanOrEqualTo(120)
        }
        
        distanceAttr.snp.makeConstraints { (make) in
            make.top.equalToSuperview()
            make.centerX.equalToSuperview()
        }
        
        actionButton.snp.makeConstraints { (make) in
            make.centerX.equalToSuperview()
            make.width.height.equalTo(58)
        }
        
        voiceButton.snp.makeConstraints { (make) in
            make.left.equalTo(actionButton.snp.right).offset(Constraints.small)
            make.centerY.equalTo(actionButton.snp.centerY)
            make.width.height.equalTo(44)
        }

        timeAttr.snp.makeConstraints { (make) in
            make.left.equalToSuperview().offset(Constraints.small);
            make.centerY.equalTo(actionButton.snp.centerY)
        }

        speedAttr.snp.makeConstraints { (make) in
            make.right.equalToSuperview().offset(-Constraints.small)
            make.centerY.equalTo(actionButton.snp.centerY)
            make.bottom.equalToSuperview().offset(-Constraints.large*2)
        }
    }

    // MARK: - View customer
    private let disposeBag = DisposeBag()

    private lazy var backgroundImageView: UIImageView = {
        let view = UIImageView(image: ImageRepo.Sport.bg_control)
        view.contentMode = .scaleAspectFill
        return view
    } ()
    
    private lazy var distanceAttr: SportBottomHeaderAttr = {
        let attr = SportBottomHeaderAttr()
        attr.title = SRString.Sport.distance.locastr
        attr.value = "0"
        attr.unit = SRString.Unit.m.locastr
        return attr
    } ()
    
    private lazy var timeAttr: SportBottomDetailAttr = {
        let attr = SportBottomDetailAttr()
        attr.title = SRString.Sport.time.locastr
        attr.value = "00:00:00"
        attr.unit = SRString.Sport.unit_time.locastr
        return attr
    } ()
    
    private lazy var speedAttr: SportBottomDetailAttr = {
        let attr = SportBottomDetailAttr()
        attr.title = SRString.Sport.speed.locastr
        attr.value = "0"
        attr.unit = SRString.Unit.km.locastr + "/\(SRString.Unit.hours.locastr)"
        return attr
    } ()
    

    private lazy var voiceButton: CustomButton = {
        let button = CustomButton()
        button.imageView?.contentMode = .scaleAspectFit
        button.imageEdgeInsets = UIEdgeInsets(top: 0, left: 8, bottom: 0, right: 12)
        button.setCustomImage(ImageRepo.Sport.voice_enable, for: .normal)
        button.setCustomImage(ImageRepo.Sport.voice_disable, for: .selected)
        button.customState = SportBottomView.voiceEnable ? .normal : .selected

        button.rx.tap.subscribe(onNext: { [weak self]_ in
            guard let strong = self else { return }
            strong.updateVoiceEnableState()
        }).disposed(by: disposeBag)
        
        return button
    } ()
    
    private lazy var historyButton: UIButton = {
        let button = UIButton(type: .custom)
        button.setTitle(SRString.Sport.sport_history.locastr, for: .normal)
        button.titleLabel?.font = .customFont(ofSize: 16)
        button.titleLabel?.numberOfLines = 0
        button.rx.tap.subscribe(onNext: { [weak self]_ in
            guard let strong = self else { return }
            strong.toHistory()
        }).disposed(by: disposeBag)
        
        return button
    } ()
    
    
    private lazy var actionButton: CustomButton = {
        let button = CustomButton()
        button.setCustomImage(ImageRepo.Sport.start, for: .normal)
        button.setCustomImage(ImageRepo.Sport.stop, for: .selected)
        button.customState = .normal
        
        button.rx.tap.subscribe(onNext: { [weak self]_ in
            guard let strong = self else { return }
            if strong.actionButton.customState == .normal {
                if !LocationHelper.showNoLocationPermissionAlert() {
                    strong.sportMenu.show()
                }
            } else {
                ToastUtils.showToastCCV(normal: SRString.Sport.longpress_sport_end.locastr)
            }
        }).disposed(by: disposeBag)
        return button
    } ()
    
    
    
    private lazy var sportMenu: SportMenu = {
        let menu = SportMenu(refView: actionButton)
        menu.didSelectSport = { [weak self] type in
            guard let strong = self, let closure = strong.tapAction else { return }
            strong.sportType = type
            closure(strong)
            strong.actionState = .selected

        }
        return menu
    } ()
    
    private lazy var countdownView: CountdownView = {
        let view = CountdownView();
        if let hasSuperView = self.superview {
            hasSuperView.addSubview(view);
            view.snp.makeConstraints({ (make) in
                make.width.height.equalTo(96)
                make.center.equalToSuperview()
            })
        }
        return view
    }()
    
    // MARK: - Fields
    
    private var _distane: Double = 0
    private var _speed: Double = 0
    private var isImperial: Bool { BandSettings.isImperial }
    private var distanceInterval: Double {
        var interval = 1000.0
        if isImperial { interval = BandCalculator.Distance.Mile.m }
        return interval
    }
    
    typealias SportContrlClosure = (SportBottomView) -> Void
    var tapAction: SportContrlClosure?
        
    var sportType: BDSportType = .normal
    var actionState: CustomButton.CustomState {
        get { actionButton.customState }
        set { actionButton.customState = newValue}
    }
    
    /// (m)
    var distance: Double = 0 {
        didSet {
            if isImperial {
                let distanceMile = BandCalculator.mile(km: distance / 1000.0)
                if distance < distanceInterval {
                    distanceAttr.value = String(format: "%.0f", distanceMile * BandCalculator.Distance.Mile.foot)
                    distanceAttr.unit = SRString.Unit.foot.locastr
                } else {
                    distanceAttr.value = String(format: "%.2f", distanceMile)
                    distanceAttr.unit = SRString.Unit.mile.locastr
                }
            } else {
                if distance < distanceInterval {
                    distanceAttr.value = String(format: "%.0f", distance)
                    distanceAttr.unit = SRString.Unit.m.locastr
                } else {
                    distanceAttr.value = String(format: "%.2f",  distance / 1000.0)
                    distanceAttr.unit = SRString.Unit.km.locastr
                }
            }
            _distane = distance
        }
    }
    
    /// (m/s)
    var speed: Double = 0 {
        didSet {
            if isImperial {
                speedAttr.value = String(format: "%.2f", BandCalculator.mile(km: speed * 3.6))
                speedAttr.unit = SRString.Unit.mile_ph.locastr
            } else {
                speedAttr.value = String(format: "%.2f", speed * 3.6)
                speedAttr.unit = SRString.Unit.km_ph.locastr
            }
            _speed = speed
        }
    }
    
    /// (s)
    var duration: Int = 0 {
        didSet {
            let hours   = duration / 3600
            let remain  = duration - hours*3600
            let minutes = remain / 60
            let seconds = remain % 60
            timeAttr.value = String(format: "%02d:%02d:%02d", hours, minutes, seconds)
        }
    }
    
}

extension SportBottomView {
    func clear() {
        distance = 0
        duration = 0
        speed = 0
    }
    func historyButton(hidden: Bool) {
        historyButton.isHidden = hidden
    }
}

extension SportBottomView {

    private func update() {
        distance = _distane
        speed = _speed
    }
    
    private func updateVoiceEnableState() {
        SportBottomView.voiceEnable = !SportBottomView.voiceEnable
        voiceButton.customState = SportBottomView.voiceEnable ? .normal : .selected
    }
    
    private func observers() {
        BandSettings.observable.subscribe(onNext: { [weak self] (options) in
            guard let strong = self else { return }
            guard options.contains(.system) else { return }
            strong.update()
        }).disposed(by: disposeBag)
    }
    
    private func toHistory(){
        let vc = SportHistoryViewController()
        Helper.currentVc?.navigationController?.pushViewController(vc, animated: true)
    }
}

// MARK: - voiceEnable

extension SportBottomView {
    
    private static let kVoiceEnableKey = "kVoiceEnableKey"
    private static var _voiceEnable: Bool? {
        didSet {
            guard _voiceEnable != oldValue else { return }
            UserDefaults.standard.setValue(_voiceEnable, forKey: kVoiceEnableKey)
            UserDefaults.standard.synchronize()
        }
    }
    static var voiceEnable: Bool {
        get {
            if _voiceEnable != nil { return _voiceEnable! }
            if  let _ = UserDefaults.standard.value(forKey: kVoiceEnableKey) {
                _voiceEnable = UserDefaults.standard.bool(forKey: kVoiceEnableKey)
            } else {
                _voiceEnable = true
            }
            return _voiceEnable!
        }
        set {
            _voiceEnable = newValue
        }
    }
}

// MARK: - 长按结束运动手势

extension SportBottomView {

    private func addLongPressGesture(){
        let longPressGesture = UILongPressGestureRecognizer(target: self, action: #selector(actionButtonLongPressAction(_:)));
        actionButton.addGestureRecognizer(longPressGesture)
    }
    
    @objc
    private func actionButtonLongPressAction(_ target: UILongPressGestureRecognizer) {
        guard actionState == .selected else { return  }
        self.superview?.hideAllToasts()
        if target.state == .began {
            self.showCountdownView()
        } else if (target.state == .cancelled
            || target.state == .failed
            || target.state == .ended) {
            self.dismissCountdownView()
        }
    }
    
    ///  App 长按结束Gps
    private func showCountdownView() {
        self.countdownView.isHidden = false
        self.countdownView.countdown(withDuration: 2) { [weak self] (completed) in
            guard let strong = self, let closure = strong.tapAction else { return }
            strong.dismissCountdownView()
            closure(strong)
            strong.actionState = .normal
        }
    }
    
    private func dismissCountdownView() {
        self.countdownView.stopCountdown()
        self.countdownView.isHidden = true
    }
}


// MARK: - HeaderAttr

class SportBottomHeaderAttr: UIControl {
    
    var title: String? {
        get { return titleLabel.text }
        set { titleLabel.text = newValue }
    }
    
    var value: String? {
        get { return valueLabel.text }
        set { valueLabel.text = newValue }
    }
    
    var unit: String? {
        get { return unitLabel.text }
        set { unitLabel.text = newValue }
    }

    // MARK: - Initialize
    
    @available(*, unavailable)
    required init?(coder: NSCoder) {
        fatalError("init(coder:) has not been implemented")
    }
       
    init() {
        super.init(frame: .zero)
        addSubviews()
    }
    
    /// View customer
    
    private func addSubviews() {
        [titleLabel, valueLabel, unitLabel].forEach { view in
            view.translatesAutoresizingMaskIntoConstraints = false
            addSubview(view)
        }
        titleLabel.snp.makeConstraints { (make) in
            make.top.centerX.equalToSuperview()
            make.height.equalTo(44)
        }
        
        valueLabel.snp.makeConstraints { (make) in
            make.top.equalTo(titleLabel.snp.bottom).offset(-Constraints.normal)
            make.centerX.equalToSuperview()
            make.bottom.equalToSuperview()
        }
        
        unitLabel.snp.makeConstraints { (make) in
            make.bottom.equalTo(valueLabel.snp.bottom).offset(-Constraints.small)
            make.left.equalTo(valueLabel.snp.right).offset(Constraints.small/2)
        }
    }
    
    private lazy var titleLabel: UILabel = {
        let label = UILabel(style: Stylesheet.Label.text)
        label.font = .customFont(ofSize: 20)
        label.textColor = Colors.lightGray
        label.text = "Current"
        return label
    }()
    
    private lazy var valueLabel: UILabel = {
        let label = UILabel(style: Stylesheet.Label.text)
        label.font = .boldSystemFont(ofSize: 50)
        label.textColor = Colors.white
        label.text = "0"
        return label
    }()
    
    private lazy var unitLabel: UILabel = {
        let label = UILabel(style: Stylesheet.Label.text)
        label.textColor = Colors.lightGray
        label.font = .customFont(ofSize: 17)
        label.text = "unit"
        return label
    }()
}


// MARK: - DetailAttr

class SportBottomDetailAttr: UIStackView {
    
    var title: String? {
        get { return titleLabel.text }
        set { titleLabel.text = newValue }
    }
    
    var value: String? {
        get { return valueLabel.text }
        set { valueLabel.text = newValue }
    }
    
    var unit: String? {
        get { return unitLabel.text }
        set { unitLabel.text = newValue }
    }
    
    
    // MARK: - initialize
    
    override init(frame: CGRect) {
        super.init(frame: frame);
        loadSubviews();
    }
    
    required init(coder: NSCoder) {
        super.init(coder: coder);
        loadSubviews();
    }
    
    /// View customer

    private func loadSubviews() {
        axis = .vertical;
        distribution = .fillEqually
        spacing = 0;
        addArrangedSubview(titleLabel);
        addArrangedSubview(valueLabel);
        addArrangedSubview(unitLabel);

    }
    
    private lazy var titleLabel: UILabel = {
        let label = UILabel(style: Stylesheet.Label.text);
        label.textColor = Colors.lightGray
        label.text = "Label"
        return label;
    }()
    
    private lazy var valueLabel: UILabel = {
        let label = UILabel(style: Stylesheet.Label.text);
        label.font = .customFont(ofSize: 18)
        label.textColor = Colors.white
        label.text = "Value"
        return label;
    }()
    
    private lazy var unitLabel: UILabel = {
        let label = UILabel(style: Stylesheet.Label.text);
        label.textColor = Colors.lightGray
        label.text = "unit"
        return label;
    }()
}

// MARK: - CustomButton

class CustomButton: UIButton {
    
    enum CustomState {
        case normal, selected
    }
    
    var customState: CustomState = .normal {
        didSet {
            switch customState {
            case .normal: setImage(customNormalImage, for: .normal)
            case .selected: setImage(customSelectedImage, for: .normal)
            }
        }
    }
    
    func setCustomImage(_ image: UIImage?, for state: CustomState) {
        if state == customState { setImage(image, for: .normal) }
        switch state {
        case .normal: customNormalImage = image
        case .selected: customSelectedImage = image
        }
    }
    
    private var customNormalImage: UIImage?
    private var customSelectedImage: UIImage?
}


// MARK: - 长按结束运动视图 以及 动画

class CountdownView: UIView {
    
    /// 动画结束监听
    typealias CountdownCompletion = (Bool) -> Void
    
    static let kWidth = 64
    static let kHeight = 64
    
    private let kPI_2 = Double.pi / 2;
    private let kPi: CGFloat = CGFloat(Float.pi);
    private let kAngleStart: CGFloat = CGFloat( -Float.pi / 2);
    private let kLineWidth: CGFloat = 4
    
    private var textLabel: UILabel!
    private var progressLayer: CAShapeLayer!
    private var timer: DispatchSourceTimer?
    private var remainSeconds: Int = 0
    
    override init(frame: CGRect) {
        super.init(frame: frame)
        self.initialize()
    }
    
    required init?(coder aDecoder: NSCoder) {
        super.init(coder: aDecoder)
        self.initialize()
    }
    
    private func initialize() {
        self.backgroundColor = UIColor(red: 0.0, green: 0.0, blue: 0.0, alpha: 0.5)
        self.borderCorner = 8
        
        // Add textLabel
        self.textLabel = UILabel()
        self.textLabel.textColor = UIColor.white
        self.textLabel.font = UIFont.systemFont(ofSize: 17)
        self.addSubview(self.textLabel)
        self.textLabel.snp.makeConstraints { (make) in
            make.center.equalToSuperview()
        }
        
        self.progressLayer = CAShapeLayer();
        self.progressLayer.lineWidth = kLineWidth
        self.progressLayer.lineCap = CAShapeLayerLineCap.round
        self.progressLayer.fillColor = UIColor.clear.cgColor
        self.progressLayer.strokeColor = UIColor.white.cgColor
        self.layer.addSublayer(self.progressLayer)
        
    }
    
    private func updateRemainTime() {
        self.textLabel.text = String(self.remainSeconds) + " " + SRString.Unit.seconds.locastr
    }

    // Only override draw() if you perform custom drawing.
    // An empty implementation adversely affects performance during animation.
    override func draw(_ rect: CGRect) {
        // Draw progress layer
        let width = self.frame.size.width;
        let height = self.frame.size.height;
        let center = CGPoint(x: width / 2, y: height / 2 );
        let radius = (width / 2 - kLineWidth / 2);
        let progressArcPath = UIBezierPath(arcCenter: center,
                                           radius: radius,
                                           startAngle: kAngleStart,
                                           endAngle: kAngleStart + kPi * 2,
                                           clockwise: true);
        self.progressLayer.path = progressArcPath.cgPath;
        
        super.draw(rect)
    }
    
    func countdown(withDuration duration: Int, completion: @escaping CountdownCompletion) {
        self.remainSeconds = duration;
        
        let toProgress: CGFloat = 1
        CATransaction.begin();
        let animation = CABasicAnimation(keyPath: "strokeEnd");
        animation.isRemovedOnCompletion = false;
        animation.fillMode = CAMediaTimingFillMode.forwards;
        animation.duration = CFTimeInterval(duration);
        animation.fromValue = 0;
        animation.toValue = 1;
        self.progressLayer.add(animation, forKey: "progress");
        self.progressLayer.strokeEnd = toProgress;
        CATransaction.commit();
        
        self.timer =  DispatchSource.makeTimerSource(flags: [], queue: DispatchQueue.main)
        self.timer?.schedule(deadline: .now(), repeating: .seconds(1), leeway: .microseconds(10))
        self.timer?.setEventHandler(handler: { [self] in
            if self.remainSeconds <= 0 {
                self.timer?.cancel()
                completion(true)
            } else {
                self.remainSeconds -= 1
                self.updateRemainTime()
            }
        })
        self.timer?.resume()
    }
    
    func stopCountdown() {
        guard let hasTimer = self.timer else {
            return
        }
        hasTimer.cancel()
        self.timer = nil
    }
}

