//
//  SportMenu.swift
//  CTFit
//
//  Created by Mac on 2020/6/16.
//  Copyright © 2020 shirajo. All rights reserved.
//

import UIKit
import Foundation
import TableViewManager
import RxSwift
import RxCocoa

class SportMenu: UIView {
    
    typealias SelectSportClosure = (BDSportType) -> Void
    
    var didSelectSport: SelectSportClosure?

    // MARK: - Feild
    /// 圆角
    var cornerRaius: CGFloat = 10 { didSet { layer.mask = drawMaskLayer() } }
    /// 菜单颜色
    var menuColor: UIColor = Colors.Base.background { didSet { backgroundColor = menuColor} }
    /// 是否显示阴影 (也可以自己通过layer属性设置)
    var showShadow: Bool = true {
        didSet {
            guard showShadow != oldValue else { return }
            if showShadow {
                self.layer.shadowOpacity = 0.0;
                self.layer.shadowOffset = CGSize(width: 0, height: 0);
                self.layer.shadowRadius = 0.0;
            } else {
                self.setDefaultShadow()
            }
        }
    }
    
    /// 菜单单元格高度
    private let menuCellHeight: CGFloat = 44
    /// 最大显示数量
    private let maxDisplayCount: Int = 5
    /// 选择菜单选项后消失
    private let dismissOnselected: Bool = true
    /// 点击菜单外消失
    private let dismissOnTouchOutside: Bool = true
    /// 设置偏移距离 default = 0（与触摸点在Y轴上的偏移）
    private let offsetY: CGFloat = 10

    
    private let menuWidth: CGFloat = UIDevice.scrWidth * 0.7
    private let kMainWindow = UIApplication.shared.keyWindow!
    private let kHeaderHeight: CGFloat = 64
    private let kArrowWidth: CGFloat = 15
    private let kArrowHeight: CGFloat = 0
    private let kDefaultMargin: CGFloat = 10
    private let kAnimationTime: Double = 0.25
    
    private var refPoint: CGPoint = CGPoint(x: UIDevice.scrWidth/2, y: 88)
    var refView: UIView?
    private var arrowPosition: CGFloat = 0
    private var topMargin: CGFloat = 0
    private var isReverse: Bool = false
    
    private lazy var tabviewHeight: CGFloat = {
        let number = (datasource.count > maxDisplayCount) ? maxDisplayCount : datasource.count
        return (kHeaderHeight +  CGFloat(number) * menuCellHeight + kArrowHeight)
    }()
    
    
    // MARK: - Initialize
    @available(*, unavailable)
    required init?(coder aDecoder: NSCoder) {
        fatalError("init(coder:) has not been implemented")
    }
    
    init(refView: UIView? = nil) {
        super.init(frame: .zero)
        // userLightMode()
        backgroundColor = menuColor
        layer.masksToBounds = true
        self.refView = refView
        calculateArrowAndFrame()
        setupMaskLayer()
        addSubviews()
        setupConstraints()
        
        datasource.forEach { (item) in
            item.selectedAction = { [weak self] e in
                guard let strong = self,
                    let closure = strong.didSelectSport else { return }
                strong.dismiss()
                closure(e.type)
            }
        }
    }
    
    
    private func addSubviews() {
        [tableView].forEach { view in
            view.translatesAutoresizingMaskIntoConstraints = false
            addSubview(view)
        }
        _ = tableViewManger
    }

    private func setupConstraints() {
        tableView.snp.makeConstraints { (make) in
            make.left.top.right.equalToSuperview()
            make.height.equalTo(tabviewHeight)
        }
    }
    
    // MARK:  View customer

    lazy var bgView: UIView = {
        let view = UIView(frame: UIDevice.scrBounds)
        view.backgroundColor = UIColor.black.withAlphaComponent(0.35)
        view.alpha = 0
        let tap = UITapGestureRecognizer(target: self, action: #selector(dismiss))
        view.addGestureRecognizer(tap)
        return view
    }()
    
    lazy var titleLabel: UILabel = {
        let label = UILabel(frame: .zero)
        label.font = .customFont(ofSize: 18)
        label.textColor = Colors.Base.title
        label.text = SRString.Sport.sport_mode.locastr
        return label
    }()
    
    lazy var tableView: UITableView = {
        let tableView = UITableView(frame: .zero, style: .plain)
        tableView.rowHeight = menuCellHeight
        tableView.estimatedRowHeight = menuCellHeight
        tableView.estimatedSectionHeaderHeight = 0;
        tableView.estimatedSectionFooterHeight = 0;
        tableView.tintColor = Colors.Cell.tint
        tableView.backgroundColor = .clear
        tableView.clipsToBounds = true
        tableView.separatorStyle = .none
        return tableView
    }()
    
    lazy var datasource: [SportMenuCellItem] = {
        let last = SportMenuCellItem(type: .bike)
        last.showSeparator = false
        return [SportMenuCellItem(type: .run),
                SportMenuCellItem(type: .walk),
                SportMenuCellItem(type: .climbing),
                last]
    }()
    
    lazy var headerView: SportMenuHeaderView = {
        let view = SportMenuHeaderView(frame: CGRect(x: 0, y: 0, width: menuWidth, height: kHeaderHeight))
        view.backgroundColor = Colors.Base.background
        view.addSubview(titleLabel)
        tableView.bounces = datasource.count > maxDisplayCount ? true : false
        titleLabel.snp.makeConstraints { (make) in
            make.left.equalToSuperview().offset(Constraints.normal)
            make.centerY.equalToSuperview().offset(8)
        }
        return view
    }()
    
    lazy var section: ZJTableViewSection = {
        let _section = ZJTableViewSection(headerView: headerView)
        // _section.footerHeight = 0.01
        datasource.forEach { (e) in _section.add(item: e) }

        return _section;
    }()
    
    lazy var tableViewManger: ZJTableViewManager = {
        let manager = ZJTableViewManager(tableView: tableView)
        manager.register(SportMenuCell.self, SportMenuCellItem.self)
        manager.add(section: section);
        return manager
    } ()
        
    
    private func setDefaultShadow() {
        self.layer.shadowOpacity = 0.5;
        self.layer.shadowOffset = CGSize(width: 0, height: 0);
        self.layer.shadowRadius = 5.0;
    }
}

extension SportMenu {
    
    private func getRefPoint() -> CGPoint? {
        guard let view = refView else { return nil }
        let absoluteRect = view.convert( view.bounds, to: kMainWindow)
        var refPoint: CGPoint? = nil
        if absoluteRect.maxY + tabviewHeight > UIDevice.scrHeight - kArrowHeight {
            refPoint = CGPoint(x: absoluteRect.origin.x + absoluteRect.size.width / 2, y: absoluteRect.origin.y)
            isReverse = true
        } else {
            refPoint = CGPoint(x: absoluteRect.origin.x + absoluteRect.size.width / 2, y: absoluteRect.origin.y + absoluteRect.size.height)
            isReverse = true
        }
        return refPoint
    }
    
    private func calculateArrowAndFrame() {
        if let point = getRefPoint() { refPoint = point }
        let width: CGFloat = menuWidth
        let height: CGFloat = tabviewHeight + kArrowHeight
        var originX: CGFloat = 0
        var originY: CGFloat = 0
        /// 默认在中间
        arrowPosition = 0.5 * width - 0.5 * kArrowWidth
        // 设置出menu的x和y（默认情况）
        originX = refPoint.x - 0.5 * width;
        originY = refPoint.y;
        
        /// 考虑向左右展示不全的情况，需要反向展示
        if (originX + width > UIDevice.scrWidth - 10) {
            originX = UIDevice.scrWidth - kDefaultMargin - width;
        }
        /// 向上的情况间距也至少是kDefaultMargin
        else if (originX < 10) {
            originX = kDefaultMargin;
        }
        
        /// 设置三角形的起始点
        if ((refPoint.x <= originX + width - cornerRaius) && (refPoint.x >= originX + cornerRaius)) {
            arrowPosition = refPoint.x - originX - 0.5 * kArrowWidth;
        }else if (refPoint.x < originX + cornerRaius) {
            arrowPosition = cornerRaius;
        }else {
            arrowPosition = width - cornerRaius - kArrowWidth;
        }
        
        /// 如果不是根据关联视图，得算一次是否反向
        if (refView == nil) {
            isReverse = (originY + height > UIDevice.scrHeight - kDefaultMargin) ? true : false
        }
        
        var anchorPoint: CGPoint
        if (isReverse) {
            originY = refPoint.y - height;
            anchorPoint = CGPoint(x: abs(arrowPosition) / width, y: 1);
            topMargin = 0;
        }else{
            anchorPoint = CGPoint(x: abs(arrowPosition) / width, y: 0);
            topMargin = kArrowHeight;
        }
        originY += originY >= refPoint.y ? offsetY : -offsetY;
        
        /// 保存原来的 frame，防止设置锚点后偏移
        self.layer.anchorPoint = anchorPoint;
        self.frame = CGRect(x: originX, y: originY, width: width, height: height);
    }
    
    private func drawMaskLayer() -> CAShapeLayer{
        let maskLayer = CAShapeLayer()
        let bottomMargin = !isReverse ? 0 : kArrowHeight
        
        /// 定出四个转角点
        let topRightArcCenter: CGPoint = CGPoint(x: self.width - cornerRaius, y: topMargin + cornerRaius);
        let topLeftArcCenter: CGPoint = CGPoint(x: cornerRaius, y: topMargin + cornerRaius);
        let bottomRightArcCenter: CGPoint = CGPoint(x: self.width - cornerRaius, y: self.height - bottomMargin - cornerRaius);
        let bottomLeftArcCenter: CGPoint = CGPoint(x: cornerRaius, y: self.height - bottomMargin - cornerRaius);
       
        let path = UIBezierPath()
        path.move(to: CGPoint(x: 0, y: topMargin + cornerRaius))
        path.addLine(to: CGPoint(x: 0, y: bottomLeftArcCenter.y))
        path.addArc(withCenter: bottomLeftArcCenter, radius: cornerRaius, startAngle: -.pi, endAngle: -.pi * 1.5, clockwise: false)
        if isReverse {
            path.addLine(to: CGPoint(x: arrowPosition, y: self.height - kArrowHeight))
            path.addLine(to: CGPoint(x: arrowPosition + 0.5*kArrowWidth, y: self.height))
            path.addLine(to: CGPoint(x: arrowPosition + kArrowWidth, y: self.height - kArrowHeight))
        }
        path.addLine(to: CGPoint(x: self.width - cornerRaius, y: self.height - bottomMargin))
        path.addArc(withCenter: bottomRightArcCenter, radius: cornerRaius, startAngle: -.pi * 1.5, endAngle: -.pi * 2, clockwise: false)
        path.addLine(to: CGPoint(x: self.width, y: self.height - bottomMargin + cornerRaius))
        path.addArc(withCenter: topRightArcCenter, radius: cornerRaius, startAngle: 0, endAngle: -.pi * 0.5, clockwise: false)
        if !isReverse {
            path.addLine(to: CGPoint(x: arrowPosition + kArrowWidth, y: topMargin))
            path.addLine(to: CGPoint(x: arrowPosition + 0.5*kArrowWidth, y: 0))
            path.addLine(to: CGPoint(x: arrowPosition + kArrowWidth, y: topMargin))
        }
        path.addLine(to: CGPoint(x: cornerRaius, y: topMargin))
        path.addArc(withCenter: topLeftArcCenter, radius: cornerRaius, startAngle: -.pi * 0.5, endAngle: -.pi, clockwise: false)
        path.close()
        
        maskLayer.path = path.cgPath
        return maskLayer
    }
    
    private func setupMaskLayer() {
        self.layer.mask = drawMaskLayer()
    }
    
    @objc func show() {
        kMainWindow.addSubview(self.bgView)
        kMainWindow.addSubview(self)
        self.layer.setAffineTransform(CGAffineTransform(scaleX: 0.1, y: 0.1))
        UIView.animate(withDuration: kAnimationTime) {
            self.layer.setAffineTransform(CGAffineTransform(scaleX: 1, y: 1))
            self.alpha = 1.0
            self.bgView.alpha = 1.0
        }
    }
    
    @objc func dismiss() {
        UIView.animate(withDuration: kAnimationTime, animations: {
            self.layer.setAffineTransform(CGAffineTransform(scaleX: 0.1, y: 0.1))
            self.alpha = 0.0
            self.bgView.alpha = 0.0
        }) { (_) in
            self.removeFromSuperview()
            self.bgView.removeFromSuperview()
        }
    }
}



// MARK: - SportMenuCellItem

class SportMenuCellItem: BaseTableViewItem {
    
    typealias SelectedClosure = (SportMenuCellItem) -> Void

    private(set) var type: BDSportType = .normal
    
    var showSeparator: Bool = true

    var separatorColor: UIColor = Colors.Base.separator

    var selectedAction: SelectedClosure?

    private override init() {
        super.init()
        cellHeight = 44
        selectionStyle = .none
        accessoryType = .none
    }
    
    convenience init(type: BDSportType = .normal) {
        self.init()
        self.type = type
    }
}

// MARK: - SportMenuCell

final class SportMenuCell: BaseTableViewCell, ZJCellProtocol {

    // MARK: ZJCellProtocol
    
    var item: SportMenuCellItem!
    
    typealias ZJCelltemClass = SportMenuCellItem
    
    override func cellWillAppear() {
        super.cellWillAppear()
        guard item != nil else { return }
        sportButton.setTitle(item.type.text, for: .normal)
        setNeedsDisplay()
    }
            
    // MARK: Initialize
    
    override init(style: UITableViewCell.CellStyle, reuseIdentifier: String?) {
        super.init(style: .default, reuseIdentifier: reuseIdentifier)
        backgroundColor = Colors.Base.background
        addSubviews()
        setConstraints()
    }
    
    override func draw(_ rect: CGRect) {
        guard item != nil else { return }
        guard item.showSeparator else { return }
        let path = UIBezierPath(rect: CGRect(x: Constraints.normal, y: rect.size.height-0.5, width: rect.size.width - 2*Constraints.normal, height: 0.5))
        item.separatorColor.withAlphaComponent(0.8).setFill()
        path.fill(with: .normal, alpha: 1)
        path.close()
    }
    
    // MARK:  View customer

    private func addSubviews() {
        [sportButton].forEach { view in
            view.translatesAutoresizingMaskIntoConstraints = false
            addSubview(view)
        }
        
        sportButton.rx.tap.subscribe(onNext: { [weak self] _ in
            guard let strong = self,
                let hasItem = strong.item,
                let closure = hasItem.selectedAction else { return }
            closure(hasItem)
        }).disposed(by: disposeBag)
    }
    
    private func setConstraints() {
        sportButton.snp.makeConstraints { (make) in
            make.edges.equalToSuperview().inset(UIEdgeInsets(top: 0, left: 0, bottom: 0, right: 0))
        }
    }
        
    private let disposeBag = DisposeBag()
    
    private lazy var sportButton: UIButton = {
        let button = UIButton(type: .custom)
        button.titleLabel?.font = .customFont(ofSize: 15)
        button.contentHorizontalAlignment = .left
        button.setTitleColor(Colors.Base.title, for: .normal)
        button.setImage(ImageRepo.Sport.normal, for: .normal)
        button.setImage(ImageRepo.Sport.selected, for: .highlighted)
        
        button.imageEdgeInsets = UIEdgeInsets(top: 0, left: Constraints.normal, bottom: 0, right: 0)
        button.titleEdgeInsets = UIEdgeInsets(top: 0, left: Constraints.normal + Constraints.small, bottom: 0, right: 0)
        return button
    }()

}

class SportMenuHeaderView: UIView {
    var separatorColor: UIColor = Colors.Base.separator

    override func draw(_ rect: CGRect) {
        let path = UIBezierPath(rect: CGRect(x: 0, y: rect.size.height-0.5, width: rect.size.width, height: 0.5))
        separatorColor.withAlphaComponent(0.8).setFill()
        path.fill(with: .normal, alpha: 1)
        path.close()
    }
}
