//
//  WeatherView.swift
//  CTFit
//
//  Created by Mac on 2020/10/10.
//  Copyright © 2020 jpaxh. All rights reserved.
//

import UIKit
import NEUtils

extension NEWeatherType {
    
    var icon: UIImage {
        switch self {
        case .fineDay: return ImageRepo.Weather.fineDay
        case .cloudy: return ImageRepo.Weather.cloudy
        case .overCast: return ImageRepo.Weather.overCast
        case .lightRain: return ImageRepo.Weather.lightRain
        case .moderateRain: return ImageRepo.Weather.moderateRain
        case .heavyRain: return ImageRepo.Weather.heavyRain
        case .shower: return ImageRepo.Weather.shower
        case .snow: return ImageRepo.Weather.snow
        case .haze: return ImageRepo.Weather.haze
        case .sandStorm: return ImageRepo.Weather.sandStorm
        case .cloudyToFine: return ImageRepo.Weather.cloudyToFine
        case .thunderStorm: return ImageRepo.Weather.thunderStorm
        case .unknown: return ImageRepo.Weather.unknown
        }
    }
    
}

class WeatherView: UIView {
    
    // MARK: - Initialize
    @available(*, unavailable)
    required init?(coder: NSCoder) {
        fatalError("init(coder:) has not been implemented")
    }
    
    init() {
        super.init(frame: CGRect(x: 0, y: 0, width: UIDevice.scrWidth, height: UIDevice.naviHeight))
        addSubviews()
    }
    
    
    // MARK: - View customer
    private func addSubviews() {
        [temperatureLabel, icon].forEach { view in
            view.translatesAutoresizingMaskIntoConstraints = false
            addSubview(view)
        }
        
        temperatureLabel.snp.makeConstraints { (make) in
            make.centerY.equalToSuperview()
            make.centerX.equalToSuperview().offset(22)
        }
        
        icon.snp.makeConstraints { (make) in
            make.centerY.equalToSuperview()
            make.right.equalTo(temperatureLabel.snp.left).offset(-4)
        }
        
    }
    
    private var item: NEWeatherHourItem! {
        didSet {
            icon.image = item.type.icon
            let tempUnit = BandSettings.format.tempUnit
            let celsius = item.temperature
            if tempUnit == .fahrenheit{
                let fahrenheit = Int(BandCalculator.fahrenheit(celsius: BDCalculatorType(celsius)))
                self.temperatureLabel.text = "\(fahrenheit)" + SRString.Device.fahrenheit.locastr
            } else {
                self.temperatureLabel.text = "\(celsius)" + SRString.Device.centigrade.locastr
            }
        }
    }
    
    private var showing: Bool = false
    
    private var navigationItem: UINavigationItem?
    
    private lazy var icon: UIImageView = {
        let view = UIImageView()
        return view
    } ()
    
    private lazy var temperatureLabel: UILabel = {
        let label = UILabel(style: Stylesheet.Label.naviTitle)
        return label
    } ()
}

extension WeatherView {
    
    private func show(item: NEWeatherHourItem, in navigationItem: UINavigationItem) {
        self.item  = item
        if !showing {
            showing = true
            self.navigationItem = navigationItem
            navigationItem.titleView = self
        }
    }
    
    private func dismiss() {
        guard showing else { return }
        guard navigationItem != nil else { return }
        self.navigationItem?.titleView = nil
        showing = false
    }
    
    public func update(in navigationItem: UINavigationItem) {
        guard let item = NEWeatherInfo.currentHourItem else {
            dismiss()
            return
        }
        show(item: item, in: navigationItem)
    }
}
