//
//  SportViewController.swift
//  CTFit
//
//  Created by Mac on 2020/5/29.
//  Copyright © 2020 shirajo. All rights reserved.
//

import UIKit
import RxSwift
import RxCocoa
import NEUtils

class SportViewController: BaseViewController, CustomView {

    typealias ViewClass = SportView
    
    var exercising: Bool { customView.exercising }

    func startExercise(_ type: BDSportType) {
        if !exercising { customView.startExercise(type, .band)}
    }
    
    func finishExercise() {
        if exercising { customView.finishExercise(.band) }
    }

    override init() {
        super.init()
        navigationItem.title = SRString.Sport.sport.locastr
        self.navigationItem.hidesBackButton = true
    }
    
    override func loadView() {
        super.loadView()
        view = ViewClass()
    }

    override func viewDidLoad() {
        super.viewDidLoad()
        view.clipsToBounds = false
        self.jz_navigationBarBackgroundAlpha = 0
        customView.onSportDidChanged({ [weak self] (type) in
            guard let strong = self else { return }
            let tracking = type == nil ? false : true
            guard strong.tabBarController == nil else { return }
            strong.jz_navigationInteractivePopGestureEnabled = !tracking
            strong.navigationItem.leftBarButtonItem = tracking ? nil : strong.backBarItem()
        })
        updateWeather()
        if tabBarController == nil { LocationHelper.showNoLocationPermissionAlert() }
        self.customView.onViewDidLoad()
    }
    
    
    override func viewDidAppear(_ animated: Bool) {
        super.viewDidAppear(animated)
        self.customView.onViewDidAppear()
    }
    
    override func viewDidDisappear(_ animated: Bool) {
        super.viewDidDisappear(animated)
        self.customView.onViewDidDisappear()
    }
    
    
    // - View customer
    lazy var weatherView: WeatherView = WeatherView()

    private func updateWeather() {
        weatherView.update(in: navigationItem)
        NEWeatherInfo.updatedCurrentOutput.observeOn(MainScheduler.instance).subscribe(
            onNext: { [weak self] _ in
                guard let strong = self else { return }
                strong.weatherView.update(in: strong.navigationItem)
            }
        ).disposed(by: disposeBag)
    }
}

extension SportViewController {
    private static var mainScheduler = MainScheduler.instance
    private static var disposeBag = DisposeBag()
    private static var watting: ExerciseNotifyWating? = nil
    private static var alert: UIAlertController? = nil

    
    public static func observe() {
        BandData.updatedOutput.observeOn(mainScheduler).subscribe(onNext: { (dataType) in
            guard case BDDataType.notify (let notifyType) = dataType,
                  case BDNotifyType.exercise(let exerciseType) = notifyType  else { return }
            exerciseNotify(exerciseType)
        }).disposed(by: disposeBag)
        
        BandData.appNotifyOutput.observeOn(MainScheduler.instance).subscribe(onNext: { type in
            switch type {
            case .didBecomeActive: cancelAllLocalNotifications()
            default: break
            }
        }).disposed(by: disposeBag)
        
        BandConnect.statusChangedOutput.observeOn(MainScheduler.instance).subscribe(onNext: { state in
            if case .disconnected = state { exercise(.ignore) }
        }).disposed(by: disposeBag)
    }
    
  
    
    private static func valid(_ type: BDNotifyType.Exercise) -> Bool {
        if let hasWatting = watting {
            if hasWatting.type == type {
                return !hasWatting.valid
            } else {
                switch type {
                case .start: return !hasWatting.valid
                case .stop, .ignore:
                    if case BDNotifyType.Exercise.sport(_) = hasWatting.type,
                       hasWatting.valid {
                        return true
                    } else {
                        return false
                    }
                case .sport(_):
                    if hasWatting.type == .start, hasWatting.valid {
                        return true
                    } else {
                        return false
                    }
                }
            }
        } else if type == .start || type == .stop {
            return true
        } else {
            return false
        }
    }
    
    private static func exerciseNotify(_ type: BDNotifyType.Exercise)  {
        guard valid(type) else { return }
        watting = ExerciseNotifyWating(type: type)
        UILog.d("Exercise sport: \(type)")
        exercise(type)
    }
    
    private static func exercise(_ type: BDNotifyType.Exercise) {
        if type == .ignore {
            watting?.handled = true;
            alert?.dismiss(animated: true, completion: nil)
            alert = nil
        }
        if runningInBackground() {
            switch type {
            case .sport(let sportType): localNotify(.sport(sportType))
            case .stop: localNotify(.stop); finishExercise();
            default: break
            }
        } else {
            switch type {
            case .sport(let sportType): startExerciseAlertShow(sportType)
            case .stop: finishExercise()
            default: break
            }
        }
    }
    
    
    private static func startExerciseAlertShow(_ sportType: BDSportType) {
        
        alert = AlertManager.show(title: nil, message: SRString.Sport.band_start_gps_alert.locastr, okHander: { (_) in
            startExercise(sportType)
            DispatchQueue.main.async { alert = nil }
        }) { (_) in
            DispatchQueue.main.async { alert = nil }
        }
    }
    
    private static func startExerciseShowDismiss() {
        DispatchQueue.main.async {
            alert?.dismiss(animated: true, completion: nil)
            alert = nil
        }
    }
    
    
    private static func startExercise(_ sportType: BDSportType) {
        guard let currentVc = Helper.currentVc else { return }
        guard let navVc = currentVc.navigationController else { return }
        var sportVc: SportViewController
        if let vc = Helper.currentVc as? SportViewController {
            sportVc = vc
        } else {
            if let index = navVc.viewControllers.firstIndex(where: { $0.isKind(of: SportViewController.self) } ),
               let vc = navVc.viewControllers[index] as? SportViewController {
                sportVc = vc
                navVc.popToViewController(sportVc, animated: true)
            } else {
                let transition = CATransition()
                transition.duration = 0.35
                if let tabbarVc = currentVc.tabBarController as? HomeTabBarController {
                   navVc.popToRootViewController(animated: false)
                   tabbarVc.selectedIndex = 1
                   tabbarVc.view.layer.add(transition, forKey: nil)
                   sportVc = tabbarVc.sportVc
               } else {
                   sportVc = SportViewController()
                   navVc.popToRootViewController(animated: false)
                   navVc.pushViewController(sportVc, animated: false)
                   navVc.view.layer.add(transition, forKey: nil)
               }
           }
        }
        sportVc.startExercise(sportType)
    }
    
    private static func finishExercise() {
        guard let sportVc = Helper.currentVc as? SportViewController else { return }
        sportVc.finishExercise()
    }
    
    static func runningInBackground() -> Bool {
        return (UIApplication.shared.applicationState == .background);
    }
    
    private static var exercising: Bool {
        guard let sportVc = Helper.currentVc as? SportViewController,
              sportVc.exercising else { return false  }
        return true
    }
    
    private static func localNotify(_ type: BDNotifyType.Exercise) {
        let localNotification = UILocalNotification();
        localNotification.alertBody = type.alertBody
        localNotification.soundName = UILocalNotificationDefaultSoundName;
        localNotification.fireDate = Date(timeIntervalSinceNow: 1.0)
        localNotification.timeZone = TimeZone.current;
        localNotification.userInfo = ["ExerciseType": type.value];
        UIApplication.shared.scheduleLocalNotification(localNotification);
    }
    
    
    static fileprivate func localNotificationTap(for type: BDNotifyType.Exercise)  {
        guard let hasWatting = watting, hasWatting.valid,
              type == hasWatting.type else { return }
        switch hasWatting.type {
        case .sport(let sportType): exercise(.sport(sportType))
        default: break
        }
    }
    /// cancelAllLocalNotifications
    static fileprivate func cancelAllLocalNotifications() {
        UIApplication.shared.applicationIconBadgeNumber = 0;
        if #available(iOS 10, *) {
            UNUserNotificationCenter.current().removeAllDeliveredNotifications()
        } else {
            guard let hasLocalNotifications = UIApplication.shared.scheduledLocalNotifications else { return }
            for localNotification in hasLocalNotifications  {
                UIApplication.shared.cancelLocalNotification(localNotification)
            }
        }
    }
}

struct ExerciseNotifyWating {

    /// 等待操作的名称
    var type: BDNotifyType.Exercise = .start
    
    /// 等待操作的起始时间, seconds from 1970
    var startTime: TimeInterval = Date().timeIntervalSince1970
    
    /// 等待操作的有效时间，超过此时间，操作将失效, in seconds
    var duration: TimeInterval = 30
    
    /// 是否已经被处理
    var handled: Bool = false;
    
    var valid: Bool {
        guard !handled else { return false }
        let curTime = Date().timeIntervalSince1970;
        if curTime - startTime > duration {
            return false
        } else {
            return true;
        }
    }
}

extension BDNotifyType.Exercise {
    var alertBody: String {
        switch self {
        case .start, .sport(_): return SRString.Sport.band_start_gps_notify.locastr
        case .stop, .ignore: return SRString.Sport.band_stop_gps_notify.locastr
        }
    }
}

extension AppDelegate {
    func application(_ application: UIApplication, didReceive notification: UILocalNotification) {
        let userInfo = notification.userInfo!
        guard let value = userInfo["ExerciseType"] as? UInt16,
              let type = BDNotifyType.Exercise.parse(value) else { return }
        SportViewController.localNotificationTap(for: type)
    }
}
