//
//  CoreDataManager.swift
//  CTFit
//
//  Created by Mac on 2020/12/22.
//  Copyright © 2020 jpaxh. All rights reserved.
//

import Foundation
import CoreData
import CoreLocation

class CoreDataManager {
    
    private static var documentUrl: URL {
        return FileManager.default.urls(for: .documentDirectory, in: .userDomainMask)[0]
    }
    
    private static var sqliteUrl: URL {
        return documentUrl.appendingPathComponent("CTFit").appendingPathExtension("db")
    }
    
    /// 判断是否存在数据库文件
    private static func exitDBFile(_ path: String?) -> Bool {
        guard let hasPath = path,
              FileManager.default.fileExists(atPath: hasPath) else { return false }
        return true
    }
    
    private static var managedObjectModel: NSManagedObjectModel = {
        let momdUrl = Bundle.main.url(forResource: "CTFitModel", withExtension: "momd");
        guard let hasMomdUrl = momdUrl  else { fatalError()}
        let momd = NSManagedObjectModel(contentsOf: hasMomdUrl)
        return momd!;
    }()
    
    private static var persistentStore: NSPersistentStoreCoordinator = {
        let coordinator = NSPersistentStoreCoordinator(managedObjectModel: managedObjectModel)
        let options = [NSMigratePersistentStoresAutomaticallyOption: true, NSInferMappingModelAutomaticallyOption:true]
        do {
            try coordinator.addPersistentStore(ofType: NSSQLiteStoreType, configurationName: nil, at: sqliteUrl, options: options);
        } catch let error {
            print("Faile to add persistent store: error = \(error)")
        }
        return coordinator
    }()
    
    private static var context: NSManagedObjectContext = {
        let context = NSManagedObjectContext(concurrencyType: .mainQueueConcurrencyType)
        context.persistentStoreCoordinator = persistentStore
        return context
    }()
    
    /// 查询数据
    fileprivate static func queryData() -> [CTFitTrack] {
        let fetchRequest = NSFetchRequest<CTFitTrack>(entityName:"CTFitTrack")
        fetchRequest.fetchLimit = 100 //限定查询结果的数量
        fetchRequest.fetchOffset = 0 //查询的偏移量
        /// 设置查询条件
        let date = CalendarUtils.date(from: "2018-01-01", format: "yyyy-MM-dd")
        let predicate = NSPredicate(format: "startTime > \(date.timeIntervalSince1970)")
        fetchRequest.predicate = predicate
        /// 设置排序
        let sortDescriptor = NSSortDescriptor(key: "startTime", ascending: false)
        fetchRequest.sortDescriptors = [sortDescriptor];
        /// 查询操作
        do {
            let tracks = try context.fetch(fetchRequest)
            return tracks
        }
        catch {
            UILog.w("查询失败：\(error)")
        }
        return []
    }
    
    @discardableResult
    static func migration() -> Bool {
        guard exitDBFile(sqliteUrl.path) else { return false }
        let recorders = queryData().map{ SportRecorder(with: $0) }
        recorders.save()
        UILog.d("迁移 CTFitTrack 完成")
        return true
    }
}



@objc(TrackPointsTransformer)
fileprivate class TrackPointsTransformer: ValueTransformer {
    override class func transformedValueClass() -> AnyClass {
        return NSArray.self
    }

    override class func allowsReverseTransformation() -> Bool {
        return true;
    }

    override func transformedValue(_ value: Any?) -> Any? {
        guard let hasValue = value else { return nil }
        return NSKeyedArchiver.archivedData(withRootObject: hasValue);
    }

    override func reverseTransformedValue(_ value: Any?) -> Any? {
        guard let hasData = value as? Data else { return nil }
        return NSKeyedUnarchiver.unarchiveObject(with: hasData);
    }
}



@objc(CTFitTrack)
fileprivate class CTFitTrack: NSManagedObject {
    
    /// 名称
    @NSManaged public var name: String?
    /// 描述
    @NSManaged public var desc: String?
    /// 运动类型
    @NSManaged public var sportType: Int32
    /// 开始时间戳
    @NSManaged public var startTime: Int64
    /// 开始时间戳
    @NSManaged public var endTime: Int64
    /// 运动距离
    @NSManaged public var distance: Double
    /// 连续的运动运动轨迹
    @NSManaged public var trackPoints: NSArray?
    
    /// 手环训练属性
    @NSManaged public var steps: Int32
    @NSManaged public var calories: Int32
    @NSManaged public var sfMax: Int32
    @NSManaged public var sfAvg: Int32
    @NSManaged public var heartRateMax: Int32
    @NSManaged public var heartRateAvg: Int32
    @NSManaged public var heartRates: NSArray?
}

/// 运动轨迹位置点模型 （WGS84）
@objc(AgpsTrackPoint)
fileprivate class AgpsTrackPoint: NSObject, NSCoding {

    /// 位置经度
    var latitude: Double = 0.0
    /// 位置纬度
    var longitude: Double = 0.0
    /// 位置海拔高度
    var elevation: Double?
    /// course:  航向度数
    var magneticVariation: Double?
    /// 时间戳（1970）.
    var timeStamp: Double?

    /// 该位置的速度（米/秒），如果无效则为 nil
    var speed: Double?
    /// 距离 （米）
    var distance: Double?
    
    /// 路径点的GPS名称.
    var name: String?
    /// GPS航点评论。发送给GPS作为评论.
    var comment: String?
    /// 元素的文本描述。保存针对用户的元素的附加信息，而不是GPS.
    var desc: String?
    
    init(lat: Double, lon: Double) {
        super.init()
        self.latitude = lat;
        self.longitude = lon;
    }
    
    convenience init(location: CLLocation){
        self.init(lat: location.coordinate.latitude, lon: location.coordinate.longitude)
        self.elevation = location.altitude;
        self.magneticVariation = location.course;
        self.timeStamp = location.timestamp.timeIntervalSince1970
        self.speed = location.speed;
    }
    
    convenience init(location: CLLocation, speed :Double){
        self.init(lat: location.coordinate.latitude, lon: location.coordinate.longitude)
        self.elevation = location.altitude;
        self.magneticVariation = location.course;
        self.timeStamp = location.timestamp.timeIntervalSince1970
        self.speed = speed;
    }
    
    required init?(coder aDecoder: NSCoder) {
        super.init()
        self.latitude = aDecoder.decodeDouble(forKey: "latitude")
        self.longitude = aDecoder.decodeDouble(forKey: "longitude")
        /// 解析必须按 objec t来解析，因为存储的为 optioanal(00.000000)
        self.elevation = aDecoder.decodeObject(forKey: "elevation") as? Double
        self.magneticVariation = aDecoder.decodeObject(forKey: "magneticVariation") as? Double
        self.timeStamp = aDecoder.decodeObject(forKey: "timeStamp") as? Double
        self.speed = aDecoder.decodeObject(forKey: "speed") as? Double
        self.distance = aDecoder.decodeObject(forKey: "distance") as? Double

        self.name = aDecoder.decodeObject(forKey: "name") as? String
        self.comment = aDecoder.decodeObject(forKey: "comment") as? String
        self.desc = aDecoder.decodeObject(forKey: "desc") as? String
        
    }
    
    func encode(with aCoder: NSCoder) {
        aCoder.encode(self.latitude, forKey: "latitude")
        aCoder.encode(self.longitude, forKey: "longitude")
        aCoder.encode(self.elevation, forKey: "elevation")
        aCoder.encode(self.magneticVariation, forKey: "magneticVariation")
        aCoder.encode(self.timeStamp, forKey: "timeStamp")
        aCoder.encode(self.speed, forKey: "speed")
        aCoder.encode(self.distance, forKey: "distance")

        aCoder.encode(self.name, forKey: "name")
        aCoder.encode(self.comment, forKey: "comment")
        aCoder.encode(self.desc, forKey: "desc")
    }
}


fileprivate extension Array where Element: SportRecorder {
    func save() { forEach { $0.save() } }
}

fileprivate extension SportRecorder {
    convenience init(with track: CTFitTrack) {
        self.init()
        sportType = Int(truncatingIfNeeded: track.sportType)
        sportState = .finish
        startTime = UInt32(truncatingIfNeeded: track.startTime)
        endTime = UInt32(truncatingIfNeeded: track.endTime)
        distance = track.distance
        tracks = [SportTrack(with: track)]
    }
}

fileprivate extension SportTrack {
    convenience init(with track: CTFitTrack) {
        self.init()
        startTime = UInt32(truncatingIfNeeded: track.startTime)
        endTime = UInt32(truncatingIfNeeded: track.endTime)
        distance = track.distance
        guard let tarckPoints = track.trackPoints as? [AgpsTrackPoint] else { return }
        points = tarckPoints.map { SportPoint(with: $0) }
    }
}
fileprivate extension SportPoint {
    convenience init(with track: AgpsTrackPoint) {
        self.init(lat: track.latitude, lon: track.longitude)
        self.altitude = track.elevation ?? 0;
        self.time = UInt32(track.timeStamp ?? 0)
        self.distance = track.distance ?? 0
        self.speed = max(track.speed ?? 0, 0)
    }
}
