//
//  GpxRecordSportTrackView.swift
//  CTFit
//
//  Created by Mac on 2020/6/19.
//  Copyright © 2020 shirajo. All rights reserved.
//

import Foundation
import GRDB

struct GpxRecord: Codable {
    var id: Int?
    /// 开始时间戳
    private(set) var startTime: UInt32?
    /// 结束时间戳
    private(set) var endTime: UInt32?
    /// 运动类型
    private(set) var sportType: Int?
    /// 运动状态
    private(set) var sportState: Int?
    /// 运动距离
    private(set) var distance: Double?
    /// 连续的运动运动轨迹组
    private(set) var tracks: String?
    
    init(_ record: SportRecorder) {
        startTime = record.startTime
        endTime = record.endTime
        sportType = record.sportType
        sportState = record.sportState.rawValue
        distance = record.distance
        tracks = record.tracks.toJSONString(prettyPrint: true)
    }
}

// MARK: - ColumnExpression

extension GpxRecord {
    enum Columns: String, ColumnExpression {
        case id, startTime, endTime, sportType, sportState, distance, tracks
    }
}

// MARK: - TableRecord
/// 使用 TableRecord 协议, 告知操作数据表
extension GpxRecord: TableRecord {
    //  数据表名字 如果不自定义 将使用默认的表名, 默认为小写开头驼峰命名 例如 UserEntity -> userEntity
    static var databaseTableName: String { return "GpxRecord" }
    // 声明列名表示的方式
    static var databaseSelection: [SQLSelectable] = [Columns.id, Columns.startTime, Columns.endTime, Columns.sportType,
                                                     Columns.sportState, Columns.distance, Columns.tracks]
}

// MARK: - FetchableRecord
/// FetchableRecord 进行查询操作
/// 使用了Codable可以不实现  init (row: Row)
/// 未使用 Codable = Decodable & Encodable 协议的需要实现 init (row: Row)
extension GpxRecord: FetchableRecord {
    init(row: Row) {
        id          = row[Columns.id]
        startTime   = row[Columns.startTime]
        endTime     = row[Columns.endTime]
        sportType   = row[Columns.sportType]
        sportState  = row[Columns.sportState]
        distance    = row[Columns.distance]
        tracks      = row[Columns.tracks]
    }
}

// MARK: - MutablePersistableRecord
/// 使用PersistableRecord / MutablePersistableRecord插入更新保存数据,
/// 存储模型是Class使用PersistableRecord,
/// 存储模型是struct使用MutablePersistableRecord.
/// 两者区别在于 MutablePersistableRecord save() insert() 是 mutating.
extension GpxRecord: MutablePersistableRecord {
    func encode(to container: inout PersistenceContainer) {
        container[Columns.id]           = id
        container[Columns.startTime]    = startTime
        container[Columns.endTime]      = endTime
        container[Columns.sportType]    = sportType
        container[Columns.sportState]   = sportState
        container[Columns.distance]     = distance
        container[Columns.tracks]       = tracks
    }
    
    mutating func didInsert(with rowID: Int64, for column: String?) { }
}

import BandKit
extension GpxRecord {
    
    /// 获取数据库对象
    private static var dbQueue: DatabaseQueue? {
        SQLiteManager.dbQueue
    }
    
    /// 创建数据库
    private static func createTable() -> Void {
        guard let _dbQueue = dbQueue else { return }
        do {
            try _dbQueue.inDatabase { (db) -> Void in
                /// 创建数据库表
                try db.create(table: databaseTableName, temporary: false, ifNotExists: true, body: { (t) in
                    t.autoIncrementedPrimaryKey("id")
                    t.column(Columns.startTime.rawValue, .integer)
                    t.column(Columns.endTime.rawValue, .integer)
                    t.column(Columns.sportType.rawValue, .integer)
                    t.column(Columns.sportState.rawValue,.integer)
                    t.column(Columns.distance.rawValue, .double)
                    t.column(Columns.tracks.rawValue, .text)
                })
            }
        } catch {
            UILog.e("\(error)")
        }
        
    }
}

// MARK: - 查询

extension GpxRecord {
    
    private static func queryDay(_ date: Date) -> [GpxRecord] {
        guard let _dbQueue = dbQueue else { return [] }
        /// 创建数据库
        self.createTable()
        /// 返回查询结果
        let startTime = CalendarUtils.dayDate(for: date).timeIntervalSince1970
        let endTime = startTime + TimeInterval(CalendarUtils.kSecondsOneDay)
        do {
            return try _dbQueue.unsafeRead({ (db) -> [GpxRecord] in
                return try GpxRecord.filter(Column(Columns.startTime.rawValue) >= startTime && Column(Columns.startTime.rawValue) < endTime).fetchAll(db)
            })
        } catch {
            UILog.e("\(error)")
            return []
        }
        
    }
    
    /// 查询单个
    private static func query(startTime: UInt32) -> GpxRecord? {
        guard let _dbQueue = dbQueue else { return nil }
        /// 创建数据库
        self.createTable()
        /// 返回查询结果
        do {
            return try _dbQueue.unsafeRead({ (db) -> GpxRecord? in
                return try GpxRecord.filter(Column(Columns.startTime.rawValue) == startTime).fetchOne(db)
            })
        } catch {
            UILog.e("\(error)")
            return nil
        }
    }
    
    /// 查询某天
    static func querySportDay(_ date: Date) -> [SportRecorder] {
        let gpxRecords = queryDay(date)
        var sportRecords = [SportRecorder]()
        gpxRecords.forEach { (e) in
            let sportRecord = SportRecorder()
            sportRecord.parse(for: e)
            sportRecords.append(sportRecord)
        }
        return sportRecords
    }
    
    /// 查询所有
    static func queryAll() -> [GpxRecord] {
        guard let _dbQueue = dbQueue else { return [] }
        /// 创建数据库
        self.createTable()
        /// 返回查询结果
        do {
            return try _dbQueue.unsafeRead({ (db) -> [GpxRecord] in
                return try GpxRecord.fetchAll(db)
            })
        } catch {
            UILog.e("\(error)")
            return []
        }
    }
}

// MARK: - 插入&更新

extension GpxRecord {
    
    /// 插入单个数据， startTime 相等则更新， 否则插入
    private static func insert(record: GpxRecord) {
        guard let _dbQueue = dbQueue else { return }
        /// 创建数据库表
        self.createTable()
        /// 事务
        do {
            try _dbQueue.inTransaction { (db) -> Database.TransactionCompletion in
                do {
                    var temp = record
                    try temp.insert(db)
                    return Database.TransactionCompletion.commit
                } catch {
                    return Database.TransactionCompletion.rollback
                }
            }
        } catch {
            UILog.e("\(error)")
        }
    }

    /// 更新
    private static func update(record: GpxRecord) {
        guard let _dbQueue = dbQueue else { return }
        /// 创建数据库表
        self.createTable()
        /// 事务 更新场景
        do {
            try _dbQueue.inTransaction { (db) -> Database.TransactionCompletion in
                do {
                    // 赋值
                    try record.update(db)
                    return Database.TransactionCompletion.commit
                } catch {
                    return Database.TransactionCompletion.rollback
                }
            }
        } catch {
            UILog.e("\(error)")
        }
        
        
    }
    
    static func insert(record: SportRecorder) {
        let gpxRecord = GpxRecord(record)
        /// 判断是否存在， 存在则更新
        if let _ = query(startTime: record.startTime) {
            self.update(record: gpxRecord)
        } else {
            insert(record: gpxRecord)
        }
    }
}


// MARK: - 删除
extension GpxRecord {
    ///: - 移除表
    internal static func remove() {
        guard let _dbQueue = dbQueue else { return }
        /// 事务
        try! _dbQueue.inTransaction { (db) -> Database.TransactionCompletion in
            do {
                try db.drop(table: databaseTableName)
                return Database.TransactionCompletion.commit
            } catch {
                return Database.TransactionCompletion.rollback
            }
        }
    }
    
    ///: - 删除单个
    internal static func delete(record: GpxRecord) {
        guard let _dbQueue = dbQueue else { return }
        /// 是否有数据库表
        self.createTable()
        /// 事务
        do {
            try _dbQueue.inTransaction { (db) -> Database.TransactionCompletion in
                do {
                    /// 删除数据
                    try record.delete(db)
                    return Database.TransactionCompletion.commit
                } catch {
                    return Database.TransactionCompletion.rollback
                }
            }
        } catch {
            UILog.e("\(error)")
        }
    }
    
    ///: - 根据 startTime 删除
    internal static func delete(startTime: UInt32) {
        guard let _dbQueue = dbQueue else { return }
        /// 是否有数据库表
        self.createTable()
        /// 事务
        do {
            try _dbQueue.inTransaction { (db) -> Database.TransactionCompletion in
                do {
                    /// 删除数据
                    try GpxRecord.filter(Column(Columns.startTime.rawValue) == startTime).deleteAll(db)
                    return Database.TransactionCompletion.commit
                } catch {
                    return Database.TransactionCompletion.rollback
                }
            }
        } catch {
            UILog.e("\(error)")
        }
    }
    
    ///: - 删除所有
    internal static func deleteAll() {
        guard let _dbQueue = dbQueue else { return }
        /// 是否有数据库表
        self.createTable()
        /// 事务
        do {
            try _dbQueue.inTransaction { (db) -> Database.TransactionCompletion in
                do {
                    /// 删除数据
                    try GpxRecord.deleteAll(db)
                    return Database.TransactionCompletion.commit
                } catch {
                    return Database.TransactionCompletion.rollback
                }
            }
        } catch {
            UILog.e("\(error)")
        }
    }
}

