//
//  SportPresenter.swift
//  CTFit
//
//  Created by Mac on 2020/6/12.
//  Copyright © 2020 shirajo. All rights reserved.
//

import UIKit
import MapKit
import Repeater

enum SportSource {
    case band, app
}

class SportPresenter {
    
    init() {

        locationHelper.startUpdatingHeading()
        
        locationHelper.didUpdateHeading = { [weak self] (heading) in
            guard let strong = self else { return }
            strong.mapView.updateHeading(heading)
        }
        
        locationHelper.didUpdateLocation = { [weak self] (location) in
            guard let strong = self else { return }
            strong.tracking(location)
        }
        
        mapView.tapMapScale = { [weak self] (_) in
            guard let strong = self else { return }
            strong.scaleAction()
        }
        
        bottomView.tapAction = { [weak self] (control) in
            guard let strong = self else { return }
            if control.actionState == .normal {
                strong.startExercise(control.sportType)
            } else {
                strong.finishExercise()
            }
            strong.mapView.exercising = strong.exercising
        }
        
        mapView.exercising = exercising
    }
    
    // MARK: - Field
    typealias SportChangedClosure = (_ type: BDSportType?) ->Void
    lazy var sportChangedClosure: SportChangedClosure? = nil

    /// View customer
    lazy var mapView: MapView = MapView()
    lazy var headerView: SportHeaderView = SportHeaderView()
    lazy var bottomView: SportBottomView = SportBottomView()
    var exercising: Bool { bottomView.actionState == .selected }

    private lazy var sportRecorder = SportRecorder()
    private lazy var locationHelper: LocationHelper = LocationHelper.shared
    private lazy var timer: Repeater = {
        Repeater(interval: .seconds(1), mode: .infinite) { [weak self] _ in
            guard let strong = self else { return }
            DispatchQueue.main.async {
                strong.bottomView.duration = strong.sportRecorder.duration
                // strong.agpsSync(of: .syncData)
            }
        }
    }()
    
    lazy var sportAgps: BDSportAgps = { BDSportAgps() } ()

}

extension SportPresenter {
    
    func requestAuthorization() {
        locationHelper.requestAuthorization()
    }
    
    func startExercise(_ type: BDSportType, _ source: SportSource = .app) {
        start(type, source)
        sportChangedClosure?(type)
        if source == .app {
            BandCommand.agpsStart(for: type)
            agpsSync(of: .start)
        } else {
            BandCommand.agpsEventStart()
        }
    }
    
    func finishExercise(_ source: SportSource = .app) {
        finish(source)
        if source == .app {
            BandCommand.agpsEnd()
            agpsSync(of: .stop)
        } else {
            BandCommand.agpsEventEnd()
        }
    }
    
    ///: - series_s
    func agpsSync(of status: BDSportStatus) {
        guard let type = BDSportType.parse(UInt8(sportRecorder.sportType)) else { return }
        let duration = Int32(bottomView.duration)
        var distance = Int32(sportRecorder.distance)
        if status == .syncData { distance = 2 * duration }
        BandCommand.agpsSync(BDSportAgps(type: type, status: status, time: duration, distance: distance))
    }
}

// MARK: - Private method

extension SportPresenter {
    private func start(_ type: BDSportType, _ source: SportSource) {
        sportRecorder = SportRecorder(sportType: type)
        sportRecorder.executeTrackingStart()
        timer.start()
        mapView.startSportTracking()
        locationHelper.startUpdatingLocation()
        headerView.sportTypeLabel.text = type.text
        bottomView.historyButton(hidden: true)
    }
    
    private func finish(_ source: SportSource) {
        sportRecorder.executeTrackingEnd()
        timer.pause()
        locationHelper.stopUpdatingLocation()
        mapView.stopSportTracking()
        toDetail()
        clear()
        headerView.sportTypeLabel.text = nil
        bottomView.historyButton(hidden: false)
        sportChangedClosure?(nil)
    }
    
    private func tracking(_ location: CLLocation) {
        guard sportRecorder.sportState == .tracking else { return }
        mapView.updateSignal(location.horizontalAccuracy)
        bottomView.speed = max(location.speed, 0)
        guard let point = sportRecorder.appendLocation(location) else { return }
        agpsSync(of: .syncData)
        bottomView.distance = sportRecorder.distance
        point.compatible { (location) in
            DispatchQueue.main.async { [weak self] in
                guard let strong = self else { return }
                strong.sportRecorder.currentCoordinates.append(location)
                /// 更新轨迹
                strong.updateCurrentSegmentOverlay()
                /// 语音播报
                if SportBottomView.voiceEnable { SportRecorder.voiceReminder(strong.sportRecorder) }
            }
        }
    }
    
    private func updateCurrentSegmentOverlay() {
        mapView.updateCurrentSegmentOverlay(sportRecorder.currentSegmentOverlay)
    }
    
    private func toDetail(){
        let vc = SportResultViewController(sportRecorder)
        Helper.currentVc?.navigationController?.pushViewController(vc, animated: true)
    }
    
    private func clear(){
        bottomView.clear()
        mapView.updateCurrentSegmentOverlay(nil)
    }
}


// MARK: - 地图显示区域放大/缩小

extension SportPresenter {
    
    private func scaleAction() {
        if self.bottomView.isHidden {
            self.showBottomView()
        } else {
            self.hideBottomView()
        }
    }
    
    private func hideBottomView() {
        let tempHeight = self.bottomView.bounds.height;
        self.bottomView.snp.updateConstraints { (make) in
            make.bottom.equalToSuperview().offset(tempHeight)
        }
        UIView.animate(withDuration: 0.3, animations: {
            self.bottomView.superview!.layoutIfNeeded()
        }) { (completed) in
            self.bottomView.isHidden = true;
        }
        
    }
    
    private func showBottomView() {
        self.bottomView.isHidden = false;
        self.bottomView.snp.updateConstraints { (make) in
            make.bottom.equalToSuperview().offset(0)
        }
        UIView.animate(withDuration: 0.3, animations: {
            self.bottomView.superview!.layoutIfNeeded()
        })
    }
}




