//
//  SportRecorder.swift
//  CTFit
//
//  Created by Mac on 2020/6/15.
//  Copyright © 2020 shirajo. All rights reserved.
//

import Foundation
import MapKit
import AVFoundation
import HandyJSON

enum SportState: Int, HandyJSONEnum {
    case none = 0
    case tracking = 1
    case pause = 2
    case finish = 3
}

class SportRecorder: HandyJSON {
    
    // MARK: - 存储字段
    /// 运动类型
    var sportType: Int
    /// 运动状态
    var sportState: SportState = .none
    /// 开始时间戳
    var startTime: UInt32 = 0
    /// 结束时间戳
    var endTime: UInt32?
    /// 运动距离
    var distance: Double = 0.0
    /// 连续的运动运动轨迹组 ( 如果支持暂停功能：多段运动轨迹 )
    var tracks = [SportTrack]()

    
    // MARK: - 非存储字段
    /// 正在开启运动时，当前跟踪段
    private(set) var currentTrack: SportTrack?
    /// 当前跟踪段的最后一个位置
    private(set) var lastLocation: CLLocation?
    /// 当前位置记录
    lazy var currentCoordinates: [CLLocationCoordinate2D] = []
    /// 当前轨迹覆盖段
    var currentSegmentOverlay: MKOverlay? {
        return MKPolyline(coordinates: self.currentCoordinates, count: self.currentCoordinates.count);
    }
    
    // MARK: - Convenience fields
    /// 持续时间
    var duration: Int {
        var totalDuration = 0
        tracks.forEach { (e) in totalDuration += e.duration }
        return totalDuration
    }
    
    var startDate: Date {
        Date(timeIntervalSince1970: TimeInterval(startTime));
    }
    
    var endDate: Date? {
        guard let time = endTime else { return nil }
        return Date(timeIntervalSince1970: TimeInterval(time));
    }
    
    lazy var formatter :DateFormatter = {
        let fm = DateFormatter();
        fm.dateFormat = "HH:mm";
        return fm;
    }()
    
    lazy var averageSpeend: Double = {
        guard duration > 0 else { return 0 }
        return distance / Double(duration)
    } ()
    
    lazy var fastSpeed: Double = {
        var speed: Double = averageSpeend
        tracks.forEach { (track) in
            track.points.forEach { (point) in
                if point.speed > speed { speed = point.speed }
            }
        }
        return speed
    } ()
    
    lazy var slowestSpeed: Double = {
        var speed: Double = averageSpeend
        tracks.forEach { (track) in
            track.points.forEach { (point) in
                if point.speed < speed, point.speed > 0 { speed = point.speed }
            }
        }
        return speed
    } ()
    
    var startTimeText: String {
        formatter.dateFormat = "yyyy-MM-dd HH:mm:ss, ";
        let startText: String = formatter.string(from: startDate);
        return startText
    }
    
    func startTimeTextTrack() -> String {
        let formatter = DateFormatter();
        let calendar = Calendar.current;
        if Calendar.current.isDateInToday(startDate) {
            let todayStr = SRString.Sport.today.locastr
            formatter.dateFormat = "HH:mm"
            let time = formatter.string(from: startDate);
            return todayStr + " " + time;
        } else if calendar.isDateInYesterday(startDate) {
            let todayStr = SRString.Sport.yesterday.locastr
            formatter.dateFormat = "HH:mm"
            let time = formatter.string(from: startDate);
            return todayStr + " " + time;
        } else {
            formatter.dateFormat = "yyyy-MM-dd HH:mm"
            let time = formatter.string(from: startDate);
            return time;
        }
    }
    
    var durationText: String {
        return String(format: "%02d:%02d:%02d", Int(duration)/3600, (Int(duration)/60)%60, Int(duration)%60)
    }
    
    var periodTimeText: String {
        guard let hasEndDate = endDate else { return "--" }
        let sameYear = CalendarUtils.calendar.isSame(in: .year, date: startDate, to: hasEndDate)
        let sameMonth = CalendarUtils.calendar.isSame(in: .month, date: startDate, to: hasEndDate)
        let sameDay = CalendarUtils.calendar.isSame(in: .day, date: startDate, to: hasEndDate)
        
        formatter.dateFormat = "yyyy-MM-dd HH:mm";
        let startText: String = formatter.string(from: startDate);
        
        formatter.dateFormat = "HH:mm";
        var endText: String = formatter.string(from: hasEndDate);
        
        if sameYear, sameMonth, !sameDay {
            formatter.dateFormat = "d HH:mm";
            endText = formatter.string(from: hasEndDate)
        } else if sameYear, !sameMonth {
            formatter.dateFormat = "MM-dd HH:mm";
            endText = formatter.string(from: hasEndDate)
        } else if !sameYear {
            formatter.dateFormat = "yyyy-MM-dd HH:mm";
            endText = formatter.string(from: hasEndDate)
        } else {
            formatter.dateFormat = "HH:mm";
            endText = formatter.string(from: hasEndDate)
        }
        
        return String(format: "%@-%@", startText, endText);
    }

    
    // MARK: - Initialize
    
    required init() {
         self.sportType = Int(BDSportType.normal.value)
    }
    
    init(sportType: BDSportType = .normal ) {
        self.sportType = Int(sportType.value)
    }
    
    init(for gpxRecord: GpxRecord ) {
        startTime = gpxRecord.startTime!
        endTime = gpxRecord.endTime!
        sportType = gpxRecord.sportType!
        sportState = SportState(rawValue: gpxRecord.sportState!)!
        distance = gpxRecord.distance!
        tracks = [SportTrack].deserialize(from: gpxRecord.tracks)! as! [SportTrack]
    }
    
    func parse(for gpxRecord: GpxRecord) {
        startTime = gpxRecord.startTime!
        endTime = gpxRecord.endTime!
        sportType = gpxRecord.sportType!
        sportState = SportState(rawValue: gpxRecord.sportState!)!
        distance = gpxRecord.distance!
        tracks = [SportTrack].deserialize(from: gpxRecord.tracks)! as! [SportTrack]
    }
    
    func mapping(mapper: HelpingMapper) {
        /// 非解析字段
        mapper >>> lastLocation
        mapper >>> currentCoordinates
    }

    func clear() {
        distance = 0
        lastLocation = nil
    }
    
    /// 启动或恢复运动跟踪
    func executeTrackingStart() {
        guard sportState == .none || sportState == .pause else { return }
        guard currentTrack == nil else { return }
        currentTrack = SportTrack()
        currentTrack!.start()
        
        tracks.append(currentTrack!)
        if sportState == .none { startTime = currentTrack!.startTime }
        sportState = .tracking
        
        /// 语音播报
        SportRecorder.voiceReminderClearCache()
        if SportBottomView.voiceEnable { SpeechUtteranceManager.text = SRString.Sport.gps_voice_start.locastr }
    }
    
    /// 结束运动跟踪
    func executeTrackingEnd() {
        guard sportState == .tracking || sportState == .pause, let track = currentTrack else { return }
        track.end()
        endTime = track.endTime
        sportState = .finish
        /// 存储数据到数据库
        save()
        /// 语音播报
        if SportBottomView.voiceEnable { SpeechUtteranceManager.text = SRString.Sport.gps_voice_end.locastr }

    }
    
    /// 暂停运动跟踪
    func executeTrackingPause() {
        guard sportState == .tracking,  let track = currentTrack else { return }
        track.end()
        currentTrack = nil
        sportState = .pause
    }
   
    /// 加入新的跟踪位置
    func appendLocation(_ location: CLLocation) -> SportPoint? {
        guard sportState == .tracking, let track = currentTrack else { return nil}
        
        var distance: CLLocationDistance = 0
        if let hasLastLocation = lastLocation, track.points.count>0 { distance = location.distance(from: hasLastLocation) }
        let point = SportPoint(location: location, distance: distance);
        guard track.appendPoint(point) else { return nil}
        
        self.distance += point.distance
        self.lastLocation = location;
        return point
    }
    
    /// 存储数据
    func save() {
        GpxRecord.insert(record: self)
    }
}

extension SportRecorder: CustomStringConvertible {
    var description: String {
        toJSONString(prettyPrint: true)!
    }
}

// MARK: - 语音播报

extension SportRecorder {
    
    static var isImperial: Bool { BandSettings.isImperial }

    /// 运动语音播报提醒距离间隔.
    static let kReminderDistance: Double = 1000
    /// 上一次提醒的距离间隔下标
    static var lastReminderIndex: Int = 0
    /// 新开运动需要清除下缓存
    static func voiceReminderClearCache() { lastReminderIndex = 0  }

    /// 按最近一公里计算
    static func voiceReminder(_ recorder: SportRecorder) {
        
        let index = Int(recorder.distance/kReminderDistance);
        if (index > self.lastReminderIndex) {
            /// 总数据
            let distance = recorder.distance
            let duration = recorder.duration
            let durationMins: Int = Int(duration / 60);
            
            /// 平均速度（m/s）
            let speedInMps = distance / Double(duration)
            /// 提醒语句
            var reminderText = SRString.Sport.gps_voice_reminder_km.locastr
            /// 已经运动距离
            var distanceValue: Double = recorder.distance / 1000 // (km)
            /// 1km 耗时多少秒
            var paceInSeconds = Int(1000 / speedInMps);
            /// 英制处理
            if isImperial {
                /// 1英里 耗时多少秒
                paceInSeconds = Int(1609 / speedInMps)
                reminderText = SRString.Sport.gps_voice_reminder_mile.locastr
                distanceValue = BandCalculator.mile(m: BDCalculatorType(distance))
            }
            /// 分钟
            let paceMins: Int = paceInSeconds / 60
            /// 秒
            let paceSeconds: Int = paceInSeconds % 60
            ///
            let text = String(format: reminderText, distanceValue, durationMins, paceMins, paceSeconds)
            SpeechUtteranceManager.text = text
            self.lastReminderIndex = index;
        }
    }
}


// MARK: - SportTrack

/// 运动轨迹模型（具有连续性）
class SportTrack: HandyJSON {
    /// 开始时间
    var startTime: UInt32
    /// 结束时间
    var endTime: UInt32?
    /// 距离（米）
    var distance: Double = 0
    /// 轨迹点集合
    var points: [SportPoint] = []

    // MARK: Initialize
    
    required init() {
        self.startTime = 0
    }
    
    func start() {
        startTime = UInt32(Date().timeIntervalSince1970)
        distance = 0
    }
    
    func end() {
        endTime = UInt32(Date().timeIntervalSince1970)
    }
    
    func appendPoint(_ point: SportPoint) -> Bool {
        var isAdd = false
        if lastPoint == nil { isAdd = true }
        if let hasLastPoint = lastPoint,
           (point.time - hasLastPoint.time) > 2,
           point.distance > 1, point.speed > 0.01 {
            isAdd = true
        }
        if isAdd {
            points.append(point)
            distance += point.distance
        }
        return isAdd
    }
    
    // MARK: Convenience
    
    var startDate: Date {
        return Date(timeIntervalSince1970: TimeInterval(startTime));
    }
    
    var endDate: Date? {
        guard let time = endTime else { return nil }
        return Date(timeIntervalSince1970: TimeInterval(time));
    }
    
    /// 持续时间 （秒）
    var duration: Int {
        guard let hasEndTime = self.endTime else { return Int(UInt32(Date().timeIntervalSince1970) - self.startTime) }
        return Int(hasEndTime - self.startTime);
    }
    
    var firstPoint: SportPoint? {
        return points.first;
    }
    
    var lastPoint: SportPoint? {
        return points.last;
    }
    
    private func getLocations(_ closure: @escaping ((_ locations: [CLLocationCoordinate2D]) -> Void)) {
        guard points.count > 0 else { closure([CLLocationCoordinate2D]()); return }
        
        JZAreaManager.default.isOutOfArea(gcj02Point: firstPoint!.coordinate, result: { [weak self ] (isOut:Bool) in
            guard let strong = self else { return }
            var locations: [CLLocationCoordinate2D] = []
            strong.points.forEach { (e) in
                let coordinate = e.coordinate
                if isOut {
                    locations.append(coordinate)
                } else {
                    let offsetPoint = coordinate.gcj02Offset()
                    let resultPoint = CLLocationCoordinate2DMake(coordinate.latitude + offsetPoint.latitude, coordinate.longitude + offsetPoint.longitude)
                    locations.append(resultPoint)
                }
            }
            DispatchQueue.main.async {
                closure(locations)
            }
        })
    }
    
    /// 地图绘制轨迹的数据
    public func getOverlay(_ closure: @escaping ((_ overlay: MKPolyline?) -> Void))  {
        guard points.count > 0 else { closure(nil); return}
        getLocations { (locations) in
            var coords: [CLLocationCoordinate2D] = locations
            let polyline = MKPolyline(coordinates: &coords, count: coords.count)
            closure(polyline)
        }
    }
    
    var centerCoordinate: CLLocationCoordinate2D? {
        guard points.count > 0 else {  return nil}
        return CLLocationCoordinate2DMake((firstPoint!.latitude + lastPoint!.latitude)/2, (firstPoint!.longitude+lastPoint!.longitude)/2)
    }
}

extension SportTrack: CustomStringConvertible {
    var description: String {
        toJSONString(prettyPrint: true)!
    }
}

// MARK: - SportPoint

class SportPoint: HandyJSON {
    /// 确定位置时的时间戳
    var time: UInt32 = 0
    /// WGS84 维度
    var latitude: Double = 0
    /// WGS84 经度
    var longitude: Double = 0
    /// 海拔高度
    var altitude: Double = 0
    /// 该位置的速度（米/秒）。如果速度无效，则为0
    var speed: Double = 0
    /// 距离 （米）
    var distance: Double = 0
    
    
    required init() { }

    init(lat: Double, lon: Double) {
        self.latitude = lat;
        self.longitude = lon;
    }

    convenience init(location: CLLocation, distance: Double){
        self.init(lat: location.coordinate.latitude, lon: location.coordinate.longitude)
        self.altitude = location.altitude;
        self.time = UInt32(location.timestamp.timeIntervalSince1970)
        self.distance = distance
        self.speed = max(location.speed, 0)
    }
    
    public func compatible(_ closure: @escaping (_ location:CLLocationCoordinate2D) -> Void)  {
        JZLocationConverter.default.wgs84ToGcj02(coordinate, result: { (point) in
            closure(point)
        })
    }
    
    public var coordinate: CLLocationCoordinate2D { CLLocationCoordinate2D(latitude: latitude, longitude: longitude) }
}

extension SportPoint: CustomStringConvertible {
    var description: String {
        toJSONString(prettyPrint: true)!
    }
}

extension SportPoint: Comparable {
    static func < (lhs: SportPoint, rhs: SportPoint) -> Bool {
        return lhs.time < rhs.time
    }
    static func == (lhs: SportPoint, rhs: SportPoint) -> Bool {
        return lhs.time == rhs.time
    }
}
