//
//  ConnectedViewController.swift
//  CTFit
//
//  Created by Mac on 2020/9/5.
//  Copyright © 2020 jpaxh. All rights reserved.
//

import UIKit
import TableViewManager

class ConnectedViewController: BaseViewController, CustomView {

    typealias ViewClass = BaseTableView
    
    override init() {
        super.init()
        hidesBottomBarWhenPushed = true
        navigationItem.title = SRString.Connected.mydevice.locastr
    }

    override func loadView() {
        super.loadView()
        view = ViewClass()
    }
    
    override func initNavigationBar() {
        super.initNavigationBar()
    }

    override func viewDidLoad() {
        super.viewDidLoad()
        observeBandConnectState()
        setupTableView()
        addLongPressGesture()
    }
    
    override func viewWillAppear(_ animated: Bool) {
        super.viewWillAppear(animated)
        customView.refreshTableView()
    }
    
    override func viewDidDisappear(_ animated: Bool) {
        super.viewDidDisappear(animated)
    }
    
    private func setupTableView() {
        tableViewManger.reload()
        datasource.updateFeaturesClosure = { [weak self]  in
            guard let strong = self else { return }
            strong.onUpdate()
        }
    }
    
    // MARK: - TableViewManager
    
    fileprivate lazy var datasource: ConnectedDatasource = ConnectedDatasource();
    
    fileprivate lazy var deviceSection: BaseTableViewSection = {
        let section = BaseTableViewSection(customHeader: SRString.Connected.device_connected.locastr, customFooter: nil)
        section.headerHeight = 40
        section.footerHeight = 0.01
        section.add(item: datasource.deviceItem)
        return section
    } ()
    
    fileprivate lazy var featureSection: BaseTableViewSection = {
        let section = BaseTableViewSection(customHeader: SRString.Connected.device_operation.locastr, customFooter: nil)
        section.headerHeight = 40
        section.footerHeight = 0.01
        datasource.featureItems.forEach { (item) in section.add(item: item) }
        return section
    } ()
    
    fileprivate lazy var tableViewManger: ZJTableViewManager = {
        let manager = ZJTableViewManager(tableView: customView.tableView)
        manager.register(ConnectedCell.self, ConnectedItem.self)
        manager.add(section: deviceSection);
        manager.add(section: featureSection);
        return manager;
    } ()
    
}


// MARK: - 监听手环连接状态

extension ConnectedViewController {
    
    fileprivate func onUpdate() {
        featureSection.removeAllItems()
        datasource.featureItems.forEach { (item) in featureSection.add(item: item) }
        tableViewManger.reload()
    }
    
    internal override func onConnected() {  }

    internal override func onDisconnected() { navigationController?.popViewController(animated: true) }
    
}

// MARK: - 固件升级

///: - 长按5秒选择 DFU 升级固件
import FileBrowser
extension ConnectedViewController {
    
    private static var selectFile: FBFile?
    
    private static func onLongPressAction() {
        let browser = FileBrowser.inboxBrower()
        ConnectedViewController.selectFile = nil
        browser.didSelectFile = { (file)  in
            UILog.d("didSelectFile: \(file.filePath)")
            ConnectedViewController.selectFile = file
            UpgradeViewController.lastSelectFile = file
        }
    }
    
    // MARK: Private
    private func addLongPressGesture() {
        let gesture = UILongPressGestureRecognizer(target: self, action: #selector(longPressGestureAction(_:)))
        // 设定最小的长按时间 按不够这个时间不响应手势
        gesture.minimumPressDuration = 5;
        // 添加到整个View
        view.addGestureRecognizer(gesture)
    }
    
    @objc
    private func longPressGestureAction(_ gesture: UILongPressGestureRecognizer) {
        if gesture.state == .began {
            ConnectedViewController.onLongPressAction()
        }
    }
}

///: - 检查固件升级
import NEUtils
import BandKit
extension ConnectedViewController {
    
    static func upgradeCheckZ() {
        guard let peripheral = SRKitService.currentPeripheral else { return }
        
        /// 本地自选固件
        if let file = selectFile {
            UpgradeViewController.enter(peripheral: peripheral, selectFile: file)
            return
        }
        
        /// 服务器检测固件
        ToastUtils.showActivityCCV()
        let model = SRDataManager.firmwareInfo.model
        let version = Int(SRDataManager.firmwareInfo.version)
        NENetworkingManger.checkFirmwareUpdate(model: model, ver: version) { response in
            ToastUtils.hideActivityCCV()
            switch response.result {
            case .success(let text):
                if let updateInfo = SRUpdateInfo.parse(text), updateInfo.valid {
                    AlertManager.sheet(title: nil, message: SRString.Upgrade.version_checked_alert.locastr, yesHander: { (_) in
                        UpgradeViewController.enter(peripheral: peripheral, updateInfo: updateInfo)
                    }) { (_) in }
                    return
                }
            case .failure(_): break;
            }
            ToastUtils.showToastCCV(normal: SRString.Upgrade.version_latested.locastr)
        }
    }
}

