import UIKit
import TableViewManager

enum ConnectedItemType {
    case device
    case findBand, tackePicture, upgrade, reset, delete
    
    var text: String {
        switch self {
        case .device: return BandConnect.peripheralName
        case .findBand: return SRString.Connected.findband.locastr
        case .tackePicture: return SRString.Connected.take_picture.locastr
        case .upgrade: return SRString.Upgrade.update_firmwre.locastr
        case .reset: return SRString.Connected.reset.locastr
        case .delete: return SRString.Connected.delete_device.locastr
        }
    }
}

final class ConnectedItem: BaseTableViewItem {
    
    ////: - Cell source
    var text: String { return type.text }
    var detail: String {
        switch type {
        case .device: return BandData.showVersion
        default: return ""
        }
    }
    
    ///: - Initialize
    override init() {
        super.init()
        cellHeight = 55
        selectionStyle = .default
        accessoryType = .none
    }
    convenience init(_ type: ConnectedItemType) {
        self.init()
        self.type = type
        onHandler()
    }
    
    ///: - Field
    var type: ConnectedItemType = .device
}

extension ConnectedItem {

    private func onHandler(){
        setSelectionHandler { [weak self] (item) in
            guard let strong = self else { return }
            strong.didSelectedHandler()
        }
    }
    
    private func didSelectedHandler() {
        switch type {
        case .findBand: onFindBand()
        case .tackePicture: onCamera()
        case .upgrade: onUpgradeCheck()
        case .reset: onReset()
        case .delete: onDelete()
        default: return
        }
    }
    
    private func alert(_ message: String,  okHander: @escaping AlertHandler) {
        AlertManager.show(title: nil, message: message, okHander: okHander, cancelHander: { (_) in }) 
    }
    
    private func onFindBand() {
        BandCommand.findBand()
    }
    
    private func onDelete() {
        alert(SRString.Connected.delete_device_alert.locastr) { (_) in
            BandConnect.unbind()
        }
    }
    
    private func onReset() {
        alert(SRString.Connected.reset_alert.locastr) { [weak self](_) in
            guard self != nil else { return }
            BandCommand.rest()
        }
    }
    
    private func onCamera() {
        let cameraNavivc = BaseNavigationController(rootViewController: SRCameraViewController(type: .device))
        cameraNavivc.modalPresentationStyle = .fullScreen
        guard let currentVc = Helper.currentVc as? ConnectedViewController else { return }
        currentVc.present(cameraNavivc, animated: true, completion: nil)
    }

    private func onUpgradeCheck() {
        BandData.upgradeCheck()
    }
}

