//
//  SearchingBandController.swift
//  ctband
//
//  Created by Mac on 2020/4/24.
//  Copyright © 2020 shirajo. All rights reserved.
//

import UIKit
import RxSwift
import TableViewManager

class SearchedViewController: BaseViewController, CustomView {

    typealias ViewClass = BaseTableView
    
    // MARK: - override
    
    override init() {
        super.init()
        hidesBottomBarWhenPushed = true
        navigationItem.title = SRString.Searched.search_device.locastr
    }

    override func loadView() {
        super.loadView()
        view = ViewClass()
    }
    
    override func initNavigationBar() {
        super.initNavigationBar()
        datasource.searchStop()
    }

    override func viewDidLoad() {
        super.viewDidLoad()
        observeBandConnectState()
        setupTableView()
    }
    
    override func viewWillAppear(_ animated: Bool) {
        super.viewWillAppear(animated)
        customView.refreshTableView()
    }
    
    override func viewDidDisappear(_ animated: Bool) {
        super.viewDidDisappear(animated)
    }
    
    // MARK: - View customer
    private func setupTableView() {
        tableViewManger.reload()
        observe();
        customView.setRefreshHeader({ [weak self] in
            self?.datasource.searchStart()
        });
    }
    
    // MARK: - TableViewManager
    
    private lazy var datasource: SearchedDatasource = SearchedDatasource();
    
    private lazy var section: BaseTableViewSection = {
        let section = BaseTableViewSection()
        section.headerHeight = 0.01
        section.footerHeight = 0.01
        return section
    } ()
    
    private lazy var tableViewManger: ZJTableViewManager = {
        let manager = ZJTableViewManager(tableView: customView.tableView)
        manager.register(SearchedCell.self, SearchedItem.self)
        manager.add(section: section);
        return manager;
    } ()
    
    
    /// : - observe searchActionOutput

    private func observe() {
        datasource.searchActionOutput.observeOn(mainScheduler).subscribe(onNext: {[weak self] action in
            guard let strong = self else { return }
            switch action {
            case .found(let value): strong.add(found: value)
            case .end(_): strong.customView.endRefreshing()
            }
        }).disposed(by: disposeBag)
        
        datasource.connectActionOutput.observeOn(mainScheduler).subscribe(onNext: {[weak self] status in
            guard let strong = self else { return }
            strong.connectOuput(status)
        }).disposed(by: disposeBag)
    }

    private func add(found value: SearchActionType.FoundType) {
        if value.first { section.removeAllItems()}
        value.item.autoHeight(tableViewManger)
        section.insert(searched: value.item, at: value.index)
        tableViewManger.reload();
        value.item.setSelectionHandler { [weak self] (item) in
            guard let strong = self else { return }
            strong.connectAction(item: item as! SearchedItem)
        }
    }

    // MARK: - establishConnection & isDuf to Upgrade

    private var connectAction = false
    
    private func connectAction(item: SearchedItem) {
        datasource.searchStop();

        /// series_z Dfu
        if let p = item.peripheral.peripheral_z, p.isDuf {
            UpgradeViewController.enter(peripheral: p)
            return;
        }
        
        connectAction = true
        indicatorShow(message: SRString.Searched.connecting.locastr)
        BandConnect.unbind()
        DispatchQueue.main.asyncAfter(deadline: .now()+0.2) { [weak self] in
            guard let strong = self else { return }
            strong.datasource.connect(for: item.peripheral)
        }
    }
    
    private func connectOuput(_ result: BDConnectAction) {
        switch (result) {
        case .success: connectSuccess();
        case .failed(let type):
            if connectAction {
                indicatorDismiss()
                ToastUtils.showToast(error: type.text, to: view)
            }
        }
        connectAction = false;
    }
    
    private func connectSuccess() {
        AutoConnectManager.valid = false
        BandConnect.bind()
        onDidChanged()
        _onConnected()
    }
}


// MARK: - 监听手环连接状态

extension SearchedViewController {
    
    private func _onConnected() {
        if indicator.showing { indicatorDismiss() }
        guard let vc = Helper.currentVc, vc.isKind(of: SearchedViewController.self)  else { return }
        navigationController?.popViewController(animated: true)
    }
    
    internal override func onConnected() {
        guard !connectAction else { return }
        _onConnected()
    }
    
    internal override func onDisconnected() { }
}
