//
//  UpgradeViewController.swift
//  CTFit
//
//  Created by Mac on 2020/8/7.
//  Copyright © 2020 shirajo. All rights reserved.
//

import UIKit
import NEUtils
import BandKit
import iOSDFULibrary
import FileBrowser

class UpgradeViewController: BaseViewController {
    
    override init() {
        super.init()
        hidesBottomBarWhenPushed = true
        navigationItem.title = SRString.Upgrade.update_firmwre.locastr
    }
    
    override func loadView() {
        super.loadView()
    }
    
    override func initNavigationBar() {
        super.initNavigationBar();
    }

    deinit {
        SRKitService.autoConnecting = true
        SRKitService.reconnectBindPeripheral()
        UIApplication.shared.isIdleTimerDisabled = false;
    }
    
    override func viewDidLoad() {
        super.viewDidLoad()
        // Do any additional setup after loading the view.
        UIApplication.shared.isIdleTimerDisabled = true;
        view.userDarkMode()
        setConstraints()
        start()
    }
    
    override func onBack() {
        self.navigationController?.dismiss(animated: true, completion: nil)
    }
    
    // MARK: - View customer
    fileprivate lazy var cirWidth: CGFloat = {
        if UIDevice.idiom == .pad {
            return min(UIDevice.scrWidth, UIDevice.scrHeight) * 0.35
        } else {
            return min(UIDevice.scrWidth, UIDevice.scrHeight) * 0.6
        }
    } ()
    
    fileprivate func setConstraints() {
        [declareLabel, progressView, deviceNameLabel].forEach { view in
            view.translatesAutoresizingMaskIntoConstraints = false
            self.view.addSubview(view)
        }
        
        progressView.snp.makeConstraints { (make) in
            make.centerX.equalToSuperview()
            make.centerY.equalToSuperview().offset(-UIDevice.naviLayoutGuide/2)
            make.width.height.equalTo(cirWidth)
        }
        
        declareLabel.snp.makeConstraints { (make) in
            make.centerX.equalToSuperview();
            make.top.equalToSuperview().offset(Constraints.medium)
            if UIDevice.idiom == .pad {
                make.width.equalToSuperview().multipliedBy(0.75)
            } else {
                make.width.equalToSuperview().offset(-Constraints.medium*2)
            }
        }
        
        deviceNameLabel.snp.makeConstraints { (make) in
            make.centerX.equalToSuperview();
            make.top.equalTo(progressView.snp.bottom).offset(Constraints.medium)
        }
    }
    
    fileprivate lazy var progressView: ProgressView = {
        let view = ProgressView(frame: CGRect(x: 0, y: 0, width: cirWidth, height: cirWidth))
        view.addTarget(self, action: #selector(progressTap), for: .touchUpInside)
        return view
    } ()
    
    fileprivate lazy var deviceNameLabel: UILabel = {
        let label = UILabel(style: Stylesheet.Label.text)
        label.font = .customFont(ofSize: 17)
        label.text = peripheral.peripheralName
        return label
    }()
    
    fileprivate lazy var declareLabel: UILabel = {
        let label = UILabel(style: Stylesheet.Label.text)
        label.textColor = Colors.white.withAlphaComponent(0.7)
        label.font = .customFont(ofSize: 14)
        label.textAlignment = .left
        label.text = SRString.Upgrade.update_firmwre_declare.locastr
        return label
    } ()
    
    // MARK: - Field
    fileprivate let dfuService: SRDfuService = SRDfuService.shared
    
    fileprivate var peripheral: SRPeripheral!
    fileprivate var selectFile: FBFile?
    fileprivate var updateInfo: SRUpdateInfo?
    fileprivate var vaild: Bool {
        if selectFile == nil,  updateInfo == nil { return false }
        if peripheral === SRKitService.currentPeripheral, !SRKitService.isConnected {
            ToastUtils.showToastCCV(error: SRString.Base.disconnect_band.locastr)
            return false
        }
        return true
    }
}
 
// MARK: - service
extension UpgradeViewController {
    
    fileprivate func start() {
        statusText(type: .loading)
        if selectFile == nil { downloadFile(); return }
        if peripheral!.isDuf { uploadFile(); return }
        enterToDfuMode()
    }
    
    fileprivate func downloadFile() {
        guard let serverUrl = updateInfo?.FileURL else { return }
        progressView.progress = 0
        statusText(type: .downloading)
        NENetworkingManger.downloadFirmware(serverUrl, { [weak self] (response) in
            guard let strong = self else { return }
            switch response.result {
            case .success(_):
                if let filePath = response.destinationURL {
                    strong.selectFile = FBFile(filePath: filePath)
                    strong.start()
                } else {
                    strong.alert(type: .failedDownload)
                }
            case .failure(_): strong.alert(type: .failedDownload)
            }
        }) { [weak self] (progress) in
            guard let strong = self else { return }
            strong.progressView.progress = CGFloat(progress.fractionCompleted)
        }
    }
    
    fileprivate func enterToDfuMode() {
        SRKitService.autoConnecting = false
        statusText(type: .enterUpgradeMode)
        dfuService.switchToDfuMode { (_, error) in }
        searchDfuDevice()
    }
    
    fileprivate func searchDfuDevice() {
        dfuService.startScanning(peripheralClosure: { [weak self] (peripheral) -> Bool in
            guard let strong = self, peripheral.isDuf else { return false }
            strong.peripheral = peripheral
            return true
        }) { [weak self] (status) in
            guard let strong = self else { return }
            switch status {
            case .searching: strong.statusText(type: .findUpgradeDevice)
            case .discovered:  strong.uploadFile()
            case .notFound: strong.alert(type: .failedFindDevice)
            }
        }
    }
    
    fileprivate func uploadFile() {
        SRKitService.autoConnecting = false
        guard let file = selectFile, let device = peripheral, device.isDuf else { return }
        dfuService.startUpLoadFMFile(filePath: file.filePath, peripheral: device, progressClosure: { [weak self] (progress, _, _) in
            guard let strong = self else { return }
            strong.progressView.progress = CGFloat(progress) / 100
        }, dfuStatusClosure: { [weak self] (state) in
            guard let strong = self else { return }
            switch state {
            case .starting:
                UpgradeViewController.lastSelectFile = file
                UpgradeViewController.lastDfuPeripheral = device
            case .uploading: strong.statusText(type: .uploading)
            case .aborted: strong.alert(type: .failedUpload)
            case .completed: UpgradeViewController.lastDfuPeripheral = nil; strong.alert(type: .successd)
            default : break
            }
        }) { [weak self] (error, _) in
            guard let strong = self else { return }
            if error == .fileInvalid { strong.alert(type: .failedInvalidFile); return }
            strong.alert(type: .failedUpload)
        }
    }
}

extension UpgradeViewController {
    @objc fileprivate func progressTap() {
        // progressView.progress = CGFloat(random())
    }
    fileprivate enum FMUProgressStatusType {
        case loading, downloading, enterUpgradeMode, findUpgradeDevice, uploading
        case success
        var text: String {
            switch self {
            case .loading: return SRString.Upgrade.loading.locastr
            case .downloading: return SRString.Upgrade.fm_downloading.locastr
            case .enterUpgradeMode: return SRString.Upgrade.enter_upgrade_mode.locastr
            case .findUpgradeDevice: return SRString.Upgrade.find_upgrade_device.locastr
            case .uploading: return SRString.Upgrade.fm_uploading.locastr
            case .success: return SRString.Upgrade.upgrade_successed.locastr
            }
        }
    }
    
    fileprivate enum FMUAlertType {
        case successd
        case failedDownload, failedInvalidFile, failedUpload, failedFindDevice
    }
    
    fileprivate func statusText(type: FMUProgressStatusType) {
        progressView.statusText = type.text
        UILog.v("statusText: \(type.text)")
    }
    
    fileprivate func alert(type: FMUAlertType) {
        var title: String? = nil
        var message: String? = nil
        switch type {
        case .successd:
            title = SRString.Upgrade.upgrade_successed.locastr
            message = SRString.Connected.unbind_alert.locastr
            statusText(type: .success)
        case .failedDownload:
            title = SRString.Upgrade.upgrade_failed.locastr
            message = SRString.Upgrade.fm_download_failed_alert.locastr
        case .failedInvalidFile:
            title = SRString.Upgrade.upgrade_failed.locastr
            message = SRString.Upgrade.fm_invalid_file_alert.locastr
        case .failedUpload:
            title = SRString.Upgrade.fm_upload_interrupt.locastr
            message = SRString.Upgrade.fm_upload_retry_alert.locastr
        case .failedFindDevice:
            title = SRString.Upgrade.upgrade_failed.locastr
            message = SRString.Upgrade.find_upgrade_device_failed_alert.locastr
        }
        self.navigationItem.leftBarButtonItem = backBarItem()
        if type == .failedUpload {
            AlertManager.show(title: title, message: message, retryHander: { [weak self] (_) in
                guard let strong = self else { return }
                strong.uploadFile()
                strong.navigationItem.leftBarButtonItem = nil
            }) { [weak self] (_) in
                guard let strong = self else { return }
                strong.onBack()
            }
        } else {
            self.navigationItem.leftBarButtonItem = backBarItem()
            AlertManager.show(title: title, message: message) { [weak self] (_) in
                guard let strong = self else { return }
                strong.onBack()
            }
        }
    }
}

extension UpgradeViewController {
    fileprivate static func enter(peripheral: SRPeripheral, updateInfo: SRUpdateInfo?, selectFile: FBFile?) {
        let vc = UpgradeViewController()
        vc.peripheral = peripheral
        vc.selectFile = selectFile
        vc.updateInfo = updateInfo
        guard vc.vaild else { return }
        
        let navVc = BaseNavigationController(rootViewController: vc)
        navVc.modalPresentationStyle = .fullScreen
        Helper.rootVc?.present(navVc, animated: true, completion: nil)
    }
    
    static func enter(peripheral: SRPeripheral, updateInfo: SRUpdateInfo) {
        enter(peripheral: peripheral, updateInfo: updateInfo, selectFile: nil)
    }
    
    static func enter(peripheral: SRPeripheral, selectFile: FBFile) {
        enter(peripheral: peripheral, updateInfo: nil, selectFile: selectFile)
    }
    
    static func enter(peripheral: SRPeripheral) {
        guard let lastSelectFile = UpgradeViewController.lastSelectFile else { return }
        enter(peripheral: peripheral, updateInfo: nil, selectFile: lastSelectFile)
    }
    
    static func enter() {
        guard let peripheral = UpgradeViewController.lastDfuPeripheral else { return }
        guard let lastSelectFile = UpgradeViewController.lastSelectFile else { return }
        enter(peripheral: peripheral, updateInfo: nil, selectFile: lastSelectFile)
    }
}


import FileBrowser
extension UpgradeViewController {
    fileprivate static let lastSelectFileKey = "LastSelectFileKey"
    fileprivate static var _lastSelectFile: FBFile?
    static var lastSelectFile: FBFile? {
        get {
            if let file = _lastSelectFile { return file}
            guard let url = UserDefaults.standard.url(forKey: lastSelectFileKey) else { return nil }
            var newUrl = url
            
            if let homeIndex = url.pathComponents.firstIndex(of: "Application"),
                url.pathComponents.count > homeIndex+1 {
                let curHomeString = FileBrowser.homePath.lastPathComponent
                let oldHomeString = url.pathComponents[homeIndex+1]
                if curHomeString != oldHomeString, curHomeString.count==oldHomeString.count {
                    let path = url.relativePath.replacingOccurrences(of: oldHomeString, with: curHomeString)
                    newUrl = URL(fileURLWithPath: path)
                }
            }
            _lastSelectFile = FBFile(filePath: newUrl)
            return _lastSelectFile
        }
        set {
            guard let url = newValue?.filePath else { return }
            UserDefaults.standard.set(url, forKey: lastSelectFileKey)
            UserDefaults.standard.synchronize()
            _lastSelectFile = FBFile(filePath: url)
        }
    }
}

extension UpgradeViewController {
    fileprivate static let dfuService = SRDfuService.shared
    fileprivate static let lastDfuPeripheralKey = "lastDfuPeripheralKey"
    fileprivate static var _lastDfuPeripheral: SRPeripheral?

    static var lastDfuPeripheral: SRPeripheral? {
        get {
            if let peripheral = _lastDfuPeripheral { return peripheral}
            guard let dict = UserDefaults.standard.dictionary(forKey: lastDfuPeripheralKey) else { return nil }
            guard let uuidString = dict["uuid"] as? String, let name = dict["name"] as? String else { return nil }
            guard let uuid = UUID(uuidString: uuidString) else { return nil }
            _lastDfuPeripheral = dfuService.retrievePeripherals(withIdentifier: uuid)
            _lastDfuPeripheral?.peripheralName = name
            return _lastDfuPeripheral
        }
        set {
            _lastDfuPeripheral = newValue
            if let peripheral = _lastDfuPeripheral {
                let dict = ["uuid": peripheral.uuid, "name": peripheral.peripheralName]
                UserDefaults.standard.set(dict, forKey: lastDfuPeripheralKey)
            } else {
                UserDefaults.standard.removeObject(forKey: lastDfuPeripheralKey)
            }
            UserDefaults.standard.synchronize()
        }
    }
}
