//
//  SRDataManager.swift
//  BandKit
//
//  Created by Mac on 2020/5/16.
//

import Foundation
import Repeater
import RxSwift
import NEUtils


// MARK: - 数据类型
public enum SRDataType {
    case time, firmwareInfo, userInfo, clockInfo, drinkInfo
    case record(SRRecordType, SRRecordTn, SRFlashDateItem)
    case pair(SRConnectType)
    /// detected：ture - 结束检测
    case detectInfo([(type: SRDetectType, detected: Bool)])
    /// 通知
    case notify(SRNotifyType)
}

// MARK: - 同步标记
struct SyncFlay {
    static var firmwareInfo: Bool = false
    static var userInfo: Bool = false
    static var clockInfo: Bool = false
    static var drinkInfo: Bool = false
    static func clear() {
        firmwareInfo = false
        userInfo = false
        clockInfo = false
        drinkInfo = false
    }
}

// MARK: - SRDataManager

public class SRDataManager {
    /// 更新数据
    private let updateSubject = PublishSubject<SRDataType>()
    private var updatedOutput: Observable<SRDataType> {
        return updateSubject.asObservable()
    }
    
    /// 信息数据
    private(set) var deviceTime: SRDeviceTime = SRDeviceTime.readForCache() {
        didSet { deviceTime.saveToCache() }
    }
    private(set) var firmwareInfo: SRFirmwareInfo = SRFirmwareInfo.readForCache() {
        didSet { firmwareInfo.saveToCache() }
    }
    private var userInfo: SRUserInfo = SRUserInfo.readForCache() {
        didSet { userInfo.saveToCache() }
    }
    private var clockInfo: SRAlarmClock = SRAlarmClock.readForCache() {
        didSet { userInfo.saveToCache() }
    }
    private var drinkInfo: SRDrinkInfo = SRDrinkInfo.readForCache() {
        didSet { userInfo.saveToCache() }
    }
    
    /// 记录数据
    private lazy var detectInfo: SRDetectInfo = SRDetectInfo()

    private lazy var flashDates: SRFlashDates = SRFlashDates()
    private lazy var todayTotal: SRTotal = SRTotal.queryToday()
    private lazy var todayStep:  SRStep  = SRStep.queryToday()
    private lazy var todayHeart: SRHeart = SRHeart.queryToday()
    private lazy var todaySleep: SRSleep = SRSleep.queryToday()
    private lazy var todayExercise: SRExercise = SRExercise.queryToday()
    
    /// Field
    private let disposeBag = DisposeBag()
    private var iOSPair: Bool = false
    private var pairiOS_errorCount: Int = 0;


    // MARK: - Initialize
    private init() {
         _ = NEWeatherTask.shared
        observes()
        if SRKitService.isConnected { toPairiOS() }
    }
    
    ///: - 检查配对
    private lazy var pairRepeater: Repeater = {
        let array: [SRRequestError] = [.timeout, .noreponse]
        let repeater = Repeater.every(.seconds(2.0)) { (repeater) in
            SRCommand.pairiOS().writeToDevice { (value, error) in
                if let requestError = error as? SRRequestError,  !array.contains(requestError)  { return }
                SRCommand.readPairWay().writeToDevice()
            }

        }
        repeater.pause()
        return repeater
    }()
    
    private func toPairiOS() {
        iOSPair = false
        pairiOS_errorCount = 0;
        CmdRequest.clear()
        SyncFlay.clear()
        SRCommand.readPairWay().writeToDevice()
        pairRepeater.start()
    }
    
    private func endPairiOS() {
        CmdRequest.clear()
        pairRepeater.pause()
    }
    
    ///: -  使用高速同步数据
    private lazy var transferSpeedRepeater: Repeater = {
        let repeater = Repeater.once(after: .seconds(20)) { (_) in
            SRCommand.transferSpeed(.low).style(.push).writeToDevice()
        }
        repeater.pause()
        return repeater
    }()
    private func useHighSpeedSync() {
        SRCommand.transferSpeed(.high).writeToDevice()
        transferSpeedRepeater.start()
    }
    
    ///: -  挂起唤醒： 执行同步
    private func performWakeUpSync() {
        BandLog.i("----------------- Wakeup sync data ----------------------")
        SRCommand.syncTime(deviceTime).writeToDevice()
        weatherStart()
    }
    
    
    ///: -  连接时同步一次
    private func toSyncDatasNormal() {
        BandLog.i("----------------- Start sync data ----------------------")
        SRHealthStore.requestAuthorization()
        NEWeatherTask.lastPushTime = 0
        useHighSpeedSync() // 使用高数模式同步数据
        SRCommand.syncTime(deviceTime).writeToDevice()
        SRCommand.readRecordDetail(.total, .t0).writeToDevice { (_, _) in }
        SRCommand.readFirmwareInfo().writeToDevice()
        SRCommand.readUserInfo().writeToDevice()
        SRCommand.readAlarmClock().writeToDevice()
    }
    
    ///: -  同步当天记录详情
    private func toSyncDatasToday() {
        if todayTotal.steps != todayStep.total { SRCommand.readRecordDetail(.step, .t0).writeToDevice() }
        if todayTotal.latestHeart != todayHeart.latest { SRCommand.readRecordDetail(.heart, .t0).writeToDevice() }
        if todayTotal.sleepMinutes != todaySleep.totalMinutes { SRCommand.readRecordDetail(.sleep, .t0).writeToDevice() }
        if todayTotal.exerciseMinutes != todayExercise.totalMinutes { SRCommand.readRecordDetail(.exercise, .t0).writeToDevice() }
    }
    
    ///: -  同步历史记录
    private func toSyncDatasHistory() {
        for (idx, dateItem) in flashDates.list.enumerated() {
            if BandRecord.queryFlag(dateItem.dateStr) != BandRecord.Columns.maskAll,
                let tn = SRRecordTn.parseFlashListIndex(idx+1)  {
                BandRecord.clear(dateStr: dateItem.dateStr)
                SRCommand.readRecordDetail(.total, tn).writeToDevice()
            }
        }
    }
}


extension SRDataManager {
    
    ///: - 监听
    private func observes() {
        
        /// 准备就绪
        SRKitService.isNotifyingOutput.subscribe(onNext: {[weak self] ready in
            guard let strong = self, ready else {return }
            strong.toPairiOS()
        }).disposed(by: disposeBag)
        
        SRKitService.connectionStateOutput.subscribe(onNext: {[weak self] state in
            guard let strong = self else {return }
            switch state {
            case .connected:  break;
            case .disconnected(_): strong.weatherPause(); strong.endPairiOS();
            }
        }).disposed(by: disposeBag)
        
        NEAppNotifyUtils.observable.subscribe(onNext: { [weak self] type in
            guard let strong = self else {return }
            switch type {
            case .willEnterForeground:
                strong.restTodayRecords()
                if SRKitService.isConnected { strong.toPairiOS() }
            case .significantTimeChange: strong.weatherStart()
            default: break
            }
        }).disposed(by: disposeBag)
        
        
        NEWeatherTask.updatedPushOutput.subscribe(onNext: {[weak self] (_) in
            guard let strong = self else {return }
            strong.toPushWeaterInfo()
        }).disposed(by: disposeBag)
        
        CmdRequest.receivedOutput.subscribe(onNext: { [weak self] (reponse) in
            guard let strong = self else {return }
            strong.received(reponse)
        }).disposed(by: disposeBag)
    }
}

///: - 天气同步
extension SRDataManager {

    private func weatherStart() {
        guard SRKitService.isConnected, firmwareInfo.feature.weather else { return }
        NEWeatherTask.start()
    }
    
    private func weatherPause() {
        NEWeatherTask.pause()

    }
    
    private func toPushWeaterInfo() {
        guard SRKitService.binded else { return }
        guard SRKitService.isConnected,
              let hourCmd = SRCommand.weatherForecastHour() else { NEWeatherTask.isActive = false; return }
        NEWeatherTask.reverseGeocodecityName {
            if NEWeatherInfo.currentHourItem != nil  { SRCommand.syncTime(SRDeviceTime()).writeToDevice() }
            if let cmd = SRCommand.weatherForecastDay() { cmd.writeToDevice() }
            hourCmd.writeToDevice({ (obj, error) in
                if error == nil { NEWeatherTask.lastPushTime = Date().timeIntervalSince1970 }
                if let rpError = error as? SRReponseError {
                    switch rpError {
                    case .nosupport, .param, .unknown, .notfound: NEWeatherTask.lastPushTime = Date().timeIntervalSince1970
                    default: break
                    }
                }
                NEWeatherTask.isActive = false
            })
        }
    }
}

// MARK: - 接收数据处理
extension SRDataManager {
    
    ///: - 接收到的数据
    private func received(_ reponse: CmdReponse) {
        guard reponse.code == .success else { return }
        if reponse.cmd >= 0x80 { receivedForRead(reponse); return }
        receivedForWrite(reponse)
    }
    
    ///: - 读取到的数据
    private func receivedForRead(_ reponse: CmdReponse) {
        guard let type = reponse.type else { return }
        let bytes = [UInt8](reponse.data ?? Data())
        
        /// 配对方式
        if type == .devPair {
            if bytes.count==2, iOSPair==false {
                if bytes[1] == SRConnectType.ios.value {
                    endPairiOS()
                    iOSPair = true
                    pairiOS_errorCount = 0
                    toSyncDatasNormal()
        
                } else {
                    pairiOS_errorCount += 1;
                    if pairiOS_errorCount >= 5 { SRKitService.disconnect() }
                }
            }
        }
        
        /// 固件 信息
        if type == .devFirmware {
            guard let model = reponse.model as? SRFirmwareInfo else { return }
            firmwareInfo = model
            updateSubject.onNext(.firmwareInfo)
            if !SyncFlay.firmwareInfo {
                SyncFlay.firmwareInfo = true
                /// 是否需要推送天气
                weatherStart()
            }
        }
        
        /// 用户信息
        if type == .devUser {
            guard let model = reponse.model as? SRUserInfo else { return }
            userInfo = model.copy()
            updateSubject.onNext(.userInfo)
            if !SyncFlay.userInfo {
                SyncFlay.userInfo = true
                
                /// 是否需要更新：语言、华摄度 Eg.
                userInfo.format.lanType = SRLanType.app
                if SRUserInfo.country_US, SRDataManager.addUUID() {
                    userInfo.format.tempUnit = .fahrenheit
                }
                if model.data() != userInfo.data() {
                    SRCommand.setUserInfo(userInfo).writeToDevice()
                }
            }
        }
        
        /// 喝水提醒指令
        if type == .devDrink {
            guard let model = reponse.model as? SRDrinkInfo else { return }
            drinkInfo = model
            updateSubject.onNext(.drinkInfo)
            if !SyncFlay.drinkInfo {
                SyncFlay.drinkInfo = true
            }
        }
        
        
        /// 闹钟信息
        if type == .devClock {
            guard let model = reponse.model as? SRAlarmClock else { return }
            clockInfo = model
            updateSubject.onNext(.clockInfo)
            if !SyncFlay.clockInfo {
                SyncFlay.clockInfo = true
                /// 是否有喝水提醒功能
                if firmwareInfo.feature.waterRemind ||
                    firmwareInfo.feature.dateformat_detect_brightScrren {
                    SRCommand.readDrinkInfo().writeToDevice()
                }
                /// 是否需要获取检测数据.
                if firmwareInfo.feature.dateformat_detect_brightScrren { SRCommand.readDetectRecords().writeToDevice()  }
                /// 开启当天记录数据同步
                toSyncDatasToday()
                /// 开启历史数据同步
                SRCommand.readFlashDates().writeToDevice()
            }
        }
        
        /// Flash 存储 日期
        if type == .devFlashDates {
            guard let model = reponse.model as? SRFlashDates else { return }
            flashDates = model
            toSyncDatasHistory()
        }
        
        /// 详细记录数据
        if  case .devRecord(_, _) = reponse.type {
            receivedForRecordDetail(reponse)
        }
        
        /// 检测记录
        if  case .devRecordDetect(_) = reponse.type {
            receivedForRecordDetect(reponse)
        }
        
        /// 事件通知
        if  case .devNotify(_) = reponse.type {
            receivedForNotify(reponse)
        }
    }
    
    
    ///: - 详细记录数据
    private func receivedForRecordDetail(_ reponse: CmdReponse) {
        guard let type = SRRecordType.parse(reponse.ext),
              let tn = SRRecordTn.parse(reponse.ext),
              let data = reponse.data,
              data.count >= 4 else { return }
        
        /// 应答数据的日期 字段
        var checkDateWrong = true
        let dateItem = SRFlashDateItem()
        dateItem.parse(data[0...3])
        
        /// 是当天 还是 历史 ？
        let isToday = tn == .t0 ? true : false
        ///: - 当天日期项
        if isToday {
            let nowDateItem = SRFlashDateItem()
            if nowDateItem == dateItem { checkDateWrong = false }
            if checkDateWrong {
                BandLog.e("当天数据日期有误：(\(dateItem))");
                return
            }
            if let model = reponse.model as? SRTotal { todayTotal = model }
            if let model = reponse.model as? SRStep  { todayStep  = model}
            if let model = reponse.model as? SRHeart { todayHeart = model}
            if let model = reponse.model as? SRSleep { todaySleep = model}
            if let model = reponse.model as? SRExercise { todayExercise = model }
            /// 后台蓝牙唤醒，提交天气跟新任务
            DispatchQueue.main.async {
                if type == .total, NELocationManager.runningInBackground() { self.weatherStart() }
            }
        }
        
        /// 历史记录：对应 flash 日期项
        if !isToday {
            if let tIdx = tn.valueFlashListIndex,
                tIdx <  flashDates.list.count,
                flashDates.list[tIdx] == dateItem {
                checkDateWrong = false
            }
            if checkDateWrong {
                BandLog.e("历史数据日期有误：( \(dateItem) -> \(flashDates) )")
                return;
            }
        }
        
        /// 记录数据 写入苹果健康
        if let model = reponse.model as? SRStep  { SRHealthStore.save(step:  model) }
        if let model = reponse.model as? SRHeart { SRHealthStore.save(heart: model) }
        if let model = reponse.model as? SRSleep { SRHealthStore.save(sleep: model) }
        /// 记录数据 数据库保存
        BandRecord.save(dateItem.dateStr, type.column, reponse.data, isToady: isToday)
        /// onNext
        updateSubject.onNext(.record(type, tn, dateItem))

        /// 历史详细记录数据
        /// 通过总数据值来确定是否需要获取详细记录数据
        /// 值为零，则不需要获取，将对应标志位掩码置为 1
        /// 值不为零， 则去获取详细记录数据
        if !isToday, type == .total, let total: SRTotal = reponse.model as? SRTotal {
            var flag = BandRecord.Columns.total.mask
            if total.steps == 0 { flag = flag | BandRecord.Columns.step.mask }
            if total.latestHeart == 0 { flag = flag | BandRecord.Columns.heart.mask }
            if total.sleepMinutes == 0 { flag = flag | BandRecord.Columns.sleep.mask }
            if total.exerciseMinutes == 0 { flag = flag | BandRecord.Columns.exercise.mask }
            BandRecord.save(dateItem.dateStr, .flag, flag, isToady: isToday)
            
            if total.steps != 0 {  SRCommand.readRecordDetail(.step, tn).writeToDevice() }
            if total.latestHeart != 0 {  SRCommand.readRecordDetail(.heart, tn).writeToDevice() }
            if total.sleepMinutes != 0 {  SRCommand.readRecordDetail(.sleep, tn).writeToDevice() }
            if total.exerciseMinutes != 0 {  SRCommand.readRecordDetail(.exercise, tn).writeToDevice() }
        }
        
    }
    
    ///: - 检测记录数据
    private func receivedForRecordDetect(_ reponse: CmdReponse) {
        var saveItems = [(SRDetectItem)]()
        if let detectInfo = reponse.model as? SRDetectInfo {
            let items = (detectInfo.heartRateItems + detectInfo.bloodPressureItems).sorted { $0.time < $1.time }
            saveItems = items.save()
        } else if let item = reponse.model as? SRDetectItem {
            if item.end {
                updateSubject.onNext(.detectInfo([(item.type, true)]))
            } else {
                saveItems = item.save()
            }
        }
        
        var types = [(SRDetectType, Bool)]()
        SRDetectType.collection.forEach { (type) in
            if saveItems.contains(where: { (item) -> Bool in item.type==type}) {
                types.append((type, false))
            }
        }
        if types.count > 0 { updateSubject.onNext(.detectInfo(types)) }
    }
    
    ///: - 通知数据 deviceNotify： 0x8C
    private func receivedForNotify(_ reponse: CmdReponse) {
        guard case .devNotify(let type) = reponse.type else { return }
        updateSubject.onNext(.notify(type))
    }
    
    
    ///: - 写入数据应答
    private func receivedForWrite(_ reponse: CmdReponse) {
    
    }
}

/// : - 记录已连接过的设备
/// 方便作为设备首次连入时，对设备默认初始值设置
extension SRDataManager {
    private static let uuidsKey = "BandKituuidsKey"
    private static var _uuids: [String]?
    private static var uuids: [String] {
        get {
            if let array = _uuids { return array}
            guard let array = UserDefaults.standard.array(forKey: uuidsKey) as? [String] else { _uuids = []; return [] }
            _uuids = array
            return array
        }
        set {
            _uuids = newValue
            UserDefaults.standard.set(newValue, forKey: uuidsKey)
            UserDefaults.standard.synchronize()
        }
    }
    
    /// : - 添加 uuid
    /// return： - 返回 false,  表示已添加;  true 添加成功
    private static func addUUID() -> Bool {
        guard let uuid: String = SRKitService.currentPeripheral?.uuid else { return false }
        if uuids.contains(uuid) { return false }
        uuids.append(uuid)
        return true
    }
    
    /// ：- 移除 uuid
    /// return： - 返回 false,  表示没有当前设备;  true 删除成功
    private static func removeUUID() -> Bool {
        guard let uuid: String = SRKitService.currentPeripheral?.uuid else { return false }
        guard let indx = uuids.firstIndex(of: uuid) else { return false }
        uuids.remove(at: indx)
        return true
    }
}


import RxBluetoothKit
extension SRDataManager {
    ///: - 重置当天记录数据
    private func restTodayRecords(){
        let dateItem = SRFlashDateItem()
        if todayTotal.date != dateItem {
            todayTotal = SRTotal.queryToday()
            updateSubject.onNext(.record(.total, .t0, dateItem))
        }
        if todayStep.date != dateItem {
            todayStep = SRStep.queryToday()
            updateSubject.onNext(.record(.step, .t0, dateItem))
        }
        if todayHeart.date != dateItem {
            todayHeart = SRHeart.queryToday()
            updateSubject.onNext(.record(.heart, .t0, dateItem))
        }
        if todaySleep.date != dateItem {
            todaySleep = SRSleep.queryToday()
            updateSubject.onNext(.record(.sleep, .t0, dateItem))
        }
        if todayExercise.date != dateItem {
            todayExercise = SRExercise.queryToday()
            updateSubject.onNext(.record(.exercise, .t0, dateItem))
        }
    }
    
    ///: - 清除数据信息
    private func clearInfo() {
        deviceTime = SRDeviceTime()
        firmwareInfo = SRFirmwareInfo()
        userInfo = SRUserInfo()
        clockInfo = SRAlarmClock()
        drinkInfo = SRDrinkInfo()
        detectInfo = SRDetectInfo()
    }
    
    ///: - 清除已读设置记录标志
    private func clearReaded() {
        SRDeviceTime.readed = false
        SRFirmwareInfo.readed = false
        SRUserInfo.readed = false
        SRAlarmClock.readed = false
        SRDrinkInfo.readed = false
    }
    
    ///: - 解除绑定设备时调用
    private func unbindClear() {
        clearInfo()
        clearReaded()
    }
    
    ///: - 删除用户数据时调用
    private func deletePersonalData() {
        BandRecord.remove()
        DetectRecord.remove()
        DetectRecordHeart.remove()
        DetectRecordBloodPressure.remove()
        Logging.clearFiles()
    }
}

// MARK: - Class for public
extension SRDataManager {
    /// 单例
    static let shared: SRDataManager = SRDataManager()

    ///: - Fields
    public static var firmwareInfo: SRFirmwareInfo { shared.firmwareInfo }
    public static var userInfo: SRUserInfo {
        get { shared.userInfo }
        set { shared.userInfo = newValue}
    }
    public static var clockInfo: SRAlarmClock {
        get { shared.clockInfo }
        set { shared.clockInfo = newValue}
    }
    public static var drinkInfo: SRDrinkInfo {
        get { shared.drinkInfo }
        set { shared.drinkInfo = newValue}
    }
    
    ///: - output 更新数据
    public static var updatedOutput: Observable<SRDataType> { shared.updateSubject }
    

    ///: - 解除绑定设备时调用
    public static func unbindClear() { shared.unbindClear() }
    ///: - 删除用户数据时调用
    public static func deletePersonalData() { shared.deletePersonalData() }
    ///: - 后台挂起唤醒要执行的操作
    public static func performWakeUp() {
        guard NELocationManager.runningInBackground() else { return }
        if SRKitService.isConnected {
            shared.performWakeUpSync()
        } else {
            SRKitService.reconnectBindPeripheral()
        }
    }
}

