//
//  SRHealthStore.swift
//  BandKit
//
//  Created by Mac on 2021/1/18.
//

import Foundation

public typealias HealthStoreClosure = (Bool, Error?) -> Void

enum SRHealthError: Error {
    case unavailable
}

#if HEALTHEANBLE

import HealthKit

enum SRHealthSampleType {
    case step, calories, distance, heart, sleep
    var sampleType: HKSampleType {
        switch self {
        case .step:     return HKSampleType.quantityType(forIdentifier: .stepCount)!
        case .calories: return HKSampleType.quantityType(forIdentifier: .activeEnergyBurned)!
        case .distance: return HKSampleType.quantityType(forIdentifier: .distanceWalkingRunning)!
        case .heart:    return HKSampleType.quantityType(forIdentifier: .heartRate)!
        case .sleep:    return HKSampleType.categoryType(forIdentifier: .sleepAnalysis)!
        }
    }
    
    func quantity(of value: Double) -> HKQuantity? {
        guard value > 0 else { return nil }
        switch self {
        case .step:     return HKQuantity(unit: HKUnit.count(), doubleValue: value)
        case .calories: return HKQuantity(unit: HKUnit.kilocalorie(), doubleValue: value)
        case .distance:
            if value > 10 {
                return HKQuantity(unit: HKUnit.meterUnit(with: .kilo), doubleValue: value/1000)
            } else {
                return HKQuantity(unit: HKUnit.meterUnit(with: .deca), doubleValue: value/10)
            }
        case .heart:    return HKQuantity(unit: HKUnit.count().unitDivided(by: HKUnit.minute()), doubleValue: value)
        case .sleep:    return nil
        }
    }
    
    static var shareTypes: Set<HKSampleType> = {
        let collection = [step, calories, distance, heart, sleep]
        return Set(collection.map { $0.sampleType })
    } ()
}

public class SRHealthStore {
            
    private var available: Bool { HKHealthStore.isHealthDataAvailable() }

    private lazy var store: HKHealthStore  = HKHealthStore()
    
    private var authorEnable: Bool = false
    
    private init() { }
    
    /// 谓词过滤：指定日期
    private func predicateDay(for date: Date) -> NSPredicate {
        let calendar = SRCalendarUtils.calendar
        let startDate = calendar.startOfDay(for: date)
        let endDate = calendar.date(byAdding: .day, value: 1, to: startDate)
        return HKQuery.predicateForSamples(withStart: startDate, end: endDate, options: .strictStartDate)
    }
    
    private func predicateDaySleep(for date: Date) -> NSPredicate {
        let calendar = SRCalendarUtils.calendar
        let startDate = calendar.date(byAdding: .hour, value: -4, to: calendar.startOfDay(for: date))!
        let endDate = calendar.date(byAdding: .hour, value: 16, to: startDate)!
        return HKQuery.predicateForSamples(withStart: startDate, end: endDate, options: .strictStartDate)
    }
    
    /// 移除上次保存的数据
    private func removeDay(of type: SRHealthSampleType, for date: Date, closure: HealthStoreClosure? = nil) {
        var predicate: NSPredicate
        if type == .sleep { predicate = predicateDaySleep(for: date)}
        else { predicate = predicateDay(for: date) }
        store.deleteObjects(of: type.sampleType, predicate: predicate) { (success, count, error) in
            closure?(success, error)
        }
    }
    
    /// Step \ Calories \ Distance \ HeartRate
    private func quantitySample(of type: SRHealthSampleType, with value: Double,
                                from start: Date, to end: Date) -> HKQuantitySample? {
        let calendar = SRCalendarUtils.calendar
        let endDate = calendar.date(byAdding: .second, value: -1, to: end)!
        if let quantityType = type.sampleType as? HKQuantityType, let quantity = type.quantity(of: value) {
            return HKQuantitySample(type: quantityType, quantity: quantity, start: start, end: endDate)
        }
        return nil
    }
    
    /// Sleep
    private func categorySample(sleep type: SRSleepType,
                                from start: Date, to end: Date) -> HKCategorySample? {
        
        guard type != .none else { return nil}
        if let categoryType = SRHealthSampleType.sleep.sampleType as? HKCategoryType {
            var value: HKCategoryValueSleepAnalysis = .inBed
            switch type {
            case .light, .deep: value = .asleep
            default: if #available(iOS 10.0, *) { value = .awake }
            }
            return HKCategorySample(type: categoryType, value: value.rawValue, start: start, end: end)
        }
        return nil
    }
}

extension SRHealthStore {
    
    /// 请求授权
    private func requestAuthorization(closure: HealthStoreClosure? = nil) {
        guard available else { closure?(false, SRHealthError.unavailable); return }
        store.requestAuthorization(toShare: SRHealthSampleType.shareTypes, read: nil, completion: { [weak self] (success, error) in
            guard let strong = self else { closure?(false, SRHealthError.unavailable); return }
            strong.authorEnable = success
            closure?(success, error)
        })
    }

    /// 存储运动数据 — 步数 \ 卡路里 \ 距离（24小时）
    private func save(step model: SRStep, closure: HealthStoreClosure? = nil) {
        guard authorEnable else { closure?(false, SRHealthError.unavailable); return }
        let date = model.date.date
        let calendar = SRCalendarUtils.calendar
        let startDayDate = calendar.startOfDay(for: date)

        let hoursSteps = model.hourRecords
        let hoursDistance = model.distanceHours() // m
        let hoursCalories = SRStep.calorieHours(for: hoursDistance)
        
        var sampleArray: [HKQuantitySample] = []
        for idx in 0..<24 {
            let startDate = calendar.date(byAdding: .hour, value: idx, to: startDayDate)!
            let endDate = calendar.date(byAdding: .hour, value: 1, to: startDate)!
            let step = hoursSteps[idx]
            let distance = hoursDistance[idx]
            let calories = hoursCalories[idx]
            if let sample = self.quantitySample(of: .step, with: Double(step), from: startDate, to: endDate) {
                sampleArray.append(sample)
            }
            if let sample = self.quantitySample(of: .distance, with: Double(distance), from: startDate, to: endDate) {
                sampleArray.append(sample)
            }
            if let sample = self.quantitySample(of: .calories, with: Double(calories), from: startDate, to: endDate) {
                sampleArray.append(sample)
            }
        }
        removeDay(of: .step, for: date)
        removeDay(of: .calories, for: date)
        removeDay(of: .distance, for: date) { [weak self] (success, error) in
            guard let strong = self, success else { closure?(false, SRHealthError.unavailable); return }
            strong.store.save(sampleArray, withCompletion: { (success, error) in
                closure?(success, error)
            })
        }
    }
    
    /// 存储静态心率 - 144
    private func save(heart model: SRHeart, closure: HealthStoreClosure? = nil) {
        guard authorEnable else { closure?(false, SRHealthError.unavailable); return }
        let date = model.date.date
        let calendar = SRCalendarUtils.calendar
        let startDayDate = calendar.startOfDay(for: date)

        var sampleArray: [HKQuantitySample] = []
        for (idx, value) in model.records.enumerated() {
            let startDate = calendar.date(byAdding: .minute, value: idx*10, to: startDayDate)!
            let endDate = calendar.date(byAdding: .minute, value: 1, to: startDate)!
            if let sample = self.quantitySample(of: .heart, with: Double(value), from: startDate, to: endDate) {
                sampleArray.append(sample)
            }
        }
        removeDay(of: .heart, for: date) { [weak self] (success, error) in
            guard let strong = self, success else { closure?(false, SRHealthError.unavailable); return }
            strong.store.save(sampleArray, withCompletion: { (success, error) in
                closure?(success, error)
            })
        }
    }
    
    /// 存储睡眠
    private func save(sleep model: SRSleep, closure: HealthStoreClosure? = nil) {
        guard authorEnable else { closure?(false, SRHealthError.unavailable); return }
        let date = model.date.date
        let calendar = SRCalendarUtils.calendar
        let startDayDate = calendar.date(byAdding: .hour, value: -4, to: calendar.startOfDay(for: date))!

        var sampleArray: [HKCategorySample] = []
        var lastIdx = 0
        var lastType: SRSleepType = model.records.first!
        for (idx, value) in model.records.enumerated() {
            if idx > 0, lastType.hkvalue != value.hkvalue {
                let startDate = calendar.date(byAdding: .minute, value: lastIdx, to: startDayDate)!
                let endDate = calendar.date(byAdding: .minute, value: idx, to: startDayDate)!
                if let sample = self.categorySample(sleep: lastType, from: startDate, to: endDate) {
                    sampleArray.append(sample)
                }
                lastIdx = idx
                lastType = value
            }
        }
        
        removeDay(of: .sleep, for: date) { [weak self] (success, error) in
            guard let strong = self, success else { closure?(false, SRHealthError.unavailable); return }
            strong.store.save(sampleArray, withCompletion: { (success, error) in
                closure?(success, error)
            })
        }
    }

}

#else

public class SRHealthStore {
    private init() { }
    private func requestAuthorization(closure: HealthStoreClosure? = nil) { closure?(false, SRHealthError.unavailable) }
    private func save(step model: SRStep, closure: HealthStoreClosure? = nil) { closure?(false, SRHealthError.unavailable) }
    private func save(heart model: SRHeart, closure: HealthStoreClosure? = nil) { closure?(false, SRHealthError.unavailable) }
    private func save(sleep model: SRSleep, closure: HealthStoreClosure? = nil) { closure?(false, SRHealthError.unavailable) }
}

#endif

extension SRHealthStore {

    private static let shared = SRHealthStore()

    static func requestAuthorization(closure: HealthStoreClosure? = nil) {
        shared.requestAuthorization(closure: closure)
    }

    /// 存储运动数据 — 步数 \ 卡路里 \ 距离（24小时）
    static func save(step model: SRStep, closure: HealthStoreClosure? = nil) {
        shared.save(step: model, closure: closure)
    }
    
    /// 存储静态心率 - 144
    static func save(heart model: SRHeart, closure: HealthStoreClosure? = nil) {
        shared.save(heart: model, closure: closure)
    }
    
    /// 存储随眠
    static func save(sleep model: SRSleep, closure: HealthStoreClosure? = nil) {
        shared.save(sleep: model, closure: closure)
    }
}
