//
//  SRClockItem.swift
//  BandKit
//
//  Created by Mac on 2020/5/7.
//

import Foundation

public enum SRWeekType: Int, SREnumtProtocol {
    case sun
    case mon
    case tue
    case wed
    case thu
    case fri
    case sat

    var value: UInt8  {
        switch self {
        case .sun: return 0x01
        case .mon: return 0x02
        case .tue: return 0x04
        case .wed: return 0x08
        case .thu: return 0x10
        case .fri: return 0x20
        case .sat: return 0x40
        }
    }
    
    public static func value(_ repeats:[SRWeekType]) -> UInt8 {
        var value: UInt8 = 0
        repeats.forEach { (e) in value |= e.value }
        return value
    }
    
    public static func parse(_ value: UInt8) -> [SRWeekType] {
        var repeats: [SRWeekType] = []
        collection.forEach { (e) in if e.value & value == e.value { repeats.append(e)} }
        return repeats
    }
    
    public static var none: UInt8     = 0x00
    public static var workday: UInt8  = 0x3E
    public static var everyday: UInt8 = 0x7F
    
    public static var collection: [SRWeekType] = [.sun, .mon, .tue, .wed, .thu, .fri, .sat]
}

extension SRWeekType: CustomStringConvertible {
    public var description: String {
        switch self {
        case .sun: return "sun"
        case .mon: return "mon"
        case .tue: return "tue"
        case .wed: return "wed"
        case .thu: return "thu"
        case .fri: return "fri"
        case .sat: return "sat"
        }
    }
}

public class SRClockItem: SRModelBase, SRModelProtocol {
    /// 开关
    public var enable: Bool
    /// 小时 ( 0 ~ 23 )
    public var hour: UInt8
    /// 分钟 ( 0 ~ 59 )
    public var minute: UInt8
    /// 周重复
    public var repeats: [SRWeekType]
    /// 震动
    public var vibrate: SRVibrate
    /// 闹钟标签：max ( 15 Byte )
    public var label: String
    
    public var date: Date {
        get {
            SRCalendarUtils.date(withHour: Int(hour), minute: Int(minute))
        }
        set {
            let components = SRCalendarUtils.components(ofDate: newValue)
            hour = UInt8(components.hour!)
            minute = UInt8(components.minute!)
        }
    }
    
    public var time: String { return String(format: "%02d:%02d", hour, minute) }

    
    ///: - Initialize
    required public init() {
        let date = Date()
        let calendar = SRCalendarUtils.calendar
        
        self.enable = true
        self.repeats = [SRWeekType]()
        self.hour = UInt8(calendar.component(.hour, from: date))
        self.minute = UInt8(calendar.component(.minute, from: date))
        self.vibrate = SRVibrate()
        self.label = ""
    }

    
    ///: - internal
    
    var invalid: Bool {
        return (!enable &&  repeats.count == 0) && (hour == 0 && minute == 0 && vibrate.byte==0 && label.count==0)
    }
    
    var maxLabelLen: Int { 15 }

    ///: - override

    override var dataLen: Int { 20 }

    override func data() -> Data {

        let _enable: UInt8 = enable ? 0x80 : 0x00
        let _repeat: UInt8 = SRWeekType.value(repeats)
        let _alarmSet: UInt8 = _enable | _repeat
        let _hour = min(hour, 23)
        let _minute = min(minute, 59)
        let _ringSet: UInt8 = vibrate.byte
        var _labelLen = 0
        var _label = Data()
        if let hasData = label.data(using: .utf8) { _labelLen = min(hasData.count, maxLabelLen); _label += hasData.prefix(maxLabelLen) }
        if _label.count < maxLabelLen { _label += Data(count: maxLabelLen-_label.count) }
   
        let buffer = SRByteBuffer()
        buffer.putByte(_alarmSet)
        buffer.putByte(_hour)
        buffer.putByte(_minute)
        buffer.putByte(_ringSet)
        buffer.putByte(UInt8(_labelLen))
        buffer.putData(_label) // 15 Byte
        
        return buffer.bufferData
    }
    
    
    @discardableResult
    override func parse(_ data: Data) -> Bool {
        guard data.count == dataLen else { return false}

        let buffer = SRByteBuffer(with: data)
        let _alarmSet = buffer.get()
        let _hour = buffer.get()
        let _minute = buffer.get()
        let _ringSet = buffer.get()
        let _lableLen = Int(buffer.get())
        let _lableData = buffer.getBytes(min(_lableLen, maxLabelLen))

        self.enable = (_alarmSet & 0x80) == 0x80
        self.hour = min(_hour, 23)
        self.minute = min(_minute, 59)
        self.vibrate.byte = _ringSet
        self.label = String(data: _lableData, encoding: String.Encoding.utf8) ?? ""
        self.repeats = SRWeekType.parse(_alarmSet)
        
        return true
    }
}


